<?php

declare(strict_types=1);

namespace Skyboard;

// Simple environment loader
defined('ROOT_PATH') || define('ROOT_PATH', __DIR__);

function sb_env(string $key, ?string $default = null): ?string {
    $value = $_ENV[$key] ?? $_SERVER[$key] ?? getenv($key);
    if ($value === false) {
        return $default;
    }
    return $value === null ? $default : (string)$value;
}

(function (): void {
    $envFile = ROOT_PATH . '/.env';
    if (!file_exists($envFile)) {
        return;
    }
    $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if (!$lines) {
        return;
    }
    foreach ($lines as $line) {
        if (str_starts_with(trim($line), '#')) {
            continue;
        }
        [$name, $value] = array_pad(explode('=', $line, 2), 2, '');
        $name = trim($name);
        if ($name === '') {
            continue;
        }
        $value = trim($value);
        $_ENV[$name] = $value;
        $_SERVER[$name] = $value;
        putenv($name . '=' . $value);
    }
})();

spl_autoload_register(static function (string $class): void {
    $prefix = 'Skyboard\\';
    if (!str_starts_with($class, $prefix)) {
        return;
    }
    $relative = substr($class, strlen($prefix));
    $path = ROOT_PATH . '/src/' . str_replace('\\', '/', $relative) . '.php';
    if (file_exists($path)) {
        require_once $path;
    }
});

function config(): array
{
    static $config;
    if ($config !== null) {
        return $config;
    }
    $dsn = sb_env('PERSISTENCE_DSN', 'mysql:host=127.0.0.1;port=3306;dbname=skyboard;charset=utf8mb4');
    $dbUser = sb_env('PERSISTENCE_USER');
    $dbPassword = sb_env('PERSISTENCE_PASSWORD');
    $rulesVersion = sb_env('RULES_VERSION', '1');
    $rulesEngineEnabled = sb_env('RULES_ENGINE_ENABLED', 'true');
    $licenseSecret = sb_env('LICENSE_SECRET', 'change-me-please');
    $appEnvRaw = sb_env('APP_ENV', 'dev');
    $appEnv = $appEnvRaw !== null ? strtolower(trim($appEnvRaw)) : 'dev';
    if ($appEnv === '') {
        $appEnv = 'dev';
    }
    // Default behavior: OFF in prod, ON elsewhere unless explicitly overridden by env
    $structureTagDefault = $appEnv === 'prod' ? 'false' : 'true';
    $structureTagReactions = sb_env('STRUCTURE_TAG_REACTIONS', $structureTagDefault);
    $structureTagReactionsEnabled = filter_var(
        $structureTagReactions,
        FILTER_VALIDATE_BOOL,
        FILTER_NULL_ON_FAILURE
    );
    if ($structureTagReactionsEnabled === null) {
        $structureTagReactionsEnabled = $appEnv !== 'prod';
    }
    if (!str_starts_with((string)$dsn, 'mysql:')) {
        throw new \RuntimeException('DB_CONFIG_INVALID: Seul MySQL est supporté, vérifier PERSISTENCE_DSN.');
    }
    if (($dbUser ?? '') === '' || ($dbPassword ?? '') === '') {
        throw new \RuntimeException('DB_CONFIG_INVALID: PERSISTENCE_USER/PERSISTENCE_PASSWORD requis pour MySQL.');
    }
    if (!str_contains($dsn, 'dbname=')) {
        throw new \RuntimeException('DB_CONFIG_INVALID: mysql DSN sans dbname=.');
    }
    if (!str_contains($dsn, 'charset=utf8mb4')) {
        throw new \RuntimeException('DB_CONFIG_INVALID: mysql DSN sans charset=utf8mb4.');
    }

    $config = [
        'dsn' => $dsn,
        'rules_version' => $rulesVersion,
        'rules_engine_enabled' => filter_var($rulesEngineEnabled, FILTER_VALIDATE_BOOL, FILTER_NULL_ON_FAILURE) ?? true,
        'structure_tag_reactions_enabled' => $structureTagReactionsEnabled,
        'license_secret' => $licenseSecret,
        'app_env' => $appEnv,
        'db_user' => $dbUser,
        'db_password' => $dbPassword,
    ];
    return $config;
}

function storage_path(string $relative = ''): string
{
    $base = ROOT_PATH . '/var';
    if (!is_dir($base)) {
        @mkdir($base, 0775, true);
    }
    return rtrim($base . '/' . ltrim($relative, '/'), '/');
}
