import '../../packages/ui/toast.js';
import '../../packages/ui/modal.js';
import { promptDialog } from '../../packages/ui/dialog.js';
import '../../packages/ui/menu.js';
import { showToast } from '../../packages/ui/toast.js';
import { ensureAuthenticated } from '../../packages/services/session.js';
import { fetchAccount, updateProfile, requestVerification, verifyEmail, redeemLicense } from '../../packages/services/account.js';
import { requestJson, setCsrfProvider } from '../../packages/services/http.js';
import {
  renderBrand as renderGlobalBrand,
  renderUserMenu as renderGlobalUserMenu,
  renderNotificationButton as renderGlobalNotificationButton,
  renderTopbar as renderGlobalTopbar,
  renderActionsGroup as renderTopbarActionsGroup,
} from '../../packages/ui/global-topbar.js';
import { goToBoards, goToModules, goToAdmin, goToBoardNotifications, goToAuth } from '../../packages/services/navigation.js';
import { openFileLibrary } from '../../packages/ui/file-library.js';
import { escapeHtml, formatDateTime } from '../../packages/utils/format.js';

const app = document.getElementById('app');
if (app) {
  app.addEventListener('click', handleGlobalAction);
}
let csrfToken = '';
let state = {
  session: null,
  account: null,
  loading: false,
  verification: null,
  subscriptions: [],
};

init().catch(error => {
  console.error(error);
  if (app) {
    app.removeAttribute('data-loading');
    app.innerHTML = '<div class="loading">Erreur de chargement.</div>';
  }
});

async function init() {
  const session = await ensureAuthenticated();
  if (!session) return;
  csrfToken = session.csrf;
  state.session = session;
  try { setCsrfProvider(() => csrfToken); } catch (_) {}
  await loadAccount();
}

async function loadAccount() {
  setState({ loading: true });
  try {
    const account = await fetchAccount();
    const subscriptions = await fetchSubscriptions();
    setState({ account, subscriptions, loading: false });
  } catch (error) {
    console.error(error);
    setState({ loading: false });
    showToast('Impossible de charger votre compte', { kind: 'error' });
  }
}

function setState(patch) {
  state = { ...state, ...patch };
  render();
}

function render() {
  if (!app) return;
  if (!state.session) {
    app.innerHTML = '<div class="loading">Initialisation…</div>';
    return;
  }
  app.removeAttribute('data-loading');

  if (state.loading && !state.account) {
    app.innerHTML = '<div class="loading">Chargement…</div>';
    return;
  }

  const account = state.account ?? {};
  const licenses = Array.isArray(account.licenses) ? account.licenses : [];
  const verification = state.verification ?? null;

  app.innerHTML = `
    <div class="account-shell app-view app-view--narrow">
    ${renderAppTopbar()}
      <section class="account-grid app-grid app-grid--responsive">
        ${renderProfileCard(account)}
        ${renderVerificationCard(account, verification)}
        ${renderSubscriptionsCard(state.subscriptions)}
        ${renderLicenseCard(licenses)}
      </section>
    </div>
  `;

  bindEvents();
}

function renderAppTopbar() {
  if (!state.session) return '';
  const brand = renderGlobalBrand({ subtitle: 'Mon compte' });
  const menuItems = [
    { id: 'open-boards', label: 'Mes boards', icon: '📋' },
    { id: 'open-modules', label: 'Modules', icon: '🧩' },
    ...(isAdmin(state.session.user) ? [{ id: 'open-admin', label: 'Administration', icon: '🛠️' }] : []),
  ];
  const userMenu = renderGlobalUserMenu(state.session.user, {
    items: menuItems,
    footer: [{ id: 'logout', label: 'Déconnexion', kind: 'danger', icon: '🚪' }],
  });
  const notifications = renderGlobalNotificationButton({
    action: 'open-notifications',
    href: '/board.html',
    title: 'Ouvrir les notifications',
    count: 0,
  });
  const primaryActions = renderTopbarActionsGroup(`
    <button class="btn ghost" data-action="open-boards">Mes boards</button>
    <button class="btn btn--icon ghost focus-ring" data-action="open-file-library" aria-label="Mes fichiers">📁</button>
  `);
  return renderGlobalTopbar({
    className: 'app-topbar--compact app-topbar--wrap',
    brand,
    actions: [primaryActions, notifications, userMenu],
  });
}

function renderProfileCard(account) {
  const emailVerified = !!account.emailVerified;
  const badgeTone = emailVerified ? 'badge--success' : 'badge--danger';
  return `
    <article class="account-card app-card surface-card surface-card--glass surface-card--bordered">
      <h2>Profil</h2>
      <div class="profile-meta app-grid app-grid--metadata">
        <div>
          <div>Email</div>
          <strong>${escapeHtml(account.email ?? '')}</strong>
          <span class="badge ${badgeTone}">
            ${emailVerified ? 'Email vérifié' : 'Email non vérifié'}
          </span>
        </div>
        <div>
          <div>Rôle effectif</div>
          <strong>${escapeHtml(account.role ?? 'standard')}</strong>
        </div>
      </div>
      <form class="form-grid" data-form="profile">
        <label>Pseudo affiché
          <input name="pseudo" value="${escapeHtml(account.pseudo ?? '')}" maxlength="60" />
        </label>
        <div class="account-hint">Ce pseudo est visible dans vos boards partagés (futur module Teams).</div>
        <footer class="app-card__footer">
          <button class="btn primary" type="submit">Mettre à jour</button>
        </footer>
      </form>
    </article>
  `;
}

function renderVerificationCard(account, verification) {
  const emailVerified = !!account.emailVerified;
  const lastToken = verification?.token ?? null;
  const expiresAt = verification?.expiresAt ? formatDateTime(verification.expiresAt * 1000) : null;
  return `
    <article class="account-card surface-card surface-card--glass surface-card--bordered">
      <h2>Vérification email</h2>
      <p class="account-hint">Confirmez votre adresse pour débloquer l'accès complet aux futures fonctionnalités collaboratives.</p>
      <div class="form-grid" data-section="verification">
        <footer class="app-card__footer">
          <button class="btn" data-action="request-verification" ${emailVerified ? 'disabled' : ''}>Recevoir un code</button>
          <button class="btn ghost" data-action="verify-email">J'ai un code</button>
        </footer>
        ${lastToken ? `<div class="account-hint">Dernier code : <strong>${escapeHtml(lastToken)}</strong> (expire ${expiresAt ? 'le ' + escapeHtml(expiresAt) : 'bientôt'})</div>` : ''}
      </div>
    </article>
  `;
}

function renderLicenseCard(licenses) {
  const list = licenses.length
    ? `<ul class="licenses-list">
        ${licenses.map(renderLicense).join('')}
      </ul>`
    : '<p class="account-hint">Aucune licence active pour l’instant.</p>';

  return `
    <article class="account-card app-card surface-card surface-card--glass surface-card--bordered">
      <h2>Licences</h2>
      <form class="form-grid" data-form="license">
        <label>Ajouter un code licence
          <input name="license" placeholder="SB-XXXX-XXXX" />
        </label>
        <footer class="app-card__footer">
          <button class="btn primary" type="submit">Activer</button>
        </footer>
      </form>
      ${list}
    </article>
  `;
}

function renderSubscriptionsCard(categories) {
  const list = Array.isArray(categories) ? categories : [];
  const rows = list.map(c => {
    const canOverride = String(c.dispatch_mode || '').toUpperCase() === 'PERSONALIZED' && !!c.allow_user_override;
    const ov = canOverride ? `
      <div class="sub-override">
        <label>Fréquence
          <select name="ov_kind">
            <option value="EVERY_N_DAYS" ${c.override_kind==='EVERY_N_DAYS'?'selected':''}>EVERY_N_DAYS</option>
            <option value="WEEKLY" ${c.override_kind==='WEEKLY'?'selected':''}>WEEKLY</option>
            <option value="MONTHLY" ${c.override_kind==='MONTHLY'?'selected':''}>MONTHLY</option>
          </select>
        </label>
        <label>Paramètre
          <input name="ov_param" type="number" value="${c.override_param ?? ''}" placeholder="1..7 / 14 / 21 ou 1..28" />
        </label>
        <button class="btn ghost" data-action="save-override" data-id="${c.id}">Enregistrer</button>
      </div>` : '<div class="account-hint">Aucun override disponible</div>';
    return `
      <li class="sub-item" data-cat="${c.id}">
        <div class="sub-head">
          <strong>${escapeHtml(c.name)}</strong>
          <label class="toggle"><input type="checkbox" data-action="sub-toggle" data-id="${c.id}" ${c.subscribed?'checked':''}/> Abonné</label>
        </div>
        ${ov}
      </li>`;
  }).join('');
  return `
    <article class="account-card app-card surface-card surface-card--glass surface-card--bordered">
      <h2>Mes abonnements</h2>
      ${list.length ? `<ul class="subs-list">${rows}</ul>` : '<p class="account-hint">Aucune catégorie disponible.</p>'}
    </article>
  `;
}

async function fetchSubscriptions() {
  try {
    const payload = await requestJson('/api/notifications/categories', { method: 'GET' });
    return Array.isArray(payload?.categories) ? payload.categories : [];
  } catch (e) {
    return [];
  }
}

function renderLicense(license) {
  const expires = license.expires_at ? formatDateTime(license.expires_at * 1000) : 'Pas d’expiration';
  const redeemed = license.redeemed_at ? formatDateTime(license.redeemed_at * 1000) : '—';
  return `
    <li class="license-card">
      <div><strong>${escapeHtml(license.code ?? '')}</strong></div>
      <div>Rôle : <span class="badge badge--info">${escapeHtml(license.role ?? 'standard')}</span></div>
      <div>Packs : ${escapeHtml((license.packs ?? []).join(', ') || 'Jamais')}</div>
      <div>Activée : ${redeemed}</div>
      <div>Expire : ${expires}</div>
    </li>
  `;
}

function bindEvents() {
  if (!app) return;
  const profileForm = app.querySelector('[data-form="profile"]');
  if (profileForm) profileForm.addEventListener('submit', handleProfileSubmit);

  const licenseForm = app.querySelector('[data-form="license"]');
  if (licenseForm) licenseForm.addEventListener('submit', handleLicenseSubmit);

  const requestBtn = app.querySelector('[data-action="request-verification"]');
  if (requestBtn) requestBtn.addEventListener('click', handleVerificationRequest);

  const verifyBtn = app.querySelector('[data-action="verify-email"]');
  if (verifyBtn) verifyBtn.addEventListener('click', handleVerifyEmail);

  // Subscriptions toggle
  app.querySelectorAll('[data-action="sub-toggle"]').forEach(el => {
    el.addEventListener('change', async () => {
      const catId = Number(el.dataset.id);
      if (!Number.isFinite(catId) || catId <= 0) return;
      try {
        if (el.checked) {
          await requestJson('/api/commands', { method: 'POST', body: { type: 'User.SubscribeCategory', payload: { categoryId: catId } } });
        } else {
          await requestJson('/api/commands', { method: 'POST', body: { type: 'User.UnsubscribeCategory', payload: { categoryId: catId } } });
        }
        const subscriptions = await fetchSubscriptions();
        setState({ subscriptions });
        showToast('Préférence enregistrée', { kind: 'success' });
      } catch (e) {
        showToast('Action impossible', { kind: 'error' });
      }
    });
  });
  app.querySelectorAll('[data-action="save-override"]').forEach(btn => {
    btn.addEventListener('click', async () => {
      const catId = Number(btn.dataset.id);
      const row = btn.closest('[data-cat]');
      if (!row) return;
      const kindSel = row.querySelector('select[name="ov_kind"]');
      const paramInput = row.querySelector('input[name="ov_param"]');
      const kind = (kindSel?.value || '').toUpperCase();
      const param = Number(paramInput?.value ?? 0) || null;
      if (!['EVERY_N_DAYS','WEEKLY','MONTHLY'].includes(kind)) {
        showToast('Fréquence invalide', { kind: 'warning' });
        return;
      }
      if (kind === 'EVERY_N_DAYS' && ![1,2,3,4,5,6,7,14,21].includes(param)) {
        showToast('Paramètre EVERY_N_DAYS invalide', { kind: 'warning' });
        return;
      }
      if (kind === 'WEEKLY' && !(param >= 1 && param <= 7)) {
        showToast('Paramètre WEEKLY invalide (1..7)', { kind: 'warning' });
        return;
      }
      if (kind === 'MONTHLY' && !(param >= 1 && param <= 28)) {
        showToast('Paramètre MONTHLY invalide (1..28)', { kind: 'warning' });
        return;
      }
      try {
        await requestJson('/api/commands', { method: 'POST', body: { type: 'User.SetCategoryFrequencyOverride', payload: { categoryId: catId, kind, param } } });
        const subscriptions = await fetchSubscriptions();
        setState({ subscriptions });
        showToast('Override enregistré', { kind: 'success' });
      } catch (e) {
        showToast('Impossible d\'enregistrer l\'override', { kind: 'error' });
      }
    });
  });
}

async function handleProfileSubmit(event) {
  event.preventDefault();
  const form = event.currentTarget;
  const data = new FormData(form);
  const pseudo = data.get('pseudo')?.toString().trim();
  if (!pseudo) {
    showToast('Veuillez indiquer un pseudo.', { kind: 'warning' });
    return;
  }
  try {
    await updateProfile({ pseudo }, csrfToken);
    showToast('Profil mis à jour', { kind: 'success' });
    await loadAccount();
  } catch (error) {
    console.error(error);
    const message = error?.payload?.error ?? 'Mise à jour impossible';
    showToast(message, { kind: 'error' });
  }
}

async function handleLicenseSubmit(event) {
  event.preventDefault();
  const form = event.currentTarget;
  const data = new FormData(form);
  const code = data.get('license')?.toString().trim();
  if (!code) {
    showToast('Entrez un code de licence.', { kind: 'warning' });
    return;
  }
  try {
    const result = await redeemLicense(code, csrfToken);
    showToast('Licence activée', { kind: 'success' });
    form.reset();
    await loadAccount();
  } catch (error) {
    console.error(error);
    const payload = error?.payload ?? {};
    const reason = payload?.error ?? 'Activation impossible';
    showToast(mapLicenseError(reason), { kind: 'error' });
  }
}

async function handleVerificationRequest(event) {
  event.preventDefault();
  try {
    const result = await requestVerification(csrfToken);
    setState({ verification: result });
    showToast('Code de vérification généré', { kind: 'success' });
  } catch (error) {
    console.error(error);
    showToast('Impossible de générer un code', { kind: 'error' });
  }
}

async function handleVerifyEmail(event) {
  event.preventDefault();
  const token = (await promptDialog({ title: 'Vérifier votre email', label: 'Code reçu par email', defaultValue: '' }))?.trim();
  if (!token) return;
  try {
    const result = await verifyEmail(token, csrfToken);
    if (result?.success) {
      showToast('Email vérifié ✅', { kind: 'success' });
      await loadAccount();
    } else {
      showToast('Code invalide', { kind: 'error' });
    }
  } catch (error) {
    console.error(error);
    showToast('Verification impossible', { kind: 'error' });
  }
}

function handleGlobalAction(event) {
  const target = event.target instanceof Element ? event.target.closest('[data-action]') : null;
  if (!target || !app || !app.contains(target)) return;
  const action = target.dataset.action;
  switch (action) {
    case 'open-boards':
      event.preventDefault();
      goToBoards();
      break;
    case 'open-modules':
      event.preventDefault();
      goToModules();
      break;
    case 'open-admin':
      event.preventDefault();
      goToAdmin();
      break;
    case 'open-notifications':
      event.preventDefault();
      goToBoardNotifications();
      break;
    case 'open-file-library':
      event.preventDefault();
      (async () => { try { await openFileLibrary({ title: 'Mes fichiers', showSelect: false }); } catch (_) {} })();
      break;
    case 'logout':
      event.preventDefault();
      logout();
      break;
    default:
      break;
  }
}

function isAdmin(user) {
  const role = (user?.role ?? 'standard').toString().toLowerCase();
  return role === 'admin' || role === 'superadmin';
}

async function logout() {
  try { await requestJson('/api/commands', { method: 'POST', body: { type: 'Account.Logout', payload: {} } }); } catch (_) {}
  goToAuth();
}

function mapLicenseError(reason) {
  switch (String(reason)) {
    case 'invalid-signature':
      return 'Code invalide.';
    case 'expired':
      return 'Cette licence est expirée.';
    case 'wrong-user':
      return 'Ce code est lié à un autre compte.';
    case 'already-used':
      return 'Ce code a déjà été utilisé.';
    case 'persist-failed':
      return 'Erreur serveur pendant l’activation.';
    case 'not-found':
      return 'Licence introuvable.';
    default:
      return 'Activation impossible.';
  }
}
