import '../../packages/ui/tabs.js';
import '../../packages/ui/modal.js';
import '../../packages/ui/menu.js';
import '../../packages/ui/popover.js';
import { ensureAuthenticated } from '../../packages/services/session.js';
import { fetchBoards, fetchBoardState, confirmAutosave } from '../../packages/services/boards.js';
import { CommandBusClient } from '../../packages/services/commandBus.js';
import { createStore } from '../../packages/state/store.js';
import { applyPatch } from '../../packages/state/applyPatch.js';
import { showToast } from '../../packages/ui/toast.js';
import { initAutosave, mapAutosaveStatus as autoMapAutosaveStatus, renderAutosavePanel as autoRenderAutosavePanel, beginAutosaveCycle as autoBeginAutosaveCycle, completeAutosaveSuccess as autoCompleteAutosaveSuccess, completeAutosaveError as autoCompleteAutosaveError, isAutosavePending, attemptAutosaveConfirmIfIdle, normalizeAutosaveDetails } from './modules/autosave.js';
import { initNotifications, defaultNotificationsState as notifDefaultState, renderNotificationsTray as notifRenderTray, toggleNotificationsTray as notifToggle, openNotificationsTray as notifOpen, closeNotificationsTray as notifClose, loadNotifications as notifLoad, dismissNotification as notifDismissOne, dismissAllNotifications as notifDismissAll } from './modules/notifications.js';
import { initTags, addTag as tagsAddTag, removeTag as tagsRemoveTag, resolveRuleExplanation as tagsResolveRuleExplanation, hideTagPicker as tagsHideTagPicker, clearTagCaches as tagsClearTagCaches } from './modules/tags.js';
import { renderDialog as dialogsRenderDialog } from './modules/ui/dialogs.js';
import {
  renderBoardConfig,
  initBoardConfig,
  syncBoardConfigPanels,
  openBoardConfig,
  closeBoardConfig,
  setBoardConfigTab,
  submitGeneralSettings as submitBoardConfigGeneral,
} from './modules/ui/board-config.js';
import { loadPackSlots, renderSlot as runtimeRenderSlot, registries as runtimeRegistries, configureSlotRuntime } from '../../packages/modules/runtime.js';
import { fetchPacks, fetchPacksManifest } from './modules/services.js';
import { renderTopbar as layoutRenderTopbar, renderSidebar as layoutRenderSidebar, renderMain as layoutRenderMain, renderEmpty as layoutRenderEmpty } from './modules/ui/layout.js';
import { initFilters, toggleFilter as filtersToggle, normalizeFilterTags, sameStringArray, filterBoardView } from './modules/filters.js';
import { initEditor, openItem as editorOpenItem, switchTab as editorSwitchTab, closeTab as editorCloseTab, closeAllTabs as editorCloseAllTabs, updateDraft as editorUpdateDraft, saveItem as editorSaveItem, collectItems as editorCollectItems, findItem as editorFindItem } from './modules/editor.js';
import { initBoardStructure, createBoardAction, createWorkspace, renameWorkspace, deleteWorkspace, createColumn, renameColumn, createFolder, createColumnItem, renameList, toggleList, createItem, switchWorkspace, refreshBoardsList, replaceBoardQuery } from './modules/board-structure.js';
import { initDnd, setupDragManager as dndSetupDragManager, teardownDnd as dndTeardown, moveItemWithKeyboard as dndMoveItemWithKeyboard, moveListWithKeyboard as dndMoveListWithKeyboard, moveColumnWithKeyboard as dndMoveColumnWithKeyboard, resetDndState as dndResetState } from './modules/dnd.js';
import { initSelection } from './modules/selection.js';
import { initMoveNode, moveNodeViaSelection } from './modules/move-node.js';
import { initUiActions } from './modules/ui/actions.js';
import { createImportExportController } from './modules/import-export.js';
import { applyDebugFromUrl } from '../../packages/services/debug.js';
import { requestJson, setCsrfProvider } from '../../packages/services/http.js';
import { goToAuth } from '../../packages/services/navigation.js';
import { initEditorModalManager } from './modules/ui/editor-modal-manager.js';
import { initContextMenu, openContextMenu, closeContextMenu } from './modules/ui/context-menu.js';

const app = document.getElementById('app');
let commandBus = null;
let store = null;
let csrfToken = '';
let lastRenderedBoardId = null;
let detachUiActions = null;
const pendingCommandPromises = new Set();


init().catch((error) => {
  console.error(error);
  if (app) {
    app.innerHTML = `<div class="loading">Erreur de chargement 🙈</div>`;
  }
});

async function init() {
  try { applyDebugFromUrl(); } catch (_) {}
  const session = await ensureAuthenticated();
  if (!session) return;
  csrfToken = session.csrf;
  try { setCsrfProvider(() => csrfToken); } catch (_) {}
  commandBus = new CommandBusClient({
    csrfProvider: () => csrfToken,
    onCsrfRefresh: (nextToken) => {
      if (typeof nextToken === 'string' && nextToken) {
        csrfToken = nextToken;
      }
    },
  });
  // no global leak; modules receive a sendCommand injection

  const search = new URLSearchParams(window.location.search);
  let boardId = search.get('board');
  const boards = await fetchBoards();
  if (!boardId && boards.length) {
    boardId = String(boards[0].id);
    replaceBoardQuery(boardId);
  }

  let boardSnapshot = null;
  if (boardId) {
    try {
      boardSnapshot = await fetchBoardState(boardId);
    } catch (error) {
      console.error(error);
    }
  }
  const initialBoardState = boardSnapshot?.state ?? null;
  const initialBoardMeta = boardSnapshot?.meta ?? null;
  const normalizedInitialMeta = initialBoardMeta
    ? {
        ...initialBoardMeta,
        history: Array.isArray(initialBoardMeta.history) ? initialBoardMeta.history : [],
      }
    : null;
  const initialFilterTags = normalizeFilterTags(initialBoardState?.tagFilter?.selected ?? []);

  const packs = await fetchPacks();
  let packsManifest = { tags: [], rules: [], datasets: [] };
  try {
    packsManifest = await fetchPacksManifest();
  } catch (error) {
    console.error(error);
  }
  try {
    await loadPackSlots();
  } catch (error) {
    console.error('PACK_SLOTS_LOAD_FAILED', error);
  }
  // Conserver les métadonnées complètes des tags système (incl. ui.filter.include)
  const systemTags = (packsManifest.tags || []).map(t => ({
    key: t.key,
    label: t.ui?.badge?.label ?? t.key,
    icon: t.ui?.badge?.icon ?? '',
    color: t.ui?.badge?.color ?? null,
    category: t.category ?? null,
    ui: t.ui || null,
  }));
  const packRules = Array.isArray(packsManifest.rules)
    ? packsManifest.rules
        .filter(rule => typeof rule?.id === 'string')
        .map(rule => {
          const source = typeof rule?.source === 'string' ? rule.source : '';
          const packId = source.startsWith('pack:') ? source.slice(5) : (rule?.pack ?? 'unknown');
          return {
            id: rule.id,
            scope: typeof rule.scope === 'string' ? rule.scope : '',
            priority: typeof rule.priority === 'number' ? rule.priority : null,
            triggers: Array.isArray(rule.triggers) ? rule.triggers.filter(t => typeof t === 'string') : [],
            description: typeof rule.description === 'string' && rule.description ? rule.description : null,
            source,
            packId,
            enabled: rule.enabled !== false,
          };
        })
    : [];
  const rulesByPack = packRules.reduce((acc, rule) => {
    const key = rule.packId || 'unknown';
    if (!acc[key]) {
      acc[key] = [];
    }
    acc[key].push(rule);
    return acc;
  }, {});
  const modulesState = {
    ...packs,
    tags: {
      system: systemTags,
      user: [],
    },
    rules: {
      list: packRules,
      byPack: rulesByPack,
    },
  };

  store = createStore({
    session,
    boards,
    currentBoardId: boardId,
    board: initialBoardState,
    boardMeta: normalizedInitialMeta,
    boardRevision: normalizedInitialMeta?.revision ?? null,
    trace: null,
    modules: modulesState,
    filter: { tags: initialFilterTags },
    rulesVersion: null,
    autosave: initialBoardState
      ? autoMapAutosaveStatus({
          status: 'saved',
          savedAt: new Date().toISOString(),
          revision: normalizedInitialMeta?.revision ?? null,
          updatedAt: normalizedInitialMeta?.updatedAt ?? null,
          history: normalizedInitialMeta?.history ?? [],
        })
      : autoMapAutosaveStatus(),
    lastEvents: { pre: [], post: [] },
    lastCommand: null,
    autosavePanelOpen: false,
    dialog: null,
    notifications: notifDefaultState(),
    boardConfig: { open: false, activeTab: 'general', busy: false, error: null },
  });

  initSelection({ app, store });
  initMoveNode({
    store,
    sendCommand: (type, payload) => sendCommand(type, payload),
  });
  initContextMenu({
    app,
    onRequestClose: () => {},
    onActionSelect: ({ action, nodeId }) => {
      if (action === 'context-move-node') {
        moveNodeViaSelection(nodeId);
      }
    },
  });

  configureSlotRuntime({
    sendCommand: (type, payload) => sendCommand(type, payload),
    tags: () => {
      const current = store?.getState?.();
      return current?.modules?.tags ?? modulesState.tags ?? { system: [], user: [] };
    },
  });

  store.subscribe((state) => render(state));
  initAutosave({ store, getCsrfToken: () => csrfToken, confirmAutosave, reloadBoard: reloadCurrentBoardState });
  initNotifications({ store, getCsrfToken: () => csrfToken });
  initTags({ store, getCsrfToken: () => csrfToken, sendCommand: (type, payload) => sendCommand(type, payload) });
  initFilters({ store, send: (type, payload) => sendCommand(type, payload) });
  initEditor({ store, send: (type, payload) => sendCommand(type, payload) });
  initEditorModalManager({ store });
  initBoardStructure({
    store,
    sendCommand: (type, payload) => sendCommand(type, payload),
    getCsrfToken: () => csrfToken,
  });
  initBoardConfig({ store, getCsrfToken: () => csrfToken });
  initDnd({
    app,
    store,
    sendCommand: (type, payload) => sendCommand(type, payload),
  });
  const importExport = createImportExportController({
    store,
    app,
    getCsrfToken: () => csrfToken,
    showToast,
    onBoardHydrated: () => {
      lastRenderedBoardId = null;
    },
  });

  detachUiActions = initUiActions({
    app,
    store,
    send: (type, payload) => sendCommand(type, payload),
    createBoardAction,
    notifToggle,
    notifClose,
    notifDismissOne,
    notifDismissAll,
    notifLoad,
    toggleAutosavePanel,
    closeAutosavePanel,
    openExportDialog: importExport.openExportDialog,
    openImportDialog: importExport.openImportDialog,
    closeDialog,
    openBoardConfig,
    closeBoardConfig,
    setBoardConfigTab,
    submitBoardConfigGeneral,
    copyExportJson: importExport.copyExportJson,
    downloadExportJson: importExport.downloadExportJson,
    submitImportDialog: importExport.submitImportDialog,
    switchWorkspace,
    createWorkspace,
    renameWorkspace,
    deleteWorkspace,
    createColumn,
    createFolder,
    createColumnItem,
    renameColumn,
    renameList,
    toggleList,
    createItem,
    tagsAddTag,
    tagsRemoveTag,
    editorOpenItem,
    editorSwitchTab,
    editorCloseTab,
    editorCloseAllTabs,
    editorSaveItem,
    filtersToggle,
    editorUpdateDraft,
    setImportMode: importExport.setImportMode,
    readImportForm: importExport.readImportForm,
    dndMoveItemWithKeyboard,
    dndMoveListWithKeyboard,
    dndMoveColumnWithKeyboard,
    moveNodeViaSelection,
    openContextMenu,
    closeContextMenu,
    logout,
  });
  registerLifecycleGuards();
  notifLoad('initial').catch(console.error);
}

 

function render(state) {
  if (!app) return;
  if (!state.board) {
    tagsHideTagPicker();
    dndTeardown();
    app.removeAttribute('data-loading');
  app.innerHTML = `${layoutRenderEmpty(state)}
    ${autoRenderAutosavePanel(state)}
    ${dialogsRenderDialog(state)}
    `;
    try {
      const btn = app.querySelector('[data-action="create-board"]');
      if (btn) {
        btn.addEventListener('click', (e) => { e.preventDefault(); try { createBoardAction(); } catch (err) { console.error(err); } }, { once: true });
      }
    } catch (_) { /* noop */ }
    return;
  }
  if (state.currentBoardId !== lastRenderedBoardId) {
    tagsHideTagPicker();
    tagsClearTagCaches();
    lastRenderedBoardId = state.currentBoardId;
  }
  const boardView = filterBoardView(state.board, state.filter.tags);
  app.removeAttribute('data-loading');
  app.innerHTML = `
    ${layoutRenderTopbar(state, boardView)}
    <div class="board-layout">
      ${layoutRenderSidebar(state, boardView)}
      ${layoutRenderMain(state, boardView)}
    </div>
    ${autoRenderAutosavePanel(state)}
    ${notifRenderTray(state)}
    ${dialogsRenderDialog(state)}
    ${renderBoardConfig(state)}
  `;
  try {
    mountSlots(app, state);
    syncBoardConfigPanels(app, state);
  } catch (e) {
    console.error('MOUNT_SLOTS_FAILED', e);
  }
  dndSetupDragManager();
}
function mountSlots(root, state) {
  const nodes = root.querySelectorAll('[data-slot]');
  for (const holder of nodes) {
    const slotId = holder.getAttribute('data-slot');
    if (!slotId) continue;
    const nodeId = holder.getAttribute('data-node');
    const item = nodeId ? (state.board?.nodes?.[nodeId] ?? null) : null;
    runtimeRenderSlot(slotId, { state, item, registries: runtimeRegistries }, holder);
  }
}

 

function registerLifecycleGuards() {
  window.addEventListener('beforeunload', handleBeforeUnload);
  document.addEventListener('visibilitychange', handleVisibilityChange);
}
 

async function logout() {
  try { await requestJson('/api/commands', { method: 'POST', body: { type: 'Account.Logout', payload: {} } }); } catch (_) {}
  goToAuth();
}


 

 

 

 

 

 

 

 

 

 

 

 

 

 

 

 

 

 

 

 

async function sendCommand(type, payload) {
  const state = store.getState();
  if (!state.currentBoardId) {
    showToast('Aucun board sélectionné', { kind: 'warning' });
    return null;
  }

  const previousMeta = {
    revision: state.boardRevision ?? state.boardMeta?.revision ?? null,
    updatedAt: state.boardMeta?.updatedAt ?? null,
    history: Array.isArray(state.autosave?.details?.history)
      ? state.autosave.details.history
      : Array.isArray(state.boardMeta?.history)
        ? state.boardMeta.history
        : [],
  };

  autoBeginAutosaveCycle();

  const command = {
    type,
    boardId: state.currentBoardId,
    revision: state.boardRevision ?? state.boardMeta?.revision ?? null,
    payload,
  };

  const request = commandBus.execute(command);
  pendingCommandPromises.add(request);
  request.finally(() => pendingCommandPromises.delete(request));

  try {
    const result = await request;
    const autosaveDetails = normalizeAutosaveDetails(result.autosave, { previous: previousMeta });
    const finalPatchOps = Array.isArray(result.finalPatch)
      ? result.finalPatch
      : Array.isArray(result.patch)
        ? result.patch
        : [];
    const lastCommand = {
      status: result.status ?? 'ok',
      autosave: result.autosave ?? null,
      trace: result.trace ?? null,
      error: result.error ?? null,
      events: result.events ?? null,
      patch: finalPatchOps,
      timestamp: Date.now(),
      command: { type },
    };
    store.setState(prev => {
      let nextBoard = prev.board;
      if (nextBoard && finalPatchOps.length) {
        try {
          nextBoard = applyPatch(nextBoard, { boardId: state.currentBoardId, operations: finalPatchOps });
        } catch (patchError) {
          console.error('PATCH_APPLY_FAILED', patchError);
        }
      }

      const derivedFilter = normalizeFilterTags(nextBoard?.tagFilter?.selected ?? prev.filter.tags);
      const nextFilter = sameStringArray(derivedFilter, prev.filter.tags)
        ? prev.filter
        : { tags: derivedFilter };

      const shouldUpdateMeta = autosaveDetails.revision !== null || autosaveDetails.updatedAt !== null;
      const nextBoardMeta = shouldUpdateMeta
        ? {
            ...(prev.boardMeta ?? {}),
            revision: autosaveDetails.revision ?? prev.boardRevision ?? null,
            updatedAt: autosaveDetails.updatedAt ?? prev.boardMeta?.updatedAt ?? null,
            history: Array.isArray(autosaveDetails.history)
              ? autosaveDetails.history
              : Array.isArray(prev.boardMeta?.history)
                ? prev.boardMeta.history
                : [],
          }
        : prev.boardMeta;

      return {
        ...prev,
        board: nextBoard,
        boardMeta: nextBoardMeta,
        boardRevision: autosaveDetails.revision ?? prev.boardRevision ?? null,
        trace: result.trace ?? prev.trace,
        autosave: autoMapAutosaveStatus(autosaveDetails),
        lastEvents: result.events ?? prev.lastEvents ?? { pre: [], post: [] },
        lastCommand,
        filter: nextFilter,
      };
    });
    autoCompleteAutosaveSuccess(autosaveDetails);
    return result;
  } catch (error) {
    const payloadError = error?.payload ?? null;
    const reasonId = payloadError?.error?.reasonId ?? null;
    let message = payloadError?.error?.message ?? null;

    if (!message && reasonId) {
      message = await tagsResolveRuleExplanation(reasonId);
    }

    if (!message) {
      switch (error?.message) {
        case 'COMMAND_NETWORK_ERROR':
          message = 'Connexion impossible au serveur de commandes';
          break;
        case 'COMMAND_RESPONSE_INVALID':
          message = 'Réponse invalide du serveur';
          break;
        default:
          message = 'Action refusée';
          break;
      }
    }

    if (payloadError?.error) {
      payloadError.error.message = message;
    }

    const autosaveDetails = normalizeAutosaveDetails(payloadError?.autosave, { fallbackStatus: 'error', previous: previousMeta });
    const errorDetails = payloadError?.error ?? { code: error?.message ?? 'COMMAND_FAILED', message };
    const lastCommand = {
      status: payloadError?.status ?? 'error',
      autosave: payloadError?.autosave ?? null,
      trace: payloadError?.trace ?? null,
      error: errorDetails,
      events: payloadError?.events ?? null,
      patch: payloadError?.patch ?? null,
      timestamp: Date.now(),
      command: { type },
    };

    store.setState(prev => ({
      ...prev,
      autosave: autoMapAutosaveStatus(autosaveDetails),
      lastEvents: payloadError?.events ?? prev.lastEvents ?? { pre: [], post: [] },
      lastCommand,
    }));

    autoCompleteAutosaveError(autosaveDetails, { message });

    if (payloadError?.error?.reasonId === 'board.conflict') {
      showToast('Le board a été modifié ailleurs. Rechargement…', { kind: 'warning' });
      await reloadCurrentBoardState().catch(console.error);
    } else if (message) {
      showToast(message, { kind: 'error' });
    }

    console.error(error);
    return null;
  }
}

 

async function reloadCurrentBoardState() {
  const state = store.getState();
  if (!state.currentBoardId) {
    return;
  }
  try {
    const snapshot = await fetchBoardState(state.currentBoardId);
    const meta = snapshot?.meta ?? null;
    store.setState(prev => {
      const snapshotFilter = normalizeFilterTags(snapshot?.state?.tagFilter?.selected ?? prev.filter.tags);
      const nextFilter = sameStringArray(snapshotFilter, prev.filter.tags)
        ? prev.filter
        : { tags: snapshotFilter };
      const history = Array.isArray(meta?.history)
        ? meta.history
        : Array.isArray(prev.boardMeta?.history)
          ? prev.boardMeta.history
          : [];

      return {
        ...prev,
        board: snapshot?.state ?? prev.board,
        boardMeta: meta ? { ...meta, history } : prev.boardMeta,
        boardRevision: meta?.revision ?? prev.boardRevision ?? null,
        autosave: autoMapAutosaveStatus({
          status: 'saved',
          savedAt: new Date().toISOString(),
          updatedAt: meta?.updatedAt ?? Date.now(),
          revision: meta?.revision ?? Date.now(),
          history,
        }),
        filter: nextFilter,
      };
    });
  } catch (error) {
    console.error('BOARD_RELOAD_FAILED', error);
    showToast('Impossible de recharger le board', { kind: 'error' });
  }
}

function toggleAutosavePanel() {
  store.setState(prev => ({
    ...prev,
    autosavePanelOpen: !prev.autosavePanelOpen,
  }));
}

function closeAutosavePanel() {
  store.setState(prev => ({
    ...prev,
    autosavePanelOpen: false,
  }));
}

function closeDialog() {
  store.setState(prev => ({
    ...prev,
    dialog: null,
  }));
}

function handleBeforeUnload(event) {
  if (isAutosavePending()) {
    event.preventDefault();
    event.returnValue = '';
  }
  if (typeof detachUiActions === 'function') {
    detachUiActions();
    detachUiActions = null;
  }
}

function handleVisibilityChange() {
  if (document.visibilityState === 'hidden') {
    if (pendingCommandPromises.size) {
      const pending = Array.from(pendingCommandPromises);
      Promise.allSettled(pending).catch(() => {});
    } else {
      try { attemptAutosaveConfirmIfIdle(); } catch (_) {}
    }
  }
}
