import { showToast as baseShowToast } from '../../../packages/ui/toast.js';
import '../../../packages/ui/modal.js';
import { confirmDialog, promptDialog, alertDialog } from '../../../packages/ui/dialog.js';
import { fetchBoards as baseFetchBoards, createBoard as baseCreateBoard } from '../../../packages/services/boards.js';
import { isDebug as baseIsDebug, toErrorMessage as baseToErrorMessage } from '../../../packages/services/debug.js';
import { normalizeFilterTags } from './filters.js';
import {
  mapAutosaveStatus as baseMapAutosaveStatus,
  beginAutosaveCycle as baseBeginAutosaveCycle,
  completeAutosaveSuccess as baseCompleteAutosaveSuccess,
  completeAutosaveError as baseCompleteAutosaveError,
} from './autosave.js';
import { findNode, isContainer, findColumnLocation } from './domain-utils.js';

const baseDeps = {
  showToast: baseShowToast,
  fetchBoards: baseFetchBoards,
  createBoard: baseCreateBoard,
  isDebug: baseIsDebug,
  toErrorMessage: baseToErrorMessage,
  mapAutosaveStatus: baseMapAutosaveStatus,
  beginAutosaveCycle: baseBeginAutosaveCycle,
  completeAutosaveSuccess: baseCompleteAutosaveSuccess,
  completeAutosaveError: baseCompleteAutosaveError,
};

let deps = { ...baseDeps };

const escapeSelector = (value) => {
  if (typeof value !== 'string') return '';
  if (typeof CSS !== 'undefined' && typeof CSS.escape === 'function') {
    return CSS.escape(value);
  }
  return value.replace(/([^a-zA-Z0-9_-])/g, '\\$1');
};

export function __setBoardStructureTestOverrides(overrides = {}) {
  if (!overrides || typeof overrides !== 'object') return;
  const entries = Object.entries(overrides).filter(([, value]) => typeof value !== 'undefined');
  if (!entries.length) return;
  deps = { ...deps, ...Object.fromEntries(entries) };
}

export function __resetBoardStructureTestOverrides() {
  deps = { ...baseDeps };
}

let storeRef = null;
let sendCommandRef = null;
let csrfProvider = () => '';

export function initBoardStructure(ctx) {
  storeRef = ctx.store ?? null;
  sendCommandRef = ctx.sendCommand ?? null;
  csrfProvider = ctx.getCsrfToken ?? (() => '');
}

export async function createBoardAction() {
  const name = (await promptDialog({ title: 'Créer un board', label: 'Nom du board', defaultValue: '' }))?.trim();
  if (!name) return;
  createBoardWithTitle(name);
}

export async function createWorkspace() {
  const title = (await promptDialog({ title: 'Créer un workspace', label: 'Nom du workspace', defaultValue: '' }))?.trim();
  if (!title) return;
  await sendCommandSafe('CreateWorkspace', { title });
}

export async function renameWorkspace(workspaceId) {
  if (!workspaceId) return;
  const title = (await promptDialog({ title: 'Renommer le workspace', label: 'Nouveau titre', defaultValue: '' }))?.trim();
  if (!title) return;
  await sendCommandSafe('RenameWorkspace', { workspaceId, title });
}

export async function deleteWorkspace(workspaceId) {
  if (!workspaceId) return;
  if (!(await confirmDialog({ title: 'Supprimer le workspace', message: 'Supprimer ce workspace et tout son contenu ?', okLabel: 'Supprimer', cancelLabel: 'Annuler' }))) return;
  await sendCommandSafe('DeleteNode', { nodeId: workspaceId });
}

export async function createColumn(workspaceIdOverride) {
  if (!storeRef) return;
  const title = (await promptDialog({ title: 'Créer une colonne', label: 'Nom de la colonne', defaultValue: '' }))?.trim();
  if (!title) return;
  const state = storeRef.getState();
  const activeWs = state.board?.activeWorkspaceId || null;
  const wsId = workspaceIdOverride || activeWs;
  if (!wsId) {
    await alertDialog({ title: 'Information', message: "Crée d'abord un workspace" });
    return;
  }
  await sendCommandSafe('CreateColumn', { workspaceId: wsId, title });
}

export async function renameColumn(columnId) {
  if (!columnId) return;
  const title = (await promptDialog({ title: 'Renommer la colonne', label: 'Nouveau titre', defaultValue: '' }))?.trim();
  if (!title) return;
  const state = storeRef.getState();
  // Exige un workspace explicite: on résout le parent workspace du columnId
  const loc = findColumnLocation(state.board, columnId);
  const ws = loc?.workspaceId || null;
  if (!ws) {
    await alertDialog({ title: 'Information', message: 'Workspace introuvable pour cette colonne' });
    return;
  }
  await sendCommandSafe('RenameColumn', { workspaceId: ws, columnId, title });
}

export async function createFolder(columnId, workspaceId, parentContainerId, options = {}) {
  if (!storeRef || !sendCommandRef) return false;
  const opts = options && typeof options === 'object' ? options : {};
  let title = typeof opts.title === 'string' ? opts.title.trim() : '';
  if (!title) {
    title = (await promptDialog({ title: 'Créer une liste', label: 'Nom de la liste', defaultValue: '' }))?.trim() ?? '';
  }
  if (!title) return false;
  const parentId = parentContainerId || columnId;
  if (!parentId) return false;
  const insertAt = Number.isInteger(opts.index) ? Math.max(0, Number(opts.index)) : 0;
  const payload = {
    parentId,
    title,
    sys: { shape: 'container' },
    tags: [],
  };
  payload.index = insertAt;
  try {
    const createResult = await sendCommandRef('CreateNode', payload);
    const nodeId = extractCreatedNodeId(createResult);
    if (!nodeId) {
      console.error('CREATE_LIST_NODE_ID_MISSING', createResult);
      notifyCommandError(null, 'Impossible de créer la liste');
      return false;
    }
    await sendCommandRef('AddTagV3', { nodeId, tag: { key: 'type/list', kind: 'system', sys: true } });
    return true;
  } catch (error) {
    notifyCommandError(error, 'Impossible de créer la liste');
    return false;
  }
}

export async function createColumnItem(columnId) {
  if (!columnId || !storeRef) return;
  const title = (await promptDialog({ title: "Créer un item", label: "Titre de l'item", defaultValue: '' }))?.trim();
  if (!title) return;
  await sendCommandSafe('CreateNode', { parentId: columnId, index: 0, title, sys: { shape: 'leaf' }, tags: [] });
}

export async function renameList(listId) {
  if (!listId) return;
  const title = (await promptDialog({ title: 'Renommer la liste', label: 'Nouveau titre', defaultValue: '' }))?.trim();
  if (!title) return;
  await sendCommandSafe('UpdateNode', { nodeId: listId, changes: { title } });
}

export async function toggleList(listId) {
  if (!storeRef || !listId) return;
  const state = storeRef.getState();
  const list = findNode(state.board, listId);
  if (!list) return;
  const wasCollapsed = !!list.collapsed;
  await sendCommandSafe('UpdateNode', { nodeId: listId, changes: { collapsed: !list.collapsed } });
  if (wasCollapsed) {
    requestAnimationFrame(() => {
      const node = document.querySelector(`.list[data-node="${escapeSelector(listId)}"]`);
      if (!node) return;
      node.scrollIntoView({ block: 'nearest', behavior: 'smooth' });
    });
  }
}

export async function createItem(listId, options = {}) {
  if (!listId || !storeRef || !sendCommandRef) return false;
  const opts = options && typeof options === 'object' ? options : {};
  let title = typeof opts.title === 'string' ? opts.title.trim() : '';
  if (!title) {
    title = (await promptDialog({ title: "Créer un item", label: "Titre de l'item", defaultValue: '' }))?.trim() ?? '';
  }
  if (!title) return false;
  const state = storeRef.getState();
  const node = findNode(state.board, listId);
  const parentId = node && isContainer(node) ? listId : listId;
  try {
    await sendCommandRef('CreateNode', { parentId, index: 0, title, sys: { shape: 'leaf' }, tags: [] });
    return true;
  } catch (error) {
    notifyCommandError(error, 'Impossible de créer l\'item');
    return false;
  }
}

export async function switchWorkspace(workspaceId) {
  if (!storeRef || !workspaceId) return;
  const state = storeRef.getState();
  if (!state.board || state.board.activeWorkspaceId === workspaceId) return;
  const updated = { ...state.board, activeWorkspaceId: workspaceId };
  storeRef.setState(prev => ({ ...prev, board: updated }));
}

export async function refreshBoardsList() {
  if (!storeRef) return;
  try {
    const boards = (await deps.fetchBoards?.()) ?? [];
    storeRef.setState(prev => {
      const currentBoardId = prev.currentBoardId;
      let nextMeta = prev.boardMeta ?? null;
      let nextRevision = prev.boardRevision ?? null;
      if (currentBoardId) {
        const match = boards.find(b => String(b.id) === String(currentBoardId));
        if (match && typeof match.updated_at === 'number') {
          nextRevision = match.updated_at;
          nextMeta = { ...(nextMeta ?? {}), updatedAt: match.updated_at, revision: match.updated_at };
        }
      }
      return {
        ...prev,
        boards,
        boardMeta: nextMeta,
        boardRevision: nextRevision,
      };
    });
  } catch (error) {
    console.error('BOARD_LIST_REFRESH_FAILED', error);
  }
}

export function replaceBoardQuery(boardId) {
  if (!boardId) return;
  const url = new URL(window.location.href);
  url.searchParams.set('board', boardId);
  history.replaceState(null, '', url);
}

async function createBoardWithTitle(title) {
  if (!storeRef) return;
  try {
    const board = await deps.createBoard?.(title, csrfProvider?.());
    replaceBoardQuery(board.id);
    const state = storeRef.getState();
    const now = typeof board.updatedAt === 'number' ? board.updatedAt : Date.now();
    const revision = typeof board.revision === 'number' ? board.revision : now;
    const history = Array.isArray(board.history) ? board.history : [];
    const boardListEntry = {
      id: board.id,
      title: board.title ?? title,
      updated_at: now,
      created_at: board.createdAt ?? now,
    };
    const createdFilter = normalizeFilterTags(board.state?.tagFilter?.selected ?? []);
    storeRef.setState(prev => ({
      ...prev,
      boards: [boardListEntry, ...prev.boards],
      currentBoardId: String(board.id),
      board: board.state,
      boardMeta: { updatedAt: board.updatedAt ?? now, revision, history },
      boardRevision: revision,
      autosave: (typeof deps.mapAutosaveStatus === 'function'
        ? deps.mapAutosaveStatus({
            status: 'saved',
            savedAt: new Date().toISOString(),
            updatedAt: board.updatedAt ?? now,
            revision,
            history,
          })
        : {
            status: 'saved',
            message: 'Enregistré',
            details: {
              status: 'saved',
              savedAt: new Date().toISOString(),
              updatedAt: board.updatedAt ?? now,
              revision,
              history,
            },
          }),
      filter: { tags: createdFilter },
    }));
  } catch (error) {
    console.error(error);
    const message = deps.isDebug?.()
      ? deps.toErrorMessage?.(error, 'Impossible de créer le board')
      : 'Impossible de créer le board';
    deps.showToast?.(message, { kind: 'error' });
  }
}

async function sendCommandSafe(type, payload, fallbackMessage = 'Action impossible') {
  try {
    await sendCommandRef?.(type, payload);
    return true;
  } catch (error) {
    notifyCommandError(error, fallbackMessage);
    return false;
  }
}

function notifyCommandError(error, fallbackMessage) {
  console.error(error);
  const fallback = typeof fallbackMessage === 'string' && fallbackMessage ? fallbackMessage : 'Action impossible';
  const message = resolveUserFacingError(error, fallback);
  const debug = deps.isDebug?.();
  if (debug && typeof deps.toErrorMessage === 'function') {
    deps.showToast?.(deps.toErrorMessage(error, fallback), { kind: 'error' });
    return;
  }
  deps.showToast?.(message, { kind: 'error' });
}

function resolveUserFacingError(error, fallback) {
  if (!error) return fallback;
  const payloadMessage = error?.payload?.error?.message
    || error?.payload?.message
    || error?.payload?.error
    || null;
  if (typeof payloadMessage === 'string' && payloadMessage.trim() !== '') {
    return payloadMessage;
  }
  const direct = typeof error?.message === 'string' ? error.message : null;
  if (direct && direct !== 'COMMAND_FAILED') {
    return direct;
  }
  return fallback;
}

function extractCreatedNodeId(result) {
  const operations = Array.isArray(result?.finalPatch)
    ? result.finalPatch
    : Array.isArray(result?.patch)
      ? result.patch
      : [];
  for (const op of operations) {
    if ((op?.op ?? '') !== 'node.create') {
      continue;
    }
    const nodeId = typeof op?.nodeId === 'string' && op.nodeId
      ? op.nodeId
      : (typeof op?.id === 'string' && op.id ? op.id : null);
    if (nodeId) {
      return nodeId;
    }
  }
  return null;
}
