export function findNode(board, nodeId) {
  if (!board || !board.nodes) return null;
  return board.nodes[nodeId] || null;
}

export function childrenOf(board, nodeId) {
  const node = findNode(board, nodeId);
  if (!node) return [];
  const ids = Array.isArray(node.children) ? node.children : [];
  return ids.map(id => board.nodes[id]).filter(Boolean);
}

export function isContainer(node) {
  return (node?.sys?.shape ?? null) === 'container';
}

export function traverseNodes(board, fn) {
  if (!board || !board.nodes || typeof fn !== 'function') return;
  const rootId = board.rootId;
  if (!rootId || !board.nodes[rootId]) return;
  (function visit(id) {
    const n = board.nodes[id];
    if (!n) return;
    fn(n);
    const children = Array.isArray(n.children) ? n.children : [];
    for (const cid of children) visit(cid);
  })(rootId);
}

export function findList(board, listId) {
  const node = findNode(board, listId);
  return isContainer(node) ? node : null;
}

export function findListLocation(board, listId) {
  if (!board || !board.nodes || !listId) return null;
  const nodes = board.nodes;
  const list = nodes[listId] || null;
  if (!list || !isContainer(list)) return null;
  const parentId = list.parentId || null;
  if (!parentId || !nodes[parentId]) return null;
  const siblings = Array.isArray(nodes[parentId].children) ? nodes[parentId].children : [];
  const containerSiblings = siblings.filter(id => isContainer(nodes[id] ?? null));
  const index = containerSiblings.indexOf(listId);
  return { columnId: parentId, index };
}

export function findColumn(board, columnId) {
  if (!board || !board.nodes || !columnId) return null;
  const column = board.nodes[columnId] || null;
  if (!column) return null;
  const lists = (Array.isArray(column.children) ? column.children : [])
    .map(id => board.nodes[id])
    .filter(Boolean)
    .filter(isContainer)
    .map(n => ({ id: n.id }));
  return { id: columnId, lists };
}

export function findColumnLocation(board, columnId) {
  if (!board || !board.nodes || !columnId) return null;
  const nodes = board.nodes;
  let workspaceId = null;
  for (const id in nodes) {
    const children = Array.isArray(nodes[id]?.children) ? nodes[id].children : [];
    if (children.includes(columnId)) { workspaceId = id; break; }
  }
  const siblings = Array.isArray(nodes[workspaceId]?.children) ? nodes[workspaceId].children : [];
  const index = siblings.indexOf(columnId);
  return { workspaceId, index };
}
