let storeRef = null;
let sendFn = null;

import { applyPatch } from '../../../packages/state/applyPatch.js';
import { showToast } from '../../../packages/ui/toast.js';
import { isContainer } from './domain-utils.js';

export function initEditor(ctx) {
  storeRef = ctx.store;
  sendFn = ctx.send;
}

export function openItem(itemId) {
  const state = storeRef.getState();
  const openTabs = state.board.openTabs ?? [];
  const tabs = openTabs.includes(itemId) ? openTabs : [...openTabs, itemId];
  const board = { ...state.board, openTabs: tabs, activeTab: itemId };
  storeRef.setState(prev => ({ ...prev, board }));
}

export function switchTab(itemId) {
  const state = storeRef.getState();
  if (state.board.activeTab === itemId) return;
  const board = { ...state.board, activeTab: itemId };
  storeRef.setState(prev => ({ ...prev, board }));
}

export function closeTab(itemId) {
  const state = storeRef.getState();
  const tabs = (state.board.openTabs ?? []).filter(id => id !== itemId);
  const board = { ...state.board, openTabs: tabs };
  if (board.activeTab === itemId) {
    board.activeTab = tabs[tabs.length - 1] ?? null;
  }
  storeRef.setState(prev => ({ ...prev, board }));
}

export function closeAllTabs() {
  const state = storeRef.getState();
  const board = { ...state.board, openTabs: [], activeTab: null };
  storeRef.setState(prev => ({ ...prev, board }));
}

export function updateDraft(nodeId, changes) {
  const state = storeRef.getState();
  const board = applyPatch(state.board, { boardId: state.currentBoardId, operations: [{ op: 'node.update', nodeId, ...changes }] });
  storeRef.setState(prev => ({ ...prev, board }));
}

export async function saveItem(itemId) {
  const state = storeRef.getState();
  const found = findNode(state.board, itemId);
  if (!found) return;
  await sendFn('UpdateNode', { nodeId: itemId, changes: {
    title: found.title,
    description: found.description,
    content: found.content,
  }});
  showToast('Item enregistré');
}

// V3 helper: find item location within its parent list-like node
export function findItem(board, itemId) {
  if (!board || !board.nodes || !itemId) return null;
  const nodes = board.nodes;
  const item = nodes[itemId] || null;
  if (!item) return null;
  let parentId = null;
  for (const id in nodes) {
    const n = nodes[id];
    const children = Array.isArray(n?.children) ? n.children : [];
    if (children.includes(itemId)) { parentId = id; break; }
  }
  if (!parentId) return { list: null, item, index: -1 };
  const parent = nodes[parentId];
  const children = Array.isArray(parent?.children) ? parent.children : [];
  const order = children.map(cid => ({ type: isContainer(nodes[cid]) ? 'list' : 'item', id: cid }));
  const itemOrder = order.filter(e => e.type === 'item').map(e => e.id);
  const index = itemOrder.indexOf(itemId);
  const list = { id: parentId, order };
  return { list, item, index };
}

export function collectItems(board) {
  const items = [];
  if (!board || !board.nodes) return items;
  const rootId = board.rootId;
  for (const id in board.nodes) {
    if (id === rootId) continue;
    const n = board.nodes[id];
    if (!n) continue;
    items.push(n);
  }
  return items;
}

export function findNode(board, nodeId) {
  if (!board || !board.nodes) return null;
  return board.nodes[nodeId] || null;
}

function isListNode(node) {
  return isContainer(node);
}
