// Filtering utilities and controller for board view

let storeRef = null;
let sendFn = null;

export function initFilters(ctx) {
  storeRef = ctx.store;
  sendFn = ctx.send;
}

export function normalizeFilterTags(value) {
  if (!Array.isArray(value)) return [];
  const seen = new Set();
  for (const raw of value) {
    if (typeof raw !== 'string') continue;
    const tag = raw.trim();
    if (!tag) continue;
    seen.add(tag);
  }
  if (seen.has('__NONE__')) {
    return ['__NONE__'];
  }
  return Array.from(seen);
}

export function sameStringArray(a, b) {
  if (a.length !== b.length) return false;
  for (let i = 0; i < a.length; i += 1) {
    if (a[i] !== b[i]) {
      return false;
    }
  }
  return true;
}

export function filterBoardView(board, tags) {
  if (!board) return board;
  const normalized = normalizeFilterTags(tags);
  if (!normalized.length) {
    const base = structuredClone(board);
    try { base.workspaces = buildWorkspaces(base); } catch (_) {}
    return base;
  }
  const allowed = new Set(normalized);
  const result = structuredClone(board);
  const nodes = result.nodes || {};
  for (const id in nodes) {
    const n = nodes[id];
    const hasTag = (key) => (Array.isArray(n.tags) ? n.tags : []).some(t => (t.key ?? t.k) === key);
    n._visible = allowed.has('__NONE__') ? (n.tags?.length === 0) : Array.from(allowed).some(k => hasTag(k));
  }
  // Optionnel: masquer les sous-arbres invisibles (ici on garde la structure et laisse le renderer décider)
  // V3 adapter: dérive un tableau de workspaces/columns pour le renderer UI
  try { result.workspaces = buildWorkspaces(result); } catch (_) {}
  return result;
}

function traverseLists(workspaces, fn) {
  for (const ws of workspaces) {
    for (const col of ws.columns) {
      col.lists = col.lists.map(list => mapList(list, fn));
      col.order = rebuildOrder(col);
    }
  }
}

function mapList(list, fn) {
  list = fn(list) ?? list;
  list.lists = list.lists.map(child => mapList(child, fn));
  return list;
}

function rebuildOrder(column) {
  const order = [];
  for (const list of column.lists) {
    order.push({ type: 'list', id: list.id });
  }
  return order;
}

export async function toggleFilter(tag) {
  if (typeof tag !== 'string' || !storeRef || !sendFn) return;
  const state = storeRef.getState();
  const current = normalizeFilterTags(state.filter.tags);
  const nextSet = new Set(current);
  if (nextSet.has(tag)) {
    nextSet.delete(tag);
  } else {
    nextSet.add(tag);
    if (tag === '__NONE__') {
      nextSet.clear();
      nextSet.add('__NONE__');
    } else {
      nextSet.delete('__NONE__');
    }
  }
  const next = normalizeFilterTags(Array.from(nextSet));
  if (sameStringArray(next, current)) return;
  storeRef.setState(prev => ({ ...prev, filter: { tags: next } }));
  const result = await sendFn('UpdateTagFilter', { selected: next });
  if (!result) {
    storeRef.setState(prev => ({ ...prev, filter: { tags: current } }));
  }
}

function buildWorkspaces(boardLike) {
  const nodes = boardLike?.nodes || {};
  const rootId = boardLike?.rootId;
  const root = nodes[rootId] || null;
  const rootChildren = Array.isArray(root?.children) ? root.children : [];
  const workspaceIds = rootChildren.filter(id => id);
  const toNode = (id) => nodes[id] || null;
  if (workspaceIds.length === 0) {
    // V3: aucun workspace implicite
    return [];
  }
  return workspaceIds.map(wsId => ({
    id: wsId,
    title: nodes[wsId]?.title ?? 'Workspace',
    columns: (Array.isArray(nodes[wsId]?.children) ? nodes[wsId].children : []).map(toNode).filter(Boolean),
  }));
}
