import { showToast } from '../../../packages/ui/toast.js';

let appRef = null;
let storeRef = null;
let lastHoverNode = null;
let selectionState = {
  active: false,
  resolve: null,
  reject: null,
  options: null,
};
let overlayEl = null;

export function initSelection({ app, store }) {
  appRef = app ?? null;
  storeRef = store ?? null;
  if (!appRef) return;
  appRef.addEventListener('click', handleSelectionClick, true);
  appRef.addEventListener('contextmenu', handleSelectionContextMenu, true);
  appRef.addEventListener('mousemove', handleSelectionHover, true);
  document.addEventListener('keydown', handleSelectionKeydown, true);
}

export function requestSelection(options = {}) {
  if (!appRef || !storeRef) {
    return Promise.reject(new Error('Selection service unavailable'));
  }
  cancelSelection();
  const opts = normalizeOptions(options);
  return new Promise((resolve, reject) => {
    selectionState = {
      active: true,
      resolve,
      reject,
      options: opts,
    };
    renderOverlay(opts.label);
  });
}

export function cancelSelection() {
  if (!selectionState.active) {
    return;
  }
  const reject = selectionState.reject;
  clearOverlay();
  selectionState = { active: false, resolve: null, reject: null, options: null };
  if (typeof reject === 'function') {
    const error = new Error('Selection cancelled');
    error.name = 'SelectionCancelled';
    reject(error);
  }
}

export function isSelectionActive() {
  return selectionState.active;
}

function normalizeOptions(options) {
  const allowedShapes = Array.isArray(options.allowedShapes) && options.allowedShapes.length
    ? options.allowedShapes
    : ['container', 'leaf'];
  return {
    label: typeof options.label === 'string' && options.label.trim()
      ? options.label.trim()
      : 'Sélectionnez une cible…',
    allowedShapes,
    multi: options.multi === true,
  };
}

function handleSelectionClick(event) {
  if (!selectionState.active) {
    return;
  }
  const cancelBtn = event.target instanceof Element
    ? event.target.closest('[data-selection-cancel]')
    : null;
  if (cancelBtn) {
    event.preventDefault();
    cancelSelection();
    return;
  }
  // Laisser les autres clics fonctionner (pour plier/déplier) sans déclencher la sélection.
}

function handleSelectionKeydown(event) {
  if (!selectionState.active) {
    return;
  }
  if (event.key === 'Escape') {
    event.preventDefault();
    cancelSelection();
  }
}

function handleSelectionContextMenu(event) {
  if (!selectionState.active) {
    return;
  }
  const nodeEl = findNodeElement(event.target);
  if (!nodeEl) {
    return;
  }
  const nodeId = nodeEl.dataset.node || nodeEl.dataset.id || '';
  if (!nodeId) {
    return;
  }
  const state = storeRef.getState();
  const boardId = state?.currentBoardId ?? null;
  const board = state?.board ?? null;
  const node = board?.nodes?.[nodeId] ?? null;
  if (!boardId || !node) {
    showToast('Élément introuvable', { kind: 'warning' });
    return;
  }
  const shape = (node?.sys?.shape ?? null) === 'container' ? 'container' : 'leaf';
  if (!selectionState.options.allowedShapes.includes(shape)) {
    return;
  }
  event.preventDefault();
  event.stopPropagation();
  finishSelection([{ boardId, nodeId, shape }]);
}

function handleSelectionHover(event) {
  if (!selectionState.active) {
    return;
  }
  // Ignorer les éléments interactifs hors nodes (boutons, liens, etc.)
  const interactive = event.target instanceof Element
    ? event.target.closest('[data-action],button,a,input,select,textarea,label')
    : null;
  if (interactive && !interactive.closest('[data-node]')) {
    return;
  }

  const nodeEl = findNodeElement(event.target);
  if (nodeEl === lastHoverNode) {
    return;
  }
  if (lastHoverNode) {
    lastHoverNode.classList.remove('selection-hover');
  }
  lastHoverNode = nodeEl;
  if (lastHoverNode) {
    lastHoverNode.classList.add('selection-hover');
  }
}

function finishSelection(addresses) {
  const resolve = selectionState.resolve;
  clearOverlay();
  selectionState = { active: false, resolve: null, reject: null, options: null };
  if (typeof resolve === 'function') {
    resolve(addresses);
  }
}

function findNodeElement(target) {
  let el = target instanceof Element ? target : null;
  // remonter jusqu'à un hôte de node (pas un sous-élément décoratif)
  while (el && !(el.dataset?.node && isNodeHost(el))) {
    el = el.parentElement;
  }
  if (!el) return null;
  // si plusieurs wrappers portent le même data-node, on prend l'ancêtre le plus haut
  const id = el.dataset.node;
  let cursor = el;
  while (
    cursor.parentElement &&
    cursor.parentElement.dataset?.node === id &&
    isNodeHost(cursor.parentElement)
  ) {
    cursor = cursor.parentElement;
  }
  return cursor;
}

function isNodeHost(element) {
  if (!(element instanceof Element)) return false;
  if (element.dataset?.draggable === 'true') return true;
  return element.classList.contains('item')
    || element.classList.contains('list')
    || element.classList.contains('board-column');
}

function renderOverlay(label) {
  clearOverlay();
  overlayEl = document.createElement('div');
  overlayEl.className = 'selection-overlay';
  overlayEl.innerHTML = `
    <div class="selection-overlay__panel">
      <div class="selection-overlay__label">${label}</div>
      <button type="button" class="selection-overlay__cancel" data-selection-cancel>Annuler</button>
    </div>
  `;
  overlayEl.addEventListener('click', (event) => {
    const btn = event.target instanceof Element
      ? event.target.closest('[data-selection-cancel]')
      : null;
    if (btn) {
      cancelSelection();
    }
  });
  appRef.appendChild(overlayEl);
  appRef.setAttribute('data-selection-active', 'true');
}

function clearOverlay() {
  if (overlayEl && overlayEl.parentElement) {
    overlayEl.parentElement.removeChild(overlayEl);
  }
  overlayEl = null;
  if (lastHoverNode) {
    lastHoverNode.classList.remove('selection-hover');
    lastHoverNode = null;
  }
  if (appRef) {
    appRef.removeAttribute('data-selection-active');
  }
}
