import { goToBoards, goToAccount, goToModules, goToAdmin } from '../../../../packages/services/navigation.js';
import { openFileLibrary } from '../../../../packages/ui/file-library.js';
import '../../../../packages/ui/modal.js';
import { confirmDialog, alertDialog } from '../../../../packages/ui/dialog.js';

export function initUiActions({
  app,
  store,
  send, // commandeur générique
  createBoardAction,
  notifToggle,
  notifClose,
  notifDismissOne,
  notifDismissAll,
  notifLoad,
  toggleAutosavePanel,
  closeAutosavePanel,
  openExportDialog,
  openImportDialog,
  closeDialog,
  openBoardConfig,
  closeBoardConfig,
  setBoardConfigTab,
  submitBoardConfigGeneral,
  copyExportJson,
  downloadExportJson,
  submitImportDialog,
  switchWorkspace,
  createWorkspace,
  renameWorkspace,
  deleteWorkspace,
  createColumn,
  createFolder,
  createColumnItem,
  renameColumn,
  renameList,
  toggleList,
  createItem,
  tagsAddTag,
  tagsRemoveTag,
  editorOpenItem,
  editorSwitchTab,
  editorCloseTab,
  editorCloseAllTabs,
  editorSaveItem,
  filtersToggle,
  editorUpdateDraft,
  setImportMode,
  readImportForm,
  dndMoveItemWithKeyboard,
  dndMoveListWithKeyboard,
  dndMoveColumnWithKeyboard,
  moveNodeViaSelection,
  openContextMenu,
  closeContextMenu,
  logout,
}) {
  if (!app) {
    return () => {};
  }

  const handleClick = (event) => {
    // Supporte les TextNode/SVG en remontant jusqu'à un Element avant d'appeler closest
    let el = event.target instanceof Element ? event.target : (event.target?.parentNode ?? null);
    while (el && !(el instanceof Element)) {
      el = el.parentNode || null;
    }
    const actionEl = el ? el.closest('[data-action]') : null;
    if (!actionEl) return;
    const action = actionEl.dataset.action;

    switch (action) {
      case 'create-board':
        createBoardAction();
        break;
      case 'open-boards':
        goToBoards();
        break;
      case 'open-account':
        goToAccount();
        break;
      case 'open-modules':
        goToModules();
        break;
      case 'open-notifications':
        notifToggle();
        break;
      case 'close-notifications': {
        if (actionEl.classList.contains('notifications-overlay')) {
          const panel = actionEl.querySelector('.notifications-panel');
          if (panel && panel.contains(event.target) && !event.target.matches('[data-action="close-notifications"]')) {
            break;
          }
        }
        notifClose();
        break;
      }
      case 'dismiss-notification':
        notifDismissOne(Number(actionEl.dataset.notification)).catch(console.error);
        break;
      case 'dismiss-all-notifications':
        notifDismissAll().catch(console.error);
        break;
      case 'load-notification-history':
        notifLoad('history').catch(console.error);
        break;
      case 'open-admin':
        goToAdmin();
        break;
      case 'open-file-library':
        (async () => {
          try { await openFileLibrary({ title: 'Mes fichiers', allowUpload: true, allowDelete: true, showSelect: false }); } catch (_) {}
        })();
        break;
      case 'logout':
        logout();
        break;
      case 'toggle-autosave':
        toggleAutosavePanel();
        break;
      case 'close-autosave': {
        if (actionEl.classList.contains('autosave-overlay')) {
          const panel = actionEl.querySelector('.autosave-panel');
          if (panel && panel.contains(event.target) && !event.target.matches('[data-action="close-autosave"]')) {
            break;
          }
        }
        closeAutosavePanel();
        break;
      }
      case 'export-board':
        openExportDialog();
        break;
      case 'import-board':
        openImportDialog();
        break;
      case 'open-board-config':
        openBoardConfig?.();
        break;
      case 'close-board-config': {
        if (actionEl.classList.contains('board-config__overlay')) {
          const container = actionEl.querySelector('[data-config-root]');
          if (container && container.contains(event.target) && !event.target.matches('[data-action="close-board-config"]')) {
            break;
          }
        }
        closeBoardConfig?.();
        break;
      }
      case 'board-config-switch-tab':
        if (actionEl.dataset.tab) {
          setBoardConfigTab?.(actionEl.dataset.tab);
        }
        break;
      case 'close-dialog': {
        if (actionEl.classList.contains('modal-overlay')) {
          const modal = actionEl.querySelector('.modal');
          if (modal && modal.contains(event.target) && !event.target.matches('[data-action="close-dialog"]')) {
            break;
          }
        }
        closeDialog();
        break;
      }
      case 'close-editor-modal': {
        const itemId = actionEl.dataset.item;
        if (itemId) {
          editorCloseTab(itemId);
        } else {
          editorCloseAllTabs();
        }
        break;
      }
      case 'copy-export':
        copyExportJson();
        break;
      case 'download-export':
        downloadExportJson();
        break;
      case 'confirm-import':
        submitImportDialog();
        break;
      case 'switch-workspace':
        switchWorkspace(actionEl.dataset.workspace);
        break;
      case 'create-workspace':
        createWorkspace();
        break;
      case 'rename-workspace':
        renameWorkspace(actionEl.dataset.workspace);
        break;
      case 'delete-workspace':
        deleteWorkspace(actionEl.dataset.workspace);
        break;
      case 'create-column': {
        const row = actionEl.closest('.column-row');
        const ws = row?.dataset?.workspace || store.getState()?.board?.activeWorkspaceId;
        if (!ws) {
          alertDialog({ title: 'Information', message: 'Crée d\'abord un workspace' });
          break;
        }
        createColumn(ws);
        break;
      }
      case 'create-folder':
        createFolder(actionEl.dataset.column, actionEl.dataset.workspace, actionEl.dataset.parentList ?? null);
        break;
      case 'create-item-root':
        createColumnItem(actionEl.dataset.column);
        break;
      case 'add-child':
        createItem(actionEl.dataset.node || actionEl.dataset.list || actionEl.dataset.target);
        break;
      case 'open-list-composer':
        openListComposer(actionEl.dataset.node);
        break;
      case 'composer-cancel':
        closeListComposer(actionEl.dataset.node);
        break;
      case 'composer-add-item':
        submitListComposer(actionEl.dataset.node, 'item').catch(console.error);
        break;
      case 'composer-add-list':
        submitListComposer(actionEl.dataset.node, 'list').catch(console.error);
        break;
      case 'rename-column':
        renameColumn(actionEl.dataset.column);
        break;
      // 'rename-list' supprimé au profit de 'rename-node'
      case 'rename-node':
        renameList(actionEl.dataset.node || actionEl.dataset.list);
        break;
      case 'toggle-list':
        toggleList(actionEl.dataset.list || actionEl.dataset.node);
        break;
      case 'add-item':
        createItem(actionEl.dataset.list);
        break;
      case 'add-tag':
        event.stopPropagation();
        event.preventDefault();
        tagsAddTag(actionEl.dataset.scope, actionEl.dataset.target || actionEl.dataset.node || actionEl.closest('[data-node]')?.dataset.node, actionEl).catch(console.error);
        break;
      case 'open-item':
        editorOpenItem(actionEl.dataset.item || actionEl.dataset.id || actionEl.dataset.node);
        break;
      case 'switch-tab':
        editorSwitchTab(actionEl.dataset.item);
        break;
      case 'close-tab':
        editorCloseTab(actionEl.dataset.item);
        break;
      case 'close-all-tabs':
        editorCloseAllTabs();
        break;
      case 'save-item':
        editorSaveItem(actionEl.dataset.item);
        break;
      case 'remove-tag':
        event.stopPropagation();
        event.preventDefault();
        tagsRemoveTag(actionEl.dataset.scope, actionEl.dataset.target || actionEl.dataset.node || actionEl.closest('[data-node]')?.dataset.node, actionEl.dataset.kind, actionEl.dataset.key).catch(console.error);
        break;
      case 'delete-node':
        (async () => {
          const id = actionEl.dataset.node;
          if (!id) return;
          if (!(await confirmDialog({ title: 'Supprimer', message: 'Supprimer cet élément ?', okLabel: 'Supprimer', cancelLabel: 'Annuler' }))) return;
          await send('DeleteNode', { nodeId: id });
        })();
        break;
      case 'delete-column':
        (async () => {
          const id = actionEl.dataset.column;
          if (!id) return;
          if (!(await confirmDialog({ title: 'Supprimer la colonne', message: 'Supprimer cette colonne ?', okLabel: 'Supprimer', cancelLabel: 'Annuler' }))) return;
          // Colonne = node enfant d’un workspace (v3) ? Si colonne est entité dédiée backend:
          // On peut envoyer MoveColumn/… mais pas DeleteColumn; fallback logique:
          // supprimer via DeleteNode pour v3 si les colonnes sont nodes, sinon ignorer.
          try {
            await send('DeleteNode', { nodeId: id });
          } catch (e) { console.error(e); }
        })();
        break;
      case 'toggle-filter':
        filtersToggle(actionEl.dataset.tag).catch(console.error);
        break;
      case 'filter-none':
        filtersToggle('__NONE__').catch(console.error);
        break;
      case 'context-move-node':
        closeContextMenu?.();
        moveNodeViaSelection?.(actionEl.dataset.node);
        break;
      default:
        break;
    }

    if (actionEl && actionEl.closest('.user-menu__panel')) {
      const panel = actionEl.closest('.user-menu__panel');
      if (panel) {
        panel.hidden = true;
      }
      const trigger = actionEl.closest('sb-menu')?.querySelector('[slot="trigger"]');
      if (trigger) {
        trigger.setAttribute('aria-expanded', 'false');
      }
    }
  };

  const handleChange = (event) => {
    const action = event.target?.dataset?.action;
    if (action === 'set-import-mode') {
      setImportMode(event.target.value);
      return;
    }

    const editor = event.target.closest('[data-editor-item]');
    if (!editor) return;
    const itemId = editor.dataset.editorItem;
    const inputs = editor.querySelectorAll('[data-field]');
    const changes = {};
    inputs.forEach(input => {
      const key = input.dataset.field;
      if (!key) {
        return;
      }
      changes[key] = input.value;
    });
    editorUpdateDraft(itemId, changes);
  };

  const handleSubmit = (event) => {
    const form = event.target;
    if (!(form instanceof HTMLFormElement)) return;
    const action = form.dataset.action;
    if (!action) return;
    switch (action) {
      case 'board-config-save-general':
        event.preventDefault();
        submitBoardConfigGeneral?.(form);
        break;
      default:
        break;
    }
  };

  const handleContextMenu = (event) => {
    if (!(event instanceof MouseEvent) || !openContextMenu) {
      return;
    }
    let el = event.target instanceof Element ? event.target : (event.target?.parentNode ?? null);
    while (el && !(el instanceof Element)) {
      el = el.parentNode || null;
    }
    const nodeEl = el ? el.closest('[data-node]') : null;
    if (!nodeEl) return;
    const nodeId = nodeEl.dataset.node || nodeEl.dataset.id || '';
    if (!nodeId) return;
    event.preventDefault();
    openContextMenu({ x: event.clientX, y: event.clientY, nodeId });
  };

  const handleInput = (event) => {
    const target = event.target;
    if (!(target instanceof HTMLInputElement || target instanceof HTMLTextAreaElement)) {
      return;
    }

    const state = store.getState();
    const dialog = state.dialog;
    if (!dialog || dialog.kind !== 'import') {
      return;
    }
    if (!target.closest('.modal')) {
      return;
    }

    const currentForm = dialog.form ?? {};
    const form = readImportForm(currentForm);
    if (
      form.json === (currentForm.json ?? '') &&
      form.replaceTitle === (currentForm.replaceTitle ?? '') &&
      form.newTitle === (currentForm.newTitle ?? '') &&
      form.workspaceTitle === (currentForm.workspaceTitle ?? 'Workspace')
    ) {
      return;
    }

    store.setState(prev => ({
      ...prev,
      dialog: prev.dialog?.kind === 'import'
        ? { ...prev.dialog, form }
        : prev.dialog,
    }));
  };

  const handleKeydown = (event) => {
    const target = event.target instanceof HTMLElement ? event.target : null;
    const key = event.key;

    if (target?.matches('[data-composer-input]')) {
      const listId = target.dataset.node;
      if (key === 'Escape') {
        event.preventDefault();
        closeListComposer(listId);
        return;
      }
      if (key === 'Enter') {
        event.preventDefault();
        submitListComposer(listId, 'item').catch(console.error);
        return;
      }
    }

    if (key === 'Escape') {
      const state = store.getState();
      if (state.boardConfig?.open) {
        event.preventDefault();
        closeBoardConfig?.();
        return;
      }
      if (state.dialog) {
        event.preventDefault();
        closeDialog();
        return;
      }
      if (state.autosavePanelOpen) {
        event.preventDefault();
        closeAutosavePanel();
        return;
      }
    }

    if (target && (
      target instanceof HTMLInputElement ||
      target instanceof HTMLTextAreaElement ||
      target.isContentEditable ||
      target.closest('[contenteditable="true"]')
    )) {
      return;
    }

    const draggable = target?.closest('[data-draggable="true"]');
    if (!draggable) {
      return;
    }
    const type = draggable.dataset.type;

    if (type === 'item' && (key === 'Enter' || key === ' ')) {
      event.preventDefault();
      editorOpenItem(draggable.dataset.id);
      return;
    }

    if (type === 'list' && (key === 'Enter' || key === ' ')) {
      event.preventDefault();
      const listId = draggable.dataset.node || draggable.dataset.list || draggable.dataset.id;
      editorOpenItem(listId);
      return;
    }

    if (!event.altKey || event.ctrlKey || event.metaKey) {
      return;
    }

    if (type === 'item' && (key === 'ArrowUp' || key === 'ArrowDown')) {
      event.preventDefault();
      const delta = key === 'ArrowUp' ? -1 : 1;
      dndMoveItemWithKeyboard(draggable.dataset.id, delta);
      return;
    }

    if (type === 'list' && (key === 'ArrowUp' || key === 'ArrowDown')) {
      event.preventDefault();
      const delta = key === 'ArrowUp' ? -1 : 1;
      dndMoveListWithKeyboard(draggable.dataset.list, delta);
      return;
    }

    if (type === 'column' && (key === 'ArrowLeft' || key === 'ArrowRight')) {
      event.preventDefault();
      const delta = key === 'ArrowLeft' ? -1 : 1;
      dndMoveColumnWithKeyboard(draggable.dataset.column, delta);
    }
  };

  const handleModalClose = (event) => {
    const target = event.target;
    if (!(target instanceof HTMLElement)) {
      return;
    }
    if (!target.matches('[data-editor-modal]')) {
      return;
    }
    const state = store.getState();
    const active = state.board?.activeTab;
    if (active) {
      editorCloseTab(active);
    } else {
      editorCloseAllTabs();
    }
  };

  const escapeSelector = (value) => {
    if (typeof value !== 'string') return '';
    if (typeof CSS !== 'undefined' && typeof CSS.escape === 'function') {
      return CSS.escape(value);
    }
    return value.replace(/([^a-zA-Z0-9_-])/g, '\\$1');
  };

  const findComposer = (listId) => {
    if (!listId) return null;
    const selector = `.list-composer[data-composer="${escapeSelector(listId)}"]`;
    return app.querySelector(selector);
  };

  const openListComposer = (listId) => {
    const composer = findComposer(listId);
    if (!composer) return;
    const opened = app.querySelectorAll('.list-composer[data-open="true"]');
    opened.forEach(node => {
      if (node instanceof HTMLElement && node !== composer) {
        node.removeAttribute('data-open');
        const inputNode = node.querySelector('[data-composer-input]');
        if (inputNode instanceof HTMLInputElement) {
          inputNode.value = '';
        }
      }
    });
    composer.setAttribute('data-open', 'true');
    let hint = composer.previousElementSibling;
    while (hint && !(hint instanceof HTMLElement)) {
      hint = hint.previousElementSibling;
    }
    if (hint instanceof HTMLElement && hint.classList.contains('list-empty-hint')) {
      hint.setAttribute('data-hidden', 'true');
    }
    const input = composer.querySelector('[data-composer-input]');
    if (input instanceof HTMLInputElement) {
      input.value = '';
      requestAnimationFrame(() => { try { input.focus(); } catch (_) {} });
    }
  };

  const closeListComposer = (listId) => {
    const composer = findComposer(listId);
    if (!composer) return;
    composer.removeAttribute('data-open');
    let hint = composer.previousElementSibling;
    while (hint && !(hint instanceof HTMLElement)) {
      hint = hint.previousElementSibling;
    }
    if (hint instanceof HTMLElement && hint.classList.contains('list-empty-hint')) {
      hint.removeAttribute('data-hidden');
    }
    const input = composer.querySelector('[data-composer-input]');
    if (input instanceof HTMLInputElement) {
      input.value = '';
    }
  };

  const submitListComposer = async (listId, kind = 'item') => {
    const composer = findComposer(listId);
    if (!composer) return;
    const input = composer.querySelector('[data-composer-input]');
    if (!(input instanceof HTMLInputElement)) {
      return;
    }
    const title = input.value.trim();
    if (!title) {
      input.focus();
      return;
    }
    let ok = false;
    if (kind === 'list') {
      ok = await createFolder(null, null, listId, { title });
    } else {
      ok = await createItem(listId, { title });
    }
    if (ok) {
      input.value = '';
      closeListComposer(listId);
    }
  };

  app.addEventListener('click', handleClick);
  app.addEventListener('change', handleChange);
  app.addEventListener('input', handleInput);
  app.addEventListener('submit', handleSubmit);
  app.addEventListener('sb-modal-close', handleModalClose);
  app.addEventListener('contextmenu', handleContextMenu);
  document.addEventListener('keydown', handleKeydown);

  return () => {
    app.removeEventListener('click', handleClick);
    app.removeEventListener('change', handleChange);
    app.removeEventListener('input', handleInput);
    app.removeEventListener('submit', handleSubmit);
    app.removeEventListener('sb-modal-close', handleModalClose);
    app.removeEventListener('contextmenu', handleContextMenu);
    document.removeEventListener('keydown', handleKeydown);
  };
}
