import { escapeHtml, formatDateTime } from './format.js';
import { getRegistered } from '../../../../packages/modules/runtime.js';
import { renameBoard, fetchBoards } from '../../../../packages/services/boards.js';
import { showToast } from '../../../../packages/ui/toast.js';
import { renderGeneralBoardPanel } from '../../../../packages/ui/board-config/general-panel.js';
import { renderLogicBoardPanel } from '../../../../packages/ui/board-config/logic-panel.js';

let storeRef = null;
let csrfProvider = () => '';

if (typeof window !== 'undefined' && !window.__sbBoardConfigSync) {
  window.addEventListener('board-config-panels:updated', () => {
    if (!storeRef) return;
    const state = storeRef.getState?.();
    if (!state) return;
    syncBoardConfigPanels(document, state);
  });
  window.__sbBoardConfigSync = true;
}

export function initBoardConfig({ store, getCsrfToken }) {
  storeRef = store;
  if (typeof getCsrfToken === 'function') {
    csrfProvider = getCsrfToken;
  }
}

export function openBoardConfig(tab = 'general') {
  if (!storeRef) return;
  storeRef.setState(prev => ({
    ...prev,
    boardConfig: {
      ...(prev.boardConfig ?? {}),
      open: true,
      activeTab: tab,
      busy: false,
      error: null,
    },
  }));
}

export function closeBoardConfig() {
  if (!storeRef) return;
  storeRef.setState(prev => ({
    ...prev,
    boardConfig: {
      ...(prev.boardConfig ?? {}),
      open: false,
      busy: false,
      error: null,
    },
  }));
}

export function setBoardConfigTab(tab) {
  if (!storeRef) return;
  storeRef.setState(prev => ({
    ...prev,
    boardConfig: {
      ...(prev.boardConfig ?? {}),
      activeTab: tab,
    },
  }));
}

export function renderBoardConfig(state) {
  const cfg = state.boardConfig ?? { open: false, activeTab: 'general', busy: false };
  if (!cfg.open) {
    return '';
  }
  const activeTab = cfg.activeTab || 'general';
  const contributions = [...getRegistered('board.configPanels')].sort((a, b) => (a.order - b.order) || a.id.localeCompare(b.id));
  const generalTab = { id: 'general', label: 'Général' };
  const logicTab = { id: 'logic', label: 'Logique' };
  const tabs = [generalTab, logicTab, ...contributions.map(c => ({
    id: `panel:${c.id}`,
    label: c.label || normalizeContributionLabel(c.id),
  }))];

  const board = state.boards?.find?.(b => String(b.id) === String(state.currentBoardId)) ?? null;
  const boardTitle = board?.title ?? 'Board';
  const panelsHiddenAttr = activeTab.startsWith('panel:') ? '' : ' hidden';

  const navHtml = tabs.map(tab => {
    const isActive = tab.id === activeTab;
    return `
      <button type="button"
        class="board-config__tab${isActive ? ' is-active' : ''}"
        data-action="board-config-switch-tab"
        data-config-tab
        data-tab="${escapeHtml(tab.id)}">
        ${escapeHtml(tab.label)}
      </button>`;
  }).join('');

  return `
    <div class="modal-overlay board-config__overlay" data-action="close-board-config">
      <div class="board-config" data-config-root data-active-tab="${escapeHtml(activeTab)}" data-stop-propagation>
        <header class="board-config__header">
          <div>
            <h3>Configuration du board</h3>
            <p class="board-config__subtitle">${escapeHtml(boardTitle)}</p>
          </div>
          <button type="button" class="btn btn--icon ghost" data-action="close-board-config" aria-label="Fermer">✕</button>
        </header>
        <div class="board-config__body">
          <nav class="board-config__tabs" data-config-tabs>
            ${navHtml}
          </nav>
          <div class="board-config__content">
            <section class="board-config__panel" data-config-panel="general">
              <div data-config-general-root></div>
            </section>
            <section class="board-config__panel" data-config-panel="logic">
              <div data-config-logic-root></div>
            </section>
            <div class="board-config__panels" data-config-panels${panelsHiddenAttr}>
              <div class="board-config__slot-root" data-config-slot-root data-slot="board.configPanels"></div>
            </div>
          </div>
        </div>
      </div>
    </div>
  `;
}

export function syncBoardConfigPanels(root, state) {
  if (!root) return;
  const configRoot = root.querySelector('[data-config-root]');
  if (!configRoot) return;
  const cfg = state.boardConfig ?? { open: false, activeTab: 'general', busy: false };
  const active = cfg.activeTab ?? 'general';
  configRoot.dataset.activeTab = active;
  const tabs = configRoot.querySelectorAll('[data-config-tab]');
  tabs.forEach(tab => {
    tab.classList.toggle('is-active', tab.dataset.tab === active);
  });

  const general = configRoot.querySelector('[data-config-panel="general"]');
  const logic = configRoot.querySelector('[data-config-panel="logic"]');

  const board = state.boards?.find?.(b => String(b.id) === String(state.currentBoardId)) ?? null;
  const boardTitle = board?.title ?? 'Board';
  const boardId = board?.id ?? state.currentBoardId ?? '';
  const updatedAt = state.boardMeta?.updatedAt ?? board?.updated_at ?? null;
  const createdAt = state.boardMeta?.createdAt ?? board?.created_at ?? null;

  if (general) {
    general.hidden = active !== 'general';
    const generalRoot = general.querySelector('[data-config-general-root]');
    if (generalRoot) {
      renderGeneralBoardPanel(generalRoot, {
        boardTitle,
        boardId: String(boardId ?? ''),
        busy: cfg.busy,
        error: cfg.error,
        updatedAt: updatedAt ? formatDateTime(updatedAt) : '',
        createdAt: createdAt ? formatDateTime(createdAt) : '',
        thumbnailPublicId: typeof board?.thumbnail_public_id === 'string' ? board.thumbnail_public_id : '',
      });
    }
  }

  if (logic) {
    logic.hidden = active !== 'logic';
      const logicRoot = logic.querySelector('[data-config-logic-root]');
      if (logicRoot) {
        const ruleCount = Array.isArray(state.board?.rules)
          ? state.board.rules.length
          : Array.isArray(state.boardMeta?.rules)
            ? state.boardMeta.rules.length
            : 0;
        const rulesVersion = state.rulesVersion ?? null;

        renderLogicBoardPanel(logicRoot, {
          ruleCount,
          updatedAt: updatedAt ? formatDateTime(updatedAt) : '',
          rulesVersion: rulesVersion ?? '',
          busy: cfg.busy,
          packs: collectPackRules(state.modules),
        });
      }
  }

  const slotRoot = configRoot.querySelector('[data-config-slot-root]');
  if (!slotRoot) return;
  const contributions = Array.from(slotRoot.querySelectorAll('[data-slot-contribution]'));
  contributions.forEach(node => {
    const id = node.dataset.slotContribution;
    if (!id) return;
    const panelId = `panel:${id}`;
    node.dataset.configPanel = panelId;
    node.classList.add('board-config__panel');
    if (!node.dataset.configLabel && node.dataset.slotContributionLabel) {
      node.dataset.configLabel = node.dataset.slotContributionLabel;
    }
    node.hidden = panelId !== active;
  });

  const panelsContainer = configRoot.querySelector('[data-config-panels]');
  if (panelsContainer) {
    panelsContainer.hidden = !active.startsWith('panel:');
  }

  if (active === 'general') {
    const focusTarget = general?.querySelector('[data-config-autofocus]');
    if (focusTarget && typeof focusTarget.focus === 'function') {
      setTimeout(() => {
        try { focusTarget.focus(); focusTarget.select?.(); } catch (_) {}
      }, 50);
    }
  }
}

export async function submitGeneralSettings(form) {
  if (!storeRef) return false;
  const state = storeRef.getState();
  const boardId = state.currentBoardId;
  if (!boardId) {
    showToast('Aucun board actif', { kind: 'warning' });
    return false;
  }
  const input = form.querySelector('[data-ref="board-title"]');
  const title = (input?.value ?? '').trim();
  if (!title) {
    if (input) {
      input.value = state.boards?.find?.(b => String(b.id) === String(boardId))?.title ?? '';
      input.focus();
    }
    showToast('Le nom du board est obligatoire', { kind: 'warning' });
    return false;
  }
  const currentBoard = state.boards?.find?.(b => String(b.id) === String(boardId)) ?? null;
  const currentTitle = currentBoard?.title ?? '';
  const doRename = title !== currentTitle;
  const initialThumb = form?.dataset?.thumbnailInitial ?? '';
  const nextThumb = form?.dataset?.thumbnailCurrent ?? initialThumb;
  const doSetThumb = nextThumb !== initialThumb;

  setBusy(true);
  storeRef.setState(prev => ({
    ...prev,
    boardConfig: { ...(prev.boardConfig ?? {}), error: null },
  }));

  try {
    if (doRename) {
      await renameBoard(boardId, title, csrfProvider());
    }
    if (doSetThumb) {
      const { setBoardThumbnail } = await import('../../../../packages/services/boards.js');
      await setBoardThumbnail(boardId, nextThumb, csrfProvider());
    }
    const boards = await fetchBoards();
    storeRef.setState(prev => ({
      ...prev,
      boards,
      boardConfig: { ...(prev.boardConfig ?? {}), busy: false, error: null },
    }));
    const actions = [];
    if (doRename) actions.push('nom');
    if (doSetThumb) actions.push('vignette');
    const label = actions.length ? `Paramètres enregistrés (${actions.join(' + ')})` : 'Aucun changement';
    showToast(label, { kind: 'success' });
    return true;
  } catch (error) {
    console.error(error);
    const message = (error?.payload?.error ?? error?.message ?? 'Impossible de renommer le board');
    storeRef.setState(prev => ({
      ...prev,
      boardConfig: { ...(prev.boardConfig ?? {}), busy: false, error: message },
    }));
    showToast('Impossible de renommer le board', { kind: 'error' });
    return false;
  } finally {
    setBusy(false);
  }
}

function collectPackRules(modulesState) {
  if (!modulesState || !modulesState.rules || !modulesState.rules.byPack) {
    return [];
  }

  const catalog = Array.isArray(modulesState.catalog?.packs) ? modulesState.catalog.packs : null;
  const byPack = modulesState.rules.byPack;
  const packIds = Object.keys(byPack).sort((a, b) => a.localeCompare(b));

  return packIds.map((packId) => {
    const rules = Array.isArray(byPack[packId]) ? byPack[packId] : [];
    const normalizedRules = rules
      .filter(rule => rule && typeof rule.id === 'string')
      .map(rule => ({
        id: rule.id,
        description: typeof rule.description === 'string' ? rule.description : null,
        enabled: rule.enabled !== false,
      }))
      .sort((a, b) => a.id.localeCompare(b.id));

    if (!normalizedRules.length) {
      return null;
    }

    return {
      id: packId,
      name: resolvePackName(packId, catalog),
      rules: normalizedRules,
    };
  }).filter(Boolean);
}

function resolvePackName(packId, catalog) {
  if (!catalog) return null;
  const entry = catalog.find(item => item && item.id === packId);
  if (!entry) return null;
  if (typeof entry.name === 'string' && entry.name.trim()) {
    return entry.name.trim();
  }
  if (typeof entry.label === 'string' && entry.label.trim()) {
    return entry.label.trim();
  }
  return null;
}

function setBusy(busy) {
  if (!storeRef) return;
  storeRef.setState(prev => ({
    ...prev,
    boardConfig: { ...(prev.boardConfig ?? {}), busy },
  }));
}

function normalizeContributionLabel(id) {
  if (!id) return 'Extension';
  const parts = id.split(':');
  return parts.length > 1 ? capitalize(parts[parts.length - 1]) : capitalize(id);
}

function capitalize(str) {
  if (!str) return '';
  const clean = String(str).replace(/[_-]+/g, ' ').trim();
  if (!clean) return '';
  return clean.charAt(0).toUpperCase() + clean.slice(1);
}
