// Persistent Editor Modal Manager
// - Keeps a single <sb-modal data-editor-modal> alive outside #app
// - Avoids full-page rerender flicker when state/autosave updates occur
// - Delegates actions locally (switch/close/save/updateDraft) for content inside the modal

import { renderEditorContent } from './editor.js';
import { collectItems as editorCollectItems } from '../../modules/editor.js';
import {
  switchTab as editorSwitchTab,
  closeTab as editorCloseTab,
  closeAllTabs as editorCloseAllTabs,
  saveItem as editorSaveItem,
  updateDraft as editorUpdateDraft,
} from '../../modules/editor.js';
import { renderSlot as runtimeRenderSlot, registries as runtimeRegistries } from '../../../../packages/modules/runtime.js';

let storeRef = null;
let modalEl = null;
let disposeSlots = [];
let mounted = false;
let lastActiveItemId = null;

export function initEditorModalManager(ctx) {
  storeRef = ctx?.store || null;
  if (!storeRef) return;

  // Subscribe to store changes and keep modal mounted while tabs are open
  storeRef.subscribe(handleStateChange);
  // Initial check
  handleStateChange(storeRef.getState());
}

function handleStateChange(state) {
  const openTabs = state?.board?.openTabs ?? [];
  if (!openTabs.length) {
    closeAndCleanup();
    return;
  }

  // Ensure modal exists and is open
  ensureModal();
  openModal();

  // Only re-render content if active item changed or modal not mounted yet
  const activeId = determineActiveTab(state.board, openTabs);
  if (!mounted || activeId !== lastActiveItemId) {
    lastActiveItemId = activeId;
    renderContent(state);
  }
}

function determineActiveTab(board, openTabs) {
  const requested = board?.activeTab;
  if (requested && openTabs.includes(requested)) return requested;
  return openTabs[0] ?? null;
}

function findActiveItem(state) {
  const openTabs = state?.board?.openTabs ?? [];
  const activeId = determineActiveTab(state?.board, openTabs);
  if (!activeId) return { id: null, item: null };
  const items = editorCollectItems(state?.board);
  const activeItem = items.find(i => i.id === activeId) ?? null;
  return { id: activeId, item: activeItem };
}

function ensureModal() {
  if (modalEl && modalEl.isConnected) return;
  disposeSlots = [];
  modalEl = document.createElement('sb-modal');
  modalEl.setAttribute('data-editor-modal', 'true');
  document.body.appendChild(modalEl);

  // Local event delegation for actions inside modal content
  modalEl.addEventListener('click', handleClick, true);
  modalEl.addEventListener('change', handleChange, true);
  modalEl.addEventListener('sb-modal-close', handleModalClose);
  mounted = true;
}

function openModal() {
  if (!modalEl) return;
  modalEl.setAttribute('open', '');
  try { if (typeof modalEl.show === 'function') modalEl.show(); } catch (_) {}
}

function closeAndCleanup() {
  lastActiveItemId = null;
  mounted = false;
  if (disposeSlots && disposeSlots.length) {
    for (const d of disposeSlots) { try { if (typeof d === 'function') d(); } catch (_) {} }
  }
  disposeSlots = [];
  if (modalEl && modalEl.isConnected) {
    try { modalEl.removeEventListener('click', handleClick, true); } catch (_) {}
    try { modalEl.removeEventListener('change', handleChange, true); } catch (_) {}
    try { modalEl.removeEventListener('sb-modal-close', handleModalClose); } catch (_) {}
    try { modalEl.removeAttribute('open'); } catch (_) {}
    try { modalEl.parentNode && modalEl.parentNode.removeChild(modalEl); } catch (_) {}
  }
  modalEl = null;
}

function renderContent(state) {
  if (!modalEl) return;

  // Clear existing nodes and slot mount cleanups
  if (disposeSlots && disposeSlots.length) {
    for (const d of disposeSlots) { try { if (typeof d === 'function') d(); } catch (_) {} }
  }
  disposeSlots = [];
  while (modalEl.firstChild) modalEl.removeChild(modalEl.firstChild);

  const { id: activeId, item } = findActiveItem(state);
  const hasSidePanel = Array.isArray(item?.tags)
    ? item.tags.some(t => {
        const k = String(t.key ?? t.k ?? '');
        return k === 'type/note' || k === 'type/text';
      })
    : false;
  const layoutClass = hasSidePanel ? 'editor-modal editor-layout' : 'editor-modal';
  const modalWidth = hasSidePanel ? 'min(1080px, 95vw)' : 'min(700px, 90vw)';
  try { modalEl.style.setProperty('--sb-modal-width', modalWidth); } catch (_) {}

  const shell = document.createElement('div');
  shell.className = layoutClass;
  shell.setAttribute('role', 'document');

  // Header + Tabs
  const header = document.createElement('div');
  header.className = 'editor-modal__main';
  header.innerHTML = `
    <header class="editor-modal__header">
      <button type="button" class="btn btn--icon ghost editor-modal__close-left" data-action="close-editor-modal" data-item="${activeId ?? ''}" aria-label="Fermer" title="Fermer">✕</button>
      <h2>Édition d'item</h2>
      <div class="editor-modal__header-actions"></div>
    </header>
    <div class="editor-modal__tabs">${renderTabs(state, activeId)}</div>
    <div class="editor-modal__body">${renderEditorContent(item || {}, state)}</div>
  `;
  shell.appendChild(header);

  if (hasSidePanel) {
    const aside = document.createElement('aside');
    aside.className = 'editor-modal__side';
    aside.innerHTML = `<div class="editor-side__content" data-slot="item.sidePanel" data-node="${activeId ?? ''}"></div>`;
    shell.appendChild(aside);
  }

  modalEl.appendChild(shell);

  // Mount slots inside the freshly rendered body
  try {
    const body = shell.querySelector('.editor-modal__body');
    if (body) {
      for (const holder of body.querySelectorAll('[data-slot]')) {
        const slotId = holder.getAttribute('data-slot');
        const nodeId = holder.getAttribute('data-node');
        if (!slotId) continue;
        const disposer = runtimeRenderSlot(slotId, { state, item, registries: runtimeRegistries }, holder);
        if (typeof disposer === 'function') disposeSlots.push(disposer);
      }
    }
    const sideHolder = shell.querySelector('[data-slot="item.sidePanel"]');
    if (sideHolder) {
      const disposer = runtimeRenderSlot('item.sidePanel', { state, item, registries: runtimeRegistries }, sideHolder);
      if (typeof disposer === 'function') disposeSlots.push(disposer);
    }
  } catch (e) {
    console.error('EDITOR_MODAL_SLOTS_MOUNT_FAILED', e);
  }
}

function renderTabs(state, activeId) {
  const openTabs = state?.board?.openTabs ?? [];
  const items = editorCollectItems(state?.board);
  const multipleTabs = openTabs.length > 1;
  const tabsMarkup = openTabs
    .map(tabId => {
      const item = items.find(i => i.id === tabId);
      if (!item) return '';
      const label = escapeHtml(item.title ?? item.id);
      const active = tabId === activeId;
      return `
        <div class="editor-tab ${active ? 'editor-tab--active' : ''}">
          <button type="button" class="editor-tab__label" data-action="switch-tab" data-item="${item.id}">
            ${label}
          </button>
          ${multipleTabs ? `<button type=\"button\" class=\"editor-tab__close\" data-action=\"close-tab\" data-item=\"${item.id}\" aria-label=\"Fermer l'onglet\">×</button>` : ''}
        </div>
      `;
    })
    .join('');
  const closeAllButton = multipleTabs
    ? `<button type="button" class="editor-tabs__close-all" data-action="close-all-tabs" title="Fermer tous les onglets" aria-label="Fermer tous les onglets">
         <span class="editor-tabs__close-icon">✕</span>
         <span class="editor-tabs__close-label">Fermer tout</span>
       </button>`
    : '';
  return `${tabsMarkup}${closeAllButton}`;
}

function handleModalClose() {
  // Mirror logic from actions.js: close active or all
  if (!storeRef) return;
  const state = storeRef.getState();
  const active = state.board?.activeTab;
  if (active) editorCloseTab(active); else editorCloseAllTabs();
}

function handleClick(event) {
  const target = event.target instanceof Element ? event.target : null;
  if (!target) return;
  const actionEl = target.closest('[data-action]');
  if (!actionEl) return;
  const action = actionEl.getAttribute('data-action');
  switch (action) {
    case 'switch-tab':
      editorSwitchTab(actionEl.getAttribute('data-item'));
      break;
    case 'close-tab':
      editorCloseTab(actionEl.getAttribute('data-item'));
      break;
    case 'close-all-tabs':
      editorCloseAllTabs();
      break;
    case 'save-item':
      editorSaveItem(actionEl.getAttribute('data-item'));
      break;
    case 'close-editor-modal': {
      const itemId = actionEl.getAttribute('data-item');
      if (itemId) editorCloseTab(itemId); else editorCloseAllTabs();
      break;
    }
    default:
      break;
  }
}

function handleChange(event) {
  const target = event.target instanceof Element ? event.target : null;
  if (!target) return;
  const editorRoot = target.closest('[data-editor-item]');
  if (!editorRoot) return;
  const itemId = editorRoot.getAttribute('data-editor-item');
  const inputs = editorRoot.querySelectorAll('[data-field]');
  const changes = {};
  inputs.forEach(input => {
    const key = input.getAttribute('data-field');
    if (!key) return;
    changes[key] = input.value;
  });
  editorUpdateDraft(itemId, changes);
}

function escapeHtml(str) {
  return String(str ?? '').replace(/[&<>"']/g, s => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[s]));
}
