const adapters = new Map();
let orderedAdapters = [];

function cloneMeta(meta) {
  if (!meta || typeof meta !== 'object') {
    return undefined;
  }
  return { ...meta };
}

function normalizeAdapter(input) {
  if (!input || typeof input !== 'object') {
    return null;
  }
  const id = typeof input.id === 'string' && input.id.trim().length ? input.id.trim() : null;
  if (!id) {
    return null;
  }
  const matches = typeof input.matches === 'function' ? input.matches : () => true;
  const create = typeof input.create === 'function' ? input.create : null;
  if (!create) {
    return null;
  }
  const priority = Number.isFinite(input.priority) ? Number(input.priority) : 100;
  const label = typeof input.label === 'string' && input.label.trim().length ? input.label.trim() : null;
  return { id, matches, create, priority, label, meta: cloneMeta(input.meta) };
}

function rebuildOrder() {
  orderedAdapters = Array.from(adapters.values())
    .sort((a, b) => (a.priority - b.priority) || a.id.localeCompare(b.id));
}

export function registerDocAdapter(adapter) {
  const normalized = normalizeAdapter(adapter);
  if (!normalized) {
    return;
  }
  adapters.set(normalized.id, normalized);
  rebuildOrder();
}

function safeMatches(entry, criteria) {
  if (!entry || typeof entry.matches !== 'function') {
    return false;
  }
  try {
    return !!entry.matches(criteria);
  } catch (error) {
    console.error('DOC_ADAPTER_MATCH_FAILED', entry.id, error);
    return false;
  }
}

export function listDocAdapters() {
  return orderedAdapters.map((entry) => entry.id);
}

export function resolveDocAdapter(criteria = {}) {
  const preferredId = typeof criteria.preferredId === 'string' ? criteria.preferredId : null;
  if (preferredId && adapters.has(preferredId)) {
    return adapters.get(preferredId);
  }
  for (const entry of orderedAdapters) {
    if (safeMatches(entry, criteria)) {
      return entry;
    }
  }
  return fallbackAdapter;
}

const fallbackAdapter = {
  id: 'core:fallback-textarea',
  label: 'Texte brut',
  priority: 1000,
  matches: () => true,
  create: createFallbackRenderer,
};

function createFallbackRenderer({ container, session } = {}) {
  if (!container) {
    return () => {};
  }
  container.innerHTML = '';

  const wrapper = document.createElement('div');
  wrapper.className = 'sb-doc-fallback';

  const editor = document.createElement('textarea');
  editor.className = 'sb-doc-fallback__textarea';
  editor.rows = 12;
  editor.placeholder = 'Associez un fichier pour éditer.';
  editor.spellcheck = true;

  const footer = document.createElement('div');
  footer.className = 'sb-doc-fallback__footer';

  const saveState = document.createElement('span');
  saveState.className = 'sb-doc-fallback__state';

  const saveButton = document.createElement('button');
  saveButton.type = 'button';
  saveButton.className = 'btn primary';
  saveButton.textContent = 'Enregistrer';

  footer.append(saveState, saveButton);
  wrapper.append(editor, footer);
  container.appendChild(wrapper);

  const update = (snapshot) => {
    const hasAttachment = !!(snapshot && snapshot.attachment);
    const busy = !!(snapshot && (snapshot.loading || snapshot.saving));
    const value = snapshot && typeof snapshot.content === 'string' ? snapshot.content : '';
    if (editor.value !== value) {
      editor.value = value;
    }
    editor.disabled = !hasAttachment || busy;
    editor.placeholder = hasAttachment ? 'Écrivez en texte brut…' : 'Associez un fichier pour éditer.';

    const saveDisabled = !hasAttachment || busy || !(snapshot && snapshot.dirty);
    saveButton.disabled = saveDisabled;
    const saveTitle = !hasAttachment
      ? 'Aucun fichier sélectionné'
      : busy
        ? (snapshot.loading ? 'Chargement en cours' : 'Enregistrement en cours')
        : (!snapshot.dirty ? 'Aucune modification à enregistrer' : 'Enregistrer la note');
    if (saveTitle) {
      saveButton.title = saveTitle;
    } else {
      saveButton.removeAttribute('title');
    }

    if (!hasAttachment) {
      saveState.textContent = '';
    } else if (snapshot.loading) {
      saveState.textContent = 'Chargement…';
    } else if (snapshot.saving) {
      saveState.textContent = 'Enregistrement…';
    } else if (snapshot.dirty) {
      saveState.textContent = 'Modifications non enregistrées';
    } else {
      saveState.textContent = 'Enregistré';
    }
  };

  const handleInput = () => {
    if (!session || typeof session.setContent !== 'function') {
      return;
    }
    session.setContent(editor.value);
  };

  const handleKeydown = (event) => {
    if ((event.metaKey || event.ctrlKey) && (event.key === 's' || event.key === 'S')) {
      event.preventDefault();
      saveButton.click();
    }
  };

  const handleSave = async () => {
    if (!session || typeof session.save !== 'function') {
      return;
    }
    try {
      await session.save();
    } catch (error) {
      console.error('DOC_FALLBACK_SAVE_FAILED', error);
    }
  };

  editor.addEventListener('input', handleInput);
  editor.addEventListener('keydown', handleKeydown);
  saveButton.addEventListener('click', handleSave);

  let unsubscribe = () => {};
  if (session && typeof session.onChange === 'function') {
    unsubscribe = session.onChange((snapshot) => {
      update(snapshot);
    });
  }
  if (session && typeof session.getState === 'function') {
    update(session.getState());
  } else {
    update(null);
  }

  return () => {
    unsubscribe?.();
    editor.removeEventListener('input', handleInput);
    editor.removeEventListener('keydown', handleKeydown);
    saveButton.removeEventListener('click', handleSave);
  };
}

