import { registerDocAdapter } from '../adapter-registry.js';
import { showToast } from '../../ui/toast.js';
import { openFileLibrary } from '../../ui/file-library.js';

const CSS_HREF = '/assets/vendor/quill/quill.snow.css';
const JS_SRC = '/assets/vendor/quill/quill.min.js';
const JS_FALLBACK = '/assets/vendor/quill/quill.js';

function ensureCssOnce() {
  if (typeof document === 'undefined') return;
  const id = 'quill-snow-css';
  if (document.getElementById(id)) return;
  const link = document.createElement('link');
  link.id = id; link.rel = 'stylesheet'; link.href = CSS_HREF;
  document.head.appendChild(link);
}

function loadQuill() {
  if (typeof window !== 'undefined' && window.Quill) return Promise.resolve(window.Quill);
  return new Promise((resolve, reject) => {
    if (typeof document === 'undefined') { reject(new Error('NO_DOCUMENT')); return; }
    const scriptId = 'quill-umd';
    const existing = document.getElementById(scriptId);
    const onLoad = () => window.Quill ? resolve(window.Quill) : reject(new Error('QUILL_NOT_AVAILABLE'));
    const onError = () => {
      // Try fallback non-minified file if minified is missing
      if (script && script.src.endsWith('quill.min.js')) {
        const fb = document.createElement('script');
        fb.id = scriptId; fb.src = JS_FALLBACK; fb.async = true;
        fb.addEventListener('load', onLoad, { once: true });
        fb.addEventListener('error', () => reject(new Error('QUILL_LOAD_FAILED')), { once: true });
        document.head.appendChild(fb);
      } else {
        reject(new Error('QUILL_LOAD_FAILED'));
      }
    };
    if (existing) {
      existing.addEventListener('load', onLoad, { once: true });
      existing.addEventListener('error', onError, { once: true });
      return;
    }
    const script = document.createElement('script');
    script.id = scriptId; script.src = JS_SRC; script.async = true;
    script.addEventListener('load', onLoad, { once: true });
    script.addEventListener('error', onError, { once: true });
    document.head.appendChild(script);
  });
}

const adapter = {
  id: 'ext:html-quill',
  label: 'Texte riche (HTML)',
  priority: 40,
  matches({ slot, attachment }) {
    if (slot && slot !== 'text') return false;
    if (!attachment) return true;
    const mime = String(attachment?.mimeType || '').toLowerCase();
    const name = String(attachment?.name || '').toLowerCase();
    return mime === 'text/html' || /\.html?$/.test(name);
  },
  async create({ container, session } = {}) {
    if (!container) return () => {};
    ensureCssOnce();
    container.innerHTML = '';
    const wrapper = document.createElement('div');
    wrapper.className = 'sb-note-editor sb-note-editor--quill';
    const host = document.createElement('div');
    host.className = 'sb-note-editor__host';
    const footer = document.createElement('div');
    footer.className = 'sb-note-panel__footer';
    const saveState = document.createElement('span');
    saveState.className = 'sb-note-panel__save-state';
    const saveButton = document.createElement('button');
    saveButton.type = 'button'; saveButton.className = 'btn primary'; saveButton.textContent = 'Enregistrer';
    footer.append(saveState, saveButton);
    wrapper.append(host, footer);
    container.appendChild(wrapper);

    let Quill;
    try { Quill = await loadQuill(); } catch (e) { console.error('QUILL_LOAD_FAILED', e); return () => {}; }

    const toolbar = [
      [{ header: [1, 2, 3, false] }],
      ['bold', 'italic', 'underline', 'strike'],
      [{ color: [] }, { background: [] }],
      [{ align: [] }],
      [{ list: 'ordered' }, { list: 'bullet' }],
      ['blockquote', 'code'],
      ['link'],
      ['clean']
    ];

    // Ensure comfortable initial height
    try { host.style.minHeight = '260px'; } catch (_) {}
    const editor = new Quill(host, { theme: 'snow', modules: { toolbar } });
    try {
      const container = host.querySelector('.ql-container');
      const inner = host.querySelector('.ql-editor');
      if (container) container.style.minHeight = '260px';
      if (inner) inner.style.minHeight = '260px';
    } catch (_) {}
    const initialRaw = session?.getState?.().content || '';
    const initial = extractEditableHtml(initialRaw);
    try { editor.root.innerHTML = String(initial || ''); } catch (_) {}

    const local = { updating: false };
    const updateFromSession = (snapshot) => {
      const hasAttachment = !!snapshot?.attachment;
      const busy = !!(snapshot?.loading || snapshot?.saving);
      saveButton.disabled = !hasAttachment || busy || !snapshot?.dirty;
      const saveTitle = !hasAttachment
        ? 'Aucun fichier sélectionné'
        : busy
          ? (snapshot.loading ? 'Chargement en cours' : 'Enregistrement en cours')
          : (!snapshot.dirty ? 'Aucune modification à enregistrer' : 'Enregistrer le document');
      if (saveTitle) saveButton.title = saveTitle; else saveButton.removeAttribute('title');
      if (!local.updating) {
        const raw = String(snapshot?.content || '');
        const html = extractEditableHtml(raw);
        if (html !== editor.root.innerHTML) {
          try { local.updating = true; editor.root.innerHTML = html; } finally { setTimeout(() => { local.updating = false; }, 0); }
        }
      }
      if (!hasAttachment) saveState.textContent = '';
      else if (snapshot.loading) saveState.textContent = 'Chargement…';
      else if (snapshot.saving) saveState.textContent = 'Enregistrement…';
      else if (snapshot.dirty) saveState.textContent = 'Modifications non enregistrées';
      else saveState.textContent = 'Enregistré';
    };

    editor.on('text-change', () => {
      if (local.updating) return;
      const html = editor.root.innerHTML;
      session?.setContent?.(html);
    });

    host.addEventListener('keydown', (event) => {
      if ((event.metaKey || event.ctrlKey) && (event.key === 's' || event.key === 'S')) { event.preventDefault(); saveButton.click(); }
    });

    saveButton.addEventListener('click', async () => {
      const snapshot = session?.getState?.();
      if (!snapshot?.attachment) { showToast('Aucun fichier sélectionné.'); return; }
      try {
        const result = await session.save();
        if (result?.ok) showToast('Document enregistré.');
        else if (result?.conflict) { showToast('Le fichier a été modifié ailleurs. Rechargement…'); await session.load(); }
        else if (!result?.cancelled) showToast('Impossible d’enregistrer le document.');
      } catch (e) { console.error('DOC_HTML_SAVE_FAILED', e); showToast('Impossible d’enregistrer le document.'); }
    });

    // Add a library image button into the toolbar
    try { injectLibraryImageButton(wrapper, editor); } catch (_) {}

    const unsubscribe = session?.onChange?.((s) => updateFromSession(s)) || (() => {});
    if (session?.getState) updateFromSession(session.getState());

    return () => { unsubscribe?.(); try { editor.off?.('text-change'); } catch (_) {} };
  },
};

registerDocAdapter(adapter);
export default adapter;

function injectLibraryImageButton(wrapper, editor) {
  const toolbar = wrapper.querySelector('.ql-toolbar');
  if (!toolbar) return;
  const btn = document.createElement('button');
  btn.type = 'button'; btn.className = 'ql-insertLibraryImage'; btn.title = 'Insérer une image (bibliothèque)';
  btn.innerHTML = '<span style="font-weight:600">🖼️</span>';
  btn.addEventListener('click', async () => {
    try {
      const result = await openFileLibrary({ title: 'Bibliothèque d’images', accept: 'image/*', autoSelectUploaded: true });
      const url = result?.file?.url || '';
      if (!url) return;
      const range = editor.getSelection(true) || { index: editor.getLength(), length: 0 };
      editor.insertEmbed(range.index, 'image', url, 'user');
      editor.setSelection(range.index + 1, 0, 'user');
    } catch (e) { console.error('LIB_IMAGE_INSERT_FAILED', e); }
  });
  toolbar.appendChild(btn);
}

// ——————————————————————————————————————————————
// Helpers: extract editable fragment from full HTML docs
// ——————————————————————————————————————————————
function isFullHtmlDoc(html) {
  if (typeof html !== 'string') return false;
  const s = html.trim().slice(0, 120).toLowerCase();
  return s.startsWith('<!doctype') || s.includes('<html');
}

function extractEditableHtml(raw) {
  const input = String(raw || '');
  if (!isFullHtmlDoc(input)) {
    return input;
  }
  try {
    const tmp = document.createElement('div');
    tmp.innerHTML = input;
    const ql = tmp.querySelector('.ql-editor');
    if (ql) return ql.innerHTML;
    const body = tmp.querySelector('body');
    if (body) return body.innerHTML;
    return input;
  } catch (_) {
    return input;
  }
}
