import { registerDocAdapter } from '../adapter-registry.js';
import { showToast } from '../../ui/toast.js';
import { openFileLibrary } from '../../ui/file-library.js';

// Toast UI Editor adapter (UMD CDN bundle expected)
// Requires the following vendorized assets (no telemetry, local files):
//  - /assets/vendor/toastui/toastui-editor-all.min.js
//  - /assets/vendor/toastui/toastui-editor.min.css

const CSS_HREF = '/assets/vendor/toastui/toastui-editor.min.css';
const JS_SRC = '/assets/vendor/toastui/toastui-editor-all.min.js';

function ensureCssOnce() {
  if (typeof document === 'undefined') return;
  const id = 'toastui-editor-css';
  if (document.getElementById(id)) return;
  const link = document.createElement('link');
  link.id = id;
  link.rel = 'stylesheet';
  link.href = CSS_HREF;
  document.head.appendChild(link);
}

function loadToastUiEditor() {
  if (typeof window !== 'undefined' && window.toastui && window.toastui.Editor) {
    return Promise.resolve(window.toastui.Editor);
  }
  return new Promise((resolve, reject) => {
    if (typeof document === 'undefined') {
      reject(new Error('NO_DOCUMENT'));
      return;
    }
    const scriptId = 'toastui-editor-umd';
    const existing = document.getElementById(scriptId);
    if (existing) {
      existing.addEventListener('load', () => {
        if (window.toastui && window.toastui.Editor) resolve(window.toastui.Editor);
        else reject(new Error('TOASTUI_EDITOR_NOT_AVAILABLE'));
      }, { once: true });
      existing.addEventListener('error', () => reject(new Error('TOASTUI_EDITOR_LOAD_FAILED')), { once: true });
      return;
    }
    const script = document.createElement('script');
    script.id = scriptId;
    script.src = JS_SRC;
    script.async = true;
    script.addEventListener('load', () => {
      if (window.toastui && window.toastui.Editor) resolve(window.toastui.Editor);
      else reject(new Error('TOASTUI_EDITOR_NOT_AVAILABLE'));
    }, { once: true });
    script.addEventListener('error', () => reject(new Error('TOASTUI_EDITOR_LOAD_FAILED')), { once: true });
    document.head.appendChild(script);
  });
}

const adapter = {
  id: 'ext:markdown-toastui',
  label: 'Markdown (Toast UI)',
  // Higher priority than textarea so it’s preferred for markdown content
  priority: 40,
  matches({ slot, attachment }) {
    if (slot && slot !== 'note') return false;
    if (!attachment) return true; // allow creating new note before file selection
    const mime = typeof attachment?.mimeType === 'string' ? attachment.mimeType.toLowerCase() : '';
    // Prefer markdown/text types
    return !mime || mime.startsWith('text/') || mime.includes('markdown');
  },
  async create({ container, session } = {}) {
    if (!container) return () => {};
    ensureCssOnce();

    container.innerHTML = '';
    const wrapper = document.createElement('div');
    wrapper.className = 'sb-note-editor sb-note-editor--toastui';

    const mount = document.createElement('div');
    mount.className = 'sb-note-editor__host';

    const footer = document.createElement('div');
    footer.className = 'sb-note-panel__footer';
    const saveState = document.createElement('span');
    saveState.className = 'sb-note-panel__save-state';
    const saveButton = document.createElement('button');
    saveButton.type = 'button';
    saveButton.className = 'btn primary';
    saveButton.textContent = 'Enregistrer';
    footer.append(saveState, saveButton);

    wrapper.append(mount, footer);
    container.appendChild(wrapper);

    let Editor = null;
    try {
      Editor = await loadToastUiEditor();
    } catch (error) {
      console.error('TOASTUI_LOAD_FAILED', error);
      // Graceful fallback: create a basic textarea
      const fallback = document.createElement('textarea');
      fallback.className = 'sb-note-panel__editor';
      fallback.rows = 12;
      mount.appendChild(fallback);
      return () => {};
    }

    const initial = session && typeof session.getState === 'function' ? (session.getState().content || '') : '';
    const editor = new Editor({
      el: mount,
      height: '380px',
      initialEditType: 'wysiwyg',
      previewStyle: 'vertical',
      initialValue: String(initial || ''),
      usageStatistics: false, // disable telemetry
      toolbarItems: [
        ['heading', 'bold', 'italic', 'strike'],
        ['hr', 'quote'],
        ['ul', 'ol', 'task'],
        // Remove default 'image' to avoid duplicate with our library button
        ['table', 'link', 'code', 'codeblock'],
      ],
    });

    const local = { updating: false };

    const updateFromSession = (snapshot) => {
      const hasAttachment = !!(snapshot && snapshot.attachment);
      const busy = !!(snapshot && (snapshot.loading || snapshot.saving));
      const value = snapshot && typeof snapshot.content === 'string' ? snapshot.content : '';
      if (!local.updating) {
        const current = editor.getMarkdown();
        if (current !== value) {
          try {
            local.updating = true;
            editor.setMarkdown(value || '');
          } finally {
            // Small async break to avoid immediate change echo
            setTimeout(() => { local.updating = false; }, 0);
          }
        }
      }
      saveButton.disabled = !hasAttachment || busy || !(snapshot && snapshot.dirty);
      const saveTitle = !hasAttachment
        ? 'Aucun fichier sélectionné'
        : busy
          ? (snapshot.loading ? 'Chargement en cours' : 'Enregistrement en cours')
          : (!snapshot.dirty ? 'Aucune modification à enregistrer' : 'Enregistrer la note');
      if (saveTitle) saveButton.title = saveTitle; else saveButton.removeAttribute('title');

      if (!hasAttachment) {
        saveState.textContent = '';
      } else if (snapshot.loading) {
        saveState.textContent = 'Chargement…';
      } else if (snapshot.saving) {
        saveState.textContent = 'Enregistrement…';
      } else if (snapshot.dirty) {
        saveState.textContent = 'Modifications non enregistrées';
      } else {
        saveState.textContent = 'Enregistré';
      }
    };

    const handleEditorChange = () => {
      if (!session || typeof session.setContent !== 'function') return;
      if (local.updating) return;
      const value = editor.getMarkdown();
      session.setContent(value);
    };

    const handleKeydown = (event) => {
      if ((event.metaKey || event.ctrlKey) && (event.key === 's' || event.key === 'S')) {
        event.preventDefault();
        saveButton.click();
      }
    };

    const handleSave = async () => {
      if (!session || typeof session.save !== 'function' || typeof session.getState !== 'function') return;
      const snapshot = session.getState();
      if (!snapshot.attachment) {
        showToast('Aucun fichier Markdown sélectionné.');
        return;
      }
      try {
        const result = await session.save();
        if (result && result.ok) {
          showToast('Note enregistrée.');
        } else if (result && result.conflict) {
          showToast('Le fichier a été modifié ailleurs. Rechargement…');
          const reload = await session.load();
          if (!reload || !reload.ok) {
            showToast('Impossible de recharger le fichier Markdown.');
          }
        } else if (!result || !result.cancelled) {
          showToast('Impossible d’enregistrer la note.');
        }
      } catch (error) {
        console.error('DOC_MARKDOWN_SAVE_FAILED', error);
        showToast('Impossible d’enregistrer la note.');
      }
    };

    editor.on('change', handleEditorChange);
    mount.addEventListener('keydown', handleKeydown);
    saveButton.addEventListener('click', handleSave);

    // Inject a small toolbar button to insert an image from the file library
  try {
    addLibraryImageButton(wrapper, editor);
  } catch (_) {}

    const unsubscribe = session && typeof session.onChange === 'function'
      ? session.onChange((snapshot) => updateFromSession(snapshot))
      : () => {};
    if (session && typeof session.getState === 'function') updateFromSession(session.getState());

    return () => {
      unsubscribe?.();
      try { editor.off('change', handleEditorChange); } catch (_) {}
      try { mount.removeEventListener('keydown', handleKeydown); } catch (_) {}
      try { saveButton.removeEventListener('click', handleSave); } catch (_) {}
      try { editor.destroy(); } catch (_) {}
    };
  },
};

registerDocAdapter(adapter);

export default adapter;

// ——————————————————————————————————————————————
// Helpers — toolbar injection + image insertion
// ——————————————————————————————————————————————

function addLibraryImageButton(wrapper, editor) {
  const toolbar = wrapper.querySelector('.toastui-editor-defaultUI-toolbar');
  if (!toolbar) return;
  const group = toolbar.querySelector('.toastui-editor-toolbar-group') || toolbar;

  const btn = document.createElement('button');
  // Reuse Toast UI icon base class and the "image" sprite for a consistent look
  btn.className = 'toastui-editor-toolbar-icons image';
  btn.type = 'button';
  btn.title = 'Insérer une image (bibliothèque)';
  // Ensure sizing consistent with default buttons
  btn.style.width = '32px';
  btn.style.height = '32px';
  btn.style.margin = '7px 5px';
  btn.addEventListener('click', async () => {
    try {
      const result = await openFileLibrary({
        title: 'Bibliothèque d’images',
        description: 'Sélectionnez une image à insérer dans la note.',
        accept: 'image/*',
        filter: isImageFile,
        selectLabel: 'Insérer',
        autoSelectUploaded: true,
      });
      const file = result?.file;
      const url = file?.url || '';
      if (!url) return;
      const alt = file?.name || 'image';
      // Use editor command to insert image node to avoid escaping issues
      try {
        if (typeof editor.exec === 'function') {
          editor.exec('addImage', { imageUrl: url, altText: alt });
        } else if (typeof editor.insertText === 'function') {
          // Fallback for older builds: insert raw markdown
          const md = `![${escapeMarkdownText(alt)}](${url})`;
          editor.insertText(md);
        }
      } catch (_) { /* noop */ }
    } catch (error) {
      console.error('LIB_IMAGE_INSERT_FAILED', error);
    }
  });

  // Append at end of first group
  group.appendChild(btn);

  return btn;
}

function isImageFile(file) {
  const mime = String(file?.mimeType || file?.type || '').toLowerCase();
  if (mime.startsWith('image/')) return true;
  const name = String(file?.name || '').toLowerCase();
  return /\.(png|jpe?g|gif|webp|svg)$/i.test(name);
}

function escapeMarkdownText(text) {
  return String(text ?? '').replace(/[\\\[\]\(\)\*\_\`]/g, (m) => `\\${m}`);
}
