const STRING_FIELDS = ['name', 'mimeType', 'sizeLabel', 'checksum', 'url', 'downloadName', 'origin'];
const NUMBER_FIELDS = ['byteSize', 'updatedAt'];

export function buildAttachmentFromFile(file, options = {}) {
  if (!file || typeof file !== 'object') {
    return null;
  }
  if (file.attachment && typeof file.attachment === 'object') {
    const cloned = cloneAttachment(file.attachment);
    if (typeof options.kind === 'string' && options.kind) {
      cloned.kind = options.kind;
    }
    if (!cloned.fileId) {
      cloned.fileId = resolveFileId(file);
    }
    if (!cloned.kind) {
      cloned.kind = 'user';
    }
    if (typeof options.origin === 'string' && options.origin) {
      cloned.origin = options.origin;
    }
    if (!cloned.origin) {
      cloned.origin = 'user-files';
    }
    if (!cloned.url && typeof file.url === 'string' && file.url) {
      cloned.url = file.url;
    }
    if (!cloned.downloadName && typeof file.downloadName === 'string' && file.downloadName) {
      cloned.downloadName = file.downloadName;
    }
    if (cloned.byteSize == null && Number.isFinite(file.byteSize)) {
      cloned.byteSize = Number(file.byteSize);
    }
    if (!cloned.sizeLabel && typeof file.sizeLabel === 'string' && file.sizeLabel) {
      cloned.sizeLabel = file.sizeLabel;
    }
    if (typeof options.role === 'string' && options.role) {
      cloned.meta = {
        ...(cloned.meta || {}),
        role: options.role,
      };
    }
    if (typeof options.meta === 'object' && options.meta !== null) {
      cloned.meta = {
        ...(cloned.meta || {}),
        ...options.meta,
      };
    }
    return cloned;
  }
  const fileId = resolveFileId(file);
  if (!fileId) {
    return null;
  }
  const attachment = {
    kind: typeof options.kind === 'string' && options.kind ? options.kind : 'user',
    fileId,
  };
  STRING_FIELDS.forEach((key) => {
    const value = file[key];
    if (typeof value === 'string' && value) {
      attachment[key] = value;
    }
  });
  const numericSize = Number.isFinite(file.byteSize) ? Number(file.byteSize) : null;
  if (numericSize !== null) {
    attachment.byteSize = numericSize;
  }
  const updatedAt = Number(file.updatedAt);
  if (Number.isFinite(updatedAt) && updatedAt > 0) {
    attachment.updatedAt = updatedAt;
  }
  if (typeof options.origin === 'string' && options.origin) {
    attachment.origin = options.origin;
  }
  if (!attachment.origin) {
    attachment.origin = 'user-files';
  }
  if (typeof file.meta === 'object' && file.meta !== null) {
    attachment.meta = cloneMeta(file.meta);
  }
  if (typeof options.role === 'string' && options.role) {
    attachment.meta = {
      ...(attachment.meta || {}),
      role: options.role,
    };
  }
  if (typeof options.meta === 'object' && options.meta !== null) {
    attachment.meta = {
      ...(attachment.meta || {}),
      ...options.meta,
    };
  }
  return attachment;
}

export function getFileAttachment(props, slot) {
  if (!props || typeof props !== 'object' || !slot || typeof slot !== 'string') {
    return null;
  }
  const files = props.files;
  if (!files || typeof files !== 'object') {
    return null;
  }
  const raw = files[slot];
  if (!raw || typeof raw !== 'object') {
    return null;
  }
  const fileId = resolveFileId(raw);
  if (!fileId) {
    return null;
  }
  const attachment = { kind: typeof raw.kind === 'string' && raw.kind ? raw.kind : 'user', fileId };
  STRING_FIELDS.forEach((key) => {
    const value = raw[key];
    if (typeof value === 'string' && value) {
      attachment[key] = value;
    }
  });
  NUMBER_FIELDS.forEach((key) => {
    const value = raw[key];
    if (Number.isFinite(value)) {
      attachment[key] = Number(value);
    }
  });
  if (raw.meta && typeof raw.meta === 'object') {
    attachment.meta = cloneMeta(raw.meta);
  }
  return attachment;
}

export function withFileAttachment(props, slot, attachment) {
  const base = props && typeof props === 'object' ? props : {};
  const next = { ...base };
  const attachments = cloneFileAttachments(base.files);
  if (attachment && typeof attachment === 'object') {
    attachments[slot] = cloneAttachment(attachment);
  } else {
    delete attachments[slot];
  }
  if (Object.keys(attachments).length > 0) {
    next.files = attachments;
  } else {
    delete next.files;
  }
  return next;
}

export function cloneFileAttachments(source) {
  if (!source || typeof source !== 'object') {
    return {};
  }
  const entries = {};
  Object.keys(source).forEach((slot) => {
    const value = source[slot];
    if (value && typeof value === 'object') {
      entries[slot] = cloneAttachment(value);
    }
  });
  return entries;
}

function resolveFileId(value) {
  if (!value || typeof value !== 'object') {
    return '';
  }
  if (typeof value.fileId === 'string' && value.fileId) {
    return value.fileId;
  }
  if (typeof value.id === 'string' && value.id) {
    return value.id;
  }
  if (typeof value.id === 'number' && Number.isFinite(value.id)) {
    return String(value.id);
  }
  return '';
}

function cloneAttachment(value) {
  const attachment = {};
  if (!value || typeof value !== 'object') {
    return attachment;
  }
  if (typeof value.kind === 'string' && value.kind) {
    attachment.kind = value.kind;
  }
  const fileId = resolveFileId(value);
  if (fileId) {
    attachment.fileId = fileId;
  }
  STRING_FIELDS.forEach((key) => {
    const field = value[key];
    if (typeof field === 'string' && field) {
      attachment[key] = field;
    }
  });
  NUMBER_FIELDS.forEach((key) => {
    const field = value[key];
    if (Number.isFinite(field)) {
      attachment[key] = Number(field);
    }
  });
  if (value.meta && typeof value.meta === 'object') {
    attachment.meta = cloneMeta(value.meta);
  }
  return attachment;
}

function cloneMeta(meta) {
  if (!meta || typeof meta !== 'object') {
    return meta;
  }
  if (Array.isArray(meta)) {
    return meta.map((entry) => {
      if (!entry || typeof entry !== 'object') {
        return entry;
      }
      return cloneMeta(entry);
    });
  }
  const result = {};
  Object.keys(meta).forEach((key) => {
    const value = meta[key];
    if (!value || typeof value !== 'object') {
      result[key] = value;
      return;
    }
    result[key] = cloneMeta(value);
  });
  return result;
}
