function renderMarkdownInternal(input, depth = 0) {
  const text = typeof input === 'string' ? input : '';
  const normalized = text.replace(/\r\n?/g, '\n');
  if (depth > 10) {
    return `<p>${escapeHtml(normalized)}</p>`;
  }
  const lines = normalized.split('\n');
  const html = [];
  let paragraph = [];
  let listType = null;
  let listItems = [];
  let blockquoteLines = [];
  let inCodeBlock = false;
  let codeLang = '';
  let codeLines = [];

  const flushParagraph = () => {
    if (!paragraph.length) {
      return;
    }
    const block = paragraph.join('\n');
    const parts = block.split('\n').map((line) => renderInline(line, depth + 1));
    const body = parts.join('<br />');
    html.push(`<p>${body || '&nbsp;'}</p>`);
    paragraph = [];
  };

  const flushList = () => {
    if (!listType) {
      return;
    }
    html.push(`<${listType}>${listItems.join('')}</${listType}>`);
    listType = null;
    listItems = [];
  };

  const flushBlockquote = () => {
    if (!blockquoteLines.length) {
      return;
    }
    const inner = blockquoteLines.join('\n');
    const rendered = renderMarkdownInternal(inner, depth + 1);
    html.push(`<blockquote>${rendered}</blockquote>`);
    blockquoteLines = [];
  };

  const flushCodeBlock = () => {
    if (!inCodeBlock) {
      return;
    }
    const escaped = escapeHtml(codeLines.join('\n'));
    const langClass = codeLang ? ` class="language-${escapeClassName(codeLang)}"` : '';
    html.push(`<pre><code${langClass}>${escaped}</code></pre>`);
    inCodeBlock = false;
    codeLang = '';
    codeLines = [];
  };

  for (let i = 0; i < lines.length; i += 1) {
    const rawLine = lines[i];
    const trimmed = rawLine.trim();

    if (trimmed.startsWith('```')) {
      if (inCodeBlock) {
        flushParagraph();
        flushList();
        flushBlockquote();
        flushCodeBlock();
        continue;
      }
      flushParagraph();
      flushList();
      flushBlockquote();
      inCodeBlock = true;
      codeLang = trimmed.slice(3).trim().toLowerCase();
      codeLines = [];
      continue;
    }

    if (inCodeBlock) {
      codeLines.push(rawLine);
      continue;
    }

    if (trimmed === '') {
      flushParagraph();
      flushList();
      flushBlockquote();
      continue;
    }

    if (trimmed.startsWith('>')) {
      flushParagraph();
      flushList();
      const content = trimmed.replace(/^>+\s*/, '');
      blockquoteLines.push(content);
      continue;
    }

    flushBlockquote();

    const headingMatch = trimmed.match(/^(#{1,6})\s+(.*)$/);
    if (headingMatch) {
      flushParagraph();
      flushList();
      const level = Math.min(6, headingMatch[1].length);
      const title = renderInline(headingMatch[2], depth + 1);
      html.push(`<h${level}>${title}</h${level}>`);
      continue;
    }

    if (/^(-{3,}|_{3,}|\*{3,})$/.test(trimmed)) {
      flushParagraph();
      flushList();
      html.push('<hr />');
      continue;
    }

    const unorderedMatch = trimmed.match(/^([-*+])\s+(.*)$/);
    const orderedMatch = unorderedMatch ? null : trimmed.match(/^(\d+)\.\s+(.*)$/);
    if (unorderedMatch || orderedMatch) {
      flushParagraph();
      const type = unorderedMatch ? 'ul' : 'ol';
      const content = unorderedMatch ? unorderedMatch[2] : orderedMatch[2];
      if (listType && listType !== type) {
        flushList();
      }
      listType = type;
      listItems.push(`<li>${renderInline(content, depth + 1) || '&nbsp;'}</li>`);
      continue;
    }

    paragraph.push(trimmed);
  }

  flushCodeBlock();
  flushParagraph();
  flushList();
  flushBlockquote();

  const output = html.join('');
  if (!output) {
    return '<p class="sb-note-editor__preview-empty">Aucun contenu à afficher.</p>';
  }
  return output;
}

export function renderMarkdown(input) {
  return renderMarkdownInternal(input, 0);
}

function renderInline(text, depth = 0) {
  if (!text) {
    return '';
  }
  if (depth > 6) {
    return escapeHtml(text);
  }
  let result = '';
  let i = 0;
  while (i < text.length) {
    const char = text[i];
    if (char === '!' && text[i + 1] === '[') {
      const endAlt = text.indexOf(']', i + 2);
      if (endAlt !== -1 && text[endAlt + 1] === '(') {
        const endUrl = findClosingParen(text, endAlt + 2);
        if (endUrl !== -1) {
          const alt = text.slice(i + 2, endAlt);
          const url = sanitizeUrl(text.slice(endAlt + 2, endUrl).trim(), { allowMailto: false });
          if (url) {
            result += `<img src="${escapeHtmlAttr(url)}" alt="${escapeHtmlAttr(alt)}" loading="lazy" />`;
            i = endUrl + 1;
            continue;
          }
        }
      }
    }
    if (char === '[') {
      const endLabel = text.indexOf(']', i + 1);
      if (endLabel !== -1 && text[endLabel + 1] === '(') {
        const endUrl = findClosingParen(text, endLabel + 2);
        if (endUrl !== -1) {
          const label = text.slice(i + 1, endLabel);
          const url = sanitizeUrl(text.slice(endLabel + 2, endUrl).trim(), { allowMailto: true });
          if (url) {
            result += `<a href="${escapeHtmlAttr(url)}" target="_blank" rel="noopener noreferrer">${renderInline(label, depth + 1)}</a>`;
            i = endUrl + 1;
            continue;
          }
        }
      }
    }
    if ((char === '*' || char === '_') && text[i + 1] === char) {
      const end = text.indexOf(char + char, i + 2);
      if (end !== -1) {
        const inner = text.slice(i + 2, end);
        result += `<strong>${renderInline(inner, depth + 1)}</strong>`;
        i = end + 2;
        continue;
      }
    }
    if (char === '*' || char === '_') {
      const end = text.indexOf(char, i + 1);
      if (end !== -1) {
        const inner = text.slice(i + 1, end);
        result += `<em>${renderInline(inner, depth + 1)}</em>`;
        i = end + 1;
        continue;
      }
    }
    if (char === '~' && text[i + 1] === '~') {
      const end = text.indexOf('~~', i + 2);
      if (end !== -1) {
        const inner = text.slice(i + 2, end);
        result += `<del>${renderInline(inner, depth + 1)}</del>`;
        i = end + 2;
        continue;
      }
    }
    if (char === '`') {
      const end = text.indexOf('`', i + 1);
      if (end !== -1) {
        const code = text.slice(i + 1, end);
        result += `<code>${escapeHtml(code)}</code>`;
        i = end + 1;
        continue;
      }
    }
    result += escapeHtml(char);
    i += 1;
  }
  return result;
}

function escapeHtml(value) {
  return String(value)
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&#39;');
}

function escapeHtmlAttr(value) {
  return escapeHtml(value);
}

function escapeClassName(value) {
  return String(value).replace(/[^a-z0-9_+-]/gi, '');
}

function sanitizeUrl(url, { allowMailto = false } = {}) {
  const value = String(url || '').trim();
  if (!value) {
    return '';
  }
  const lower = value.toLowerCase();
  if (lower.startsWith('http://') || lower.startsWith('https://')) {
    return value;
  }
  if (allowMailto && lower.startsWith('mailto:')) {
    return value;
  }
  return '';
}

function findClosingParen(text, start) {
  let depth = 0;
  for (let i = start; i < text.length; i += 1) {
    const char = text[i];
    if (char === '(') {
      depth += 1;
    } else if (char === ')') {
      if (depth === 0) {
        return i;
      }
      depth -= 1;
    }
  }
  return -1;
}
