import { requestJson } from '../services/http.js';
import { showToast } from '../ui/toast.js';
import '../ui/tabs.js';

const registry = new Map();
const mountedCleanups = new WeakMap(); // container -> disposer
const defaultTags = Object.freeze({ system: [], user: [] });
const datasetCache = new Map();
const datasetPromises = new Map();
const datasetEtags = new Map();
const EMPTY_CATEGORIES = Object.freeze({ id: 'org:categories', version: 0, items: [], byKey: {} });
const fallbackContributions = new Set();

const rawAssetVersion = typeof window !== 'undefined' && typeof window.__ASSET_VERSION__ === 'string'
  ? window.__ASSET_VERSION__
  : null;
const assetVersion = rawAssetVersion && rawAssetVersion !== '__ASSET_VERSION__'
  ? rawAssetVersion
  : null;

if (typeof window !== 'undefined' && !window.__sbDatasetInvalidation) {
  window.addEventListener('org:categories:updated', () => {
    invalidateDataset('org:categories');
  });
  window.__sbDatasetInvalidation = true;
}

let environment = {
  sendCommand: null,
  tagsSelector: () => defaultTags,
};

function normalizeContribution(slotId, c) {
  const render = typeof c.render === 'function' ? c.render : () => {};
  const order = Number.isFinite(c.order) ? c.order : 100;
  const packPrefix = typeof c.pack === 'string' && c.pack ? `${c.pack}:` : '';
  const id = typeof c.id === 'string' && c.id.length ? `${packPrefix}${c.id}` : `${packPrefix}anon:${slotId}:${Math.random().toString(36).slice(2)}`;
  const label = typeof c.label === 'string' && c.label.trim().length ? c.label.trim() : null;
  const pack = typeof c.pack === 'string' && c.pack ? c.pack : null;
  return { id, slot: slotId, order, render, label, pack };
}

function register(slotId, contribution) {
  if (!registry.has(slotId)) registry.set(slotId, []);
  registry.get(slotId).push(contribution);
}

export function getRegistered(slotId) {
  return registry.get(slotId) || [];
}

export async function loadPackSlots() {
  try {
    const response = await requestJson('/api/packs/slots', { method: 'GET' });
    const slots = Array.isArray(response?.slots) ? response.slots : [];
    for (const s of slots) {
      const modulePath = s.module || s.component;
      const slotId = s.slot || s.id;
      const exportName = s.export || 'default';
      if (typeof modulePath !== 'string' || typeof slotId !== 'string') continue;
      const baseModulePath = modulePath.split('?')[0];
      // Sanitize: allow only /assets/packs/<id>/... paths
      if (!/^\/assets\/packs\/[a-z0-9-]+\/.+\.js$/.test(baseModulePath)) {
        console.warn('PACK_SLOT_MODULE_REJECTED', modulePath);
        continue;
      }
      try {
        const url = withAssetVersion(modulePath);
        const mod = await import(url);
        const exposed = exportName === 'default' ? (mod.default ?? mod) : mod[exportName];
        const list = Array.isArray(exposed) ? exposed : [exposed];
        let loaded = false;
        for (const raw of list) {
          if (!raw) continue;
          if (typeof raw.render !== 'function') {
            notifySlotIssue(raw.slot || slotId, s.pack, new Error('render() manquant'));
            registerFallback(raw.slot || slotId, s.pack, 'Contribution invalide (render manquant).');
            continue;
          }
          const contrib = normalizeContribution(raw.slot || slotId, { ...raw, pack: s.pack || raw.pack });
          register(contrib.slot, contrib);
          loaded = true;
        }
        if (!loaded) {
          registerFallback(slotId, s.pack, 'Aucune contribution exposée par le module.');
        }
      } catch (error) {
        console.error('PACK_SLOT_LOAD_FAILED', slotId, error);
        notifySlotIssue(slotId, s.pack, error);
        registerFallback(slotId, s.pack, error?.message);
      }
    }
  } catch (error) {
    console.error('PACK_SLOTS_FETCH_FAILED', error);
  }
}

export async function renderSlot(slotId, ctx, container) {
  const prev = mountedCleanups.get(container);
  if (typeof prev === 'function') {
    try { prev(); } catch (_) {}
  }

  while (container.firstChild) {
    container.removeChild(container.firstChild);
  }

  const list = [...getRegistered(slotId)].sort((a, b) => (a.order - b.order) || a.id.localeCompare(b.id));
  const cleanups = [];
  const enriched = await prepareContext(slotId, ctx);

  // Special handling: side panel can render as tabs only if multiple contributions
  if (slotId === 'item.sidePanel' && list.length > 1) {
    const tabs = document.createElement('sb-tabs');
    for (const c of list) {
      const panel = document.createElement('sb-tab');
      if (c.label) panel.setAttribute('label', c.label);
      panel.dataset.slotContribution = c.id;
      if (c.label) panel.dataset.slotContributionLabel = c.label;
      if (c.pack) panel.dataset.slotContributionPack = c.pack;
      tabs.appendChild(panel);
      try {
        const result = c.render(panel, enriched);
        const dispose = result && typeof result.then === 'function' ? await result : result;
        if (typeof dispose === 'function') cleanups.push(dispose);
      } catch (e) {
        console.error(`MODULE_SLOT_RENDER_FAILED ${slotId}`, e);
      }
    }
    container.appendChild(tabs);
  } else {
    for (const c of list) {
      const holder = document.createElement('div');
      holder.dataset.slotContribution = c.id;
      if (c.label) {
        holder.dataset.slotContributionLabel = c.label;
      }
      if (c.pack) {
        holder.dataset.slotContributionPack = c.pack;
      }
      container.appendChild(holder);
      try {
        const result = c.render(holder, enriched);
        const dispose = result && typeof result.then === 'function' ? await result : result;
        if (typeof dispose === 'function') cleanups.push(dispose);
      } catch (e) {
        console.error(`MODULE_SLOT_RENDER_FAILED ${slotId}`, e);
      }
    }
  }

  const disposer = () => cleanups.splice(0).forEach(fn => { try { fn(); } catch (_) {} });
  mountedCleanups.set(container, disposer);
  return disposer;
}

// Lightweight datasets facade for slots (expects backend endpoints to exist)
export const registries = {
  datasets: {
    async get(id) {
      if (datasetCache.has(id)) return datasetCache.get(id);
      if (datasetPromises.has(id)) return datasetPromises.get(id);

      const p = requestJson(`/api/packs/datasets/${encodeURIComponent(id)}`, { method: 'GET' })
        .then(res => {
          const etag = typeof res?.etag === 'string' ? res.etag : '';
          if (etag) datasetEtags.set(id, etag); else datasetEtags.delete(id);
          const data = res?.data ?? null;
          datasetCache.set(id, data);
          datasetPromises.delete(id);
          return data;
        })
        .catch(err => { datasetPromises.delete(id); throw err; });

      datasetPromises.set(id, p);
      return p;
    },

    async patch(id, partial) {
      // Base courante (404 toléré = création from scratch)
      let current = null;
      try { current = await this.get(id); }
      catch (err) { if (!(err && err.status === 404)) throw err; }

      const base = current && typeof current === 'object' ? { ...current } : {};
      const next = { ...base, ...(partial && typeof partial === 'object' ? partial : {}) };

      const etag = datasetEtags.get(id) || '';

      try {
        const upd = await requestJson('/api/commands', {
          method: 'POST',
          body: { type: 'PackDataset.Upsert', payload: { id, data: next, ifMatch: etag || undefined } }
        });
        const nextEtag = typeof upd?.etag === 'string' ? upd.etag : '';
        if (nextEtag) datasetEtags.set(id, nextEtag); else datasetEtags.delete(id);

        const fresh = upd?.data ?? null;
        datasetCache.set(id, fresh);
        datasetPromises.delete(id);
        return upd; // { ok:true, data, etag }
      } catch (error) {
        if (error && error.status === 412) { dropDatasetEtag(id); invalidateDataset(id); }
        throw error;
      }
    },

    async remove(id, key) {
      // Obtenir un ETag si absent pour éviter les 412 fantômes
      if (!datasetEtags.has(id)) { try { await this.get(id); } catch (_) {} }

      const etag = datasetEtags.get(id) || '';

      try {
        const del = await requestJson('/api/commands', {
          method: 'POST',
          body: { type: 'PackDataset.Delete', payload: { id, key, ifMatch: etag || undefined } }
        });
        const nextEtag = typeof del?.etag === 'string' ? del.etag : '';
        if (nextEtag) datasetEtags.set(id, nextEtag); else datasetEtags.delete(id);

        const fresh = del?.data ?? null; // peut contenir unchanged:true
        datasetCache.set(id, fresh);
        datasetPromises.delete(id);
        return del; // { ok:true, (unchanged?:true), data, etag }
      } catch (error) {
        if (error && error.status === 412) { dropDatasetEtag(id); invalidateDataset(id); }
        throw error;
      }
    }
  }
};

async function prepareContext(slotId, ctx) {
  const base = enrichContext(ctx);
  if (slotId === 'item.badges') {
    const categories = await loadDataset('org:categories', normalizeCategoriesDataset);
    base.datasets.categories = categories ?? EMPTY_CATEGORIES;
  }
  return base;
}

function enrichContext(ctx) {
  const state = ctx?.state ?? null;
  const item = ctx?.item ?? null;
  const tags = safeTags(ctx);
  const commands = createCommandApi();
  return {
    ...ctx,
    state,
    item,
    commands,
    registries,
    datasets: {},
    tags,
  };
}

async function loadDataset(id, normalize) {
  if (datasetCache.has(id)) {
    return datasetCache.get(id);
  }
  if (datasetPromises.has(id)) {
    return datasetPromises.get(id);
  }
  const promise = registries.datasets.get(id)
    .then(data => {
      const normalized = typeof normalize === 'function' ? normalize(data) : data;
      datasetCache.set(id, normalized);
      datasetPromises.delete(id);
      return normalized;
    })
    .catch(error => {
      console.error('PACK_DATASET_LOAD_FAILED', id, error);
      invalidateDataset(id);
      return null;
    });
  datasetPromises.set(id, promise);
  return promise;
}

function normalizeCategoriesDataset(raw) {
  if (!raw || typeof raw !== 'object') {
    return EMPTY_CATEGORIES;
  }
  const version = Number.isFinite(raw.version) ? Number(raw.version) : 0;
  const items = Array.isArray(raw.items) ? raw.items.filter(entry => entry && typeof entry.key === 'string') : [];
  const byKey = {};
  for (const entry of items) {
    const key = entry.key;
    if (typeof key !== 'string' || key.length === 0) continue;
    byKey[key] = {
      key,
      label: typeof entry.label === 'string' ? entry.label : key,
      icon: typeof entry.icon === 'string' ? entry.icon : '',
      themeColor: typeof entry.themeColor === 'string' ? entry.themeColor : null,
    };
  }
  return { id: raw.id ?? 'org:categories', version, items, byKey };
}

function invalidateDataset(id) {
  datasetCache.delete(id);
  datasetPromises.delete(id);
  // ⚠️ On ne touche PAS à l’ETag ici (géré explicitement sur 412)
}

function dropDatasetEtag(id) {
  datasetEtags.delete(id);
}

function withAssetVersion(url) {
  if (!assetVersion || typeof url !== 'string') {
    return url;
  }
  if (url.includes('__ASSET_VERSION__')) {
    return url;
  }
  const hasQuery = url.includes('?');
  const versionParam = `v=${encodeURIComponent(assetVersion)}`;
  if (url.includes(versionParam)) {
    return url;
  }
  return `${url}${hasQuery ? '&' : '?'}${versionParam}`;
}

function notifySlotIssue(slotId, packId, error) {
  const key = `${slotId}::${packId || 'unknown'}`;
  if (fallbackContributions.has(`toast:${key}`)) {
    return;
  }
  fallbackContributions.add(`toast:${key}`);
  const label = packId ? `le pack ${packId}` : 'un pack';
  const detail = error?.message ? ` — ${error.message}` : '';
  const message = `Module UI manquant pour ${label} (${slotId})${detail}`;
  try {
    showToast(message, { kind: 'error' });
  } catch (_) {
    /* noop */
  }
}

function registerFallback(slotId, packId, reason) {
  const key = `${slotId}::${packId || 'unknown'}`;
  if (fallbackContributions.has(key)) {
    return;
  }
  fallbackContributions.add(key);
  const label = packId ? `Pack ${packId}` : 'Pack inconnu';
  const contribution = normalizeContribution(slotId, {
    id: `fallback:${key}`,
    order: 1000,
    pack: packId || null,
    label,
    render(el) {
      if (!el) return;
      el.innerHTML = '';
      const wrapper = document.createElement('div');
      wrapper.className = 'pack-slot-error surface-card surface-card--compact surface-card--bordered';
      wrapper.setAttribute('role', 'status');
      const title = document.createElement('strong');
      title.textContent = 'Module indisponible';
      const message = document.createElement('p');
      const base = packId ? `Impossible de charger la contribution UI du pack ${packId}.` : 'Impossible de charger la contribution UI déclarée par un pack.';
      const hint = reason ? ` (${String(reason)})` : '';
      message.textContent = `${base}${hint}`;
      wrapper.appendChild(title);
      wrapper.appendChild(message);
      el.appendChild(wrapper);
    }
  });
  register(contribution.slot, contribution);
}

function createCommandApi() {
  const send = environment.sendCommand;
  if (typeof send !== 'function') {
    return {
      raw: () => Promise.reject(new Error('COMMANDS_UNAVAILABLE')),
      updateNode: () => Promise.reject(new Error('COMMANDS_UNAVAILABLE')),
      addTag: () => Promise.reject(new Error('COMMANDS_UNAVAILABLE')),
      removeTag: () => Promise.reject(new Error('COMMANDS_UNAVAILABLE')),
    };
  }
  // Debounce queue for state/* tags per node
  const stateQueues = new Map(); // nodeId -> { timer, key, resolve, reject, promise }
  const STATE_DEBOUNCE_MS = 200;

  function scheduleStateAdd(nodeId, key, tag) {
    // Reuse existing promise if present; reschedule timer
    const existing = stateQueues.get(nodeId);
    if (existing) {
      if (existing.timer) { try { clearTimeout(existing.timer); } catch (_) {} }
      existing.key = key;
      existing.timer = setTimeout(() => {
        send('AddTagV3', { nodeId, tag: { ...tag } })
          .then(res => existing.resolve(res))
          .catch(err => existing.reject(err))
          .finally(() => stateQueues.delete(nodeId));
      }, STATE_DEBOUNCE_MS);
      return existing.promise;
    }

    let resolve, reject;
    const promise = new Promise((res, rej) => { resolve = res; reject = rej; });
    const timer = setTimeout(() => {
      send('AddTagV3', { nodeId, tag: { ...tag } })
        .then(res => resolve(res))
        .catch(err => reject(err))
        .finally(() => stateQueues.delete(nodeId));
    }, STATE_DEBOUNCE_MS);
    stateQueues.set(nodeId, { timer, key, resolve, reject, promise });
    return promise;
  }

  return {
    raw(type, payload) {
      return send(type, payload);
    },
    updateNode(nodeId, changes) {
      return send('UpdateNode', { nodeId, changes });
    },
    addTag(nodeId, tag) {
      const key = String(tag?.key || tag?.k || '');
      if (key.startsWith('state/')) {
        return scheduleStateAdd(nodeId, key, tag);
      }
      return send('AddTagV3', { nodeId, tag });
    },
    removeTag(nodeId, key, value) {
      return send('RemoveTagV3', { nodeId, key, value: value ?? null });
    },
  };
}

function safeTags(ctx) {
  try {
    const tags = environment.tagsSelector(ctx);
    if (!tags || typeof tags !== 'object') {
      return defaultTags;
    }
    const system = Array.isArray(tags.system) ? tags.system : defaultTags.system;
    const user = Array.isArray(tags.user) ? tags.user : defaultTags.user;
    return { system, user };
  } catch (error) {
    console.error('SLOT_TAGS_RESOLVE_FAILED', error);
    return defaultTags;
  }
}

export function configureSlotRuntime(options = {}) {
  environment = {
    sendCommand: typeof options.sendCommand === 'function' ? options.sendCommand : null,
    tagsSelector: resolveTagSelector(options.tags),
  };
}

function resolveTagSelector(tagsOption) {
  if (typeof tagsOption === 'function') {
    return tagsOption;
  }
  if (tagsOption && typeof tagsOption === 'object') {
    const snapshot = {
      system: Array.isArray(tagsOption.system) ? tagsOption.system : defaultTags.system,
      user: Array.isArray(tagsOption.user) ? tagsOption.user : defaultTags.user,
    };
    return () => snapshot;
  }
  return () => defaultTags;
}
