import { fetchSession } from './session.js';
import { debugLog } from './debug.js';

export class CommandBusClient {
  constructor({ endpoint = '/api/commands', csrfProvider, onCsrfRefresh } = {}) {
    this.endpoint = endpoint;
    this.csrfProvider = typeof csrfProvider === 'function' ? csrfProvider : () => '';
    this.onCsrfRefresh = typeof onCsrfRefresh === 'function' ? onCsrfRefresh : null;
  }

  async execute(command) {
    const token = this.csrfProvider();
    return this.#send(command, token, true);
  }

  async #send(command, token, allowRetry) {
    let response;
    try {
      response = await fetch(this.endpoint, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-Token': token,
        },
        credentials: 'include',
        body: JSON.stringify(command),
      });
    } catch (error) {
      const networkError = new Error('COMMAND_NETWORK_ERROR');
      networkError.cause = error;
      throw networkError;
    }

    const { payload, rawExcerpt } = await this.#parseResponse(response);

    if (response.ok && payload?.ok !== false) {
      return payload;
    }

    const status = response.status;

    if (allowRetry && [401, 403, 419].includes(status)) {
      try {
        const session = await fetchSession();
        const fresh = session?.csrf;
        if (fresh && fresh !== token) {
          debugLog('CSRF retry (CommandBus)', { previous: token ? 'present' : 'empty' });
          this.onCsrfRefresh?.(fresh);
          return this.#send(command, fresh, false);
        }
      } catch (error) {
        debugLog('CSRF retry (CommandBus) failed', error);
      }
    }

    const errPayload = payload && typeof payload === 'object' ? payload : null;
    const errorInfo = errPayload?.error ?? {};
    const errMessage = errorInfo.message || errorInfo.code || 'COMMAND_FAILED';
    const error = new Error(errMessage);
    if (errPayload) {
      error.payload = errPayload;
    }
    if (rawExcerpt) {
      error.rawExcerpt = rawExcerpt;
    }
    error.status = status;
    throw error;
  }

  async #parseResponse(response) {
    const contentType = response.headers.get('content-type') || '';
    const isJson = contentType.toLowerCase().includes('application/json');
    if (!isJson) {
      const raw = (await response.text()).slice(0, 500);
      debugLog('COMMAND_RESPONSE_INVALID', { status: response.status, raw });
      const error = new Error('COMMAND_RESPONSE_INVALID');
      error.status = response.status;
      error.rawExcerpt = raw;
      throw error;
    }

    try {
      const payload = await response.json();
      return { payload, rawExcerpt: null };
    } catch (error) {
      debugLog('COMMAND_RESPONSE_INVALID_PARSE', error);
      const parseError = new Error('COMMAND_RESPONSE_INVALID');
      parseError.status = response.status;
      parseError.cause = error;
      throw parseError;
    }
  }
}
