import { cloneDeep } from './clone.js';

export function applyPatch(board, patch) {
  const state = cloneDeep(board);
  for (const operation of patch.operations) {
    applyOperation(state, operation);
  }
  return state;
}

function applyOperation(state, op) {
  switch (op.op) {
    case 'node.create':
      return nodeCreate(state, op);
    case 'node.update':
      return nodeUpdate(state, op);
    case 'node.move':
      return nodeMove(state, op);
    case 'node.delete':
      return nodeDelete(state, op);
    case 'tag.add':
      return tagAdd(state, op);
    case 'tag.remove':
      return tagRemove(state, op);
    case 'column.rename':
      return columnRename(state, op);
    case 'workspace.rename':
      return workspaceRename(state, op);
    case 'tagFilter.set':
      return tagFilterSet(state, op);
    default:
      return state;
  }
}

function sanitizeProps(input) {
  if (!input || typeof input !== 'object') {
    return {};
  }
  if (Array.isArray(input)) {
    return {};
  }
  return cloneDeep(input);
}

// legacy V2 removed in v3

function tagAdd(state, op) {
  const nodeId = op.nodeId ?? op.targetId;
  if (!nodeId) return;
  const n = state.nodes[nodeId];
  if (!n) return;
  const tags = Array.isArray(n.tags) ? n.tags : [];
  const filtered = tags.filter(t => t.k !== op.tag.k);
  filtered.push(op.tag);
  n.tags = filtered;
}

function tagRemove(state, op) {
  const nodeId = op.nodeId ?? op.targetId;
  if (!nodeId) return;
  const n = state.nodes[nodeId];
  if (!n) return;
  let key = op.key;
  if (!key && op.tag && typeof op.tag === 'object') {
    key = op.tag.k || op.tag.key;
  }
  if (!key) return;
  n.tags = (n.tags || []).filter(t => t.k !== key && t.key !== key);
}

function columnRename(state, op) {
  const columnId = op.columnId;
  if (!columnId) return;
  const column = state.nodes[columnId];
  if (!column) return;
  if (op.title !== undefined && op.title !== null && op.title !== '') {
    column.title = op.title;
  }
  column.updatedAt = Date.now();
}

function workspaceRename(state, op) {
  const workspaceId = op.workspaceId;
  if (!workspaceId) return;
  const workspace = state.nodes[workspaceId];
  if (!workspace) return;
  if (op.title !== undefined && op.title !== null && op.title !== '') {
    workspace.title = op.title;
  }
  workspace.updatedAt = Date.now();
}

function tagFilterSet(state, op) {
  const raw = Array.isArray(op?.selected) ? op.selected : [];
  const seen = new Set();
  const selected = [];
  for (const v of raw) {
    if (typeof v !== 'string') continue;
    const k = v.trim();
    if (!k || seen.has(k)) continue;
    seen.add(k);
    selected.push(k);
  }
  state.tagFilter = {
    selected: selected.includes('__NONE__') ? ['__NONE__'] : selected,
  };
}

// v3: tags par nodeId uniquement

// v3: plus d'ordre list/item

// v3: plus de workspaces/columns

// v3 helpers
function findNode(state, nodeId) { return state.nodes[nodeId]; }

function nodeCreate(state, op) {
  const parent = state.nodes[op.parentId];
  if (!parent) return;
  const id = op.nodeId;
  state.nodes[id] = {
    id,
    parentId: op.parentId,
    children: [],
    order: 0,
    title: op.title ?? 'Nouveau',
    description: op.description ?? null,
    content: op.content ?? null,
    tags: Array.isArray(op.tags) ? op.tags : [],
    sys: typeof op.sys === 'object' && op.sys ? { ...op.sys } : { shape: 'leaf' },
    props: sanitizeProps(op.props),
    updatedAt: Date.now(),
  };
  if (state.nodes[id].sys.shape === 'container') {
    state.nodes[id].children = Array.isArray(op.children) ? [...op.children] : [];
  }
  const arr = parent.children || [];
  const pos = Number.isInteger(op.index) ? Math.max(0, Math.min(op.index, arr.length)) : arr.length;
  arr.splice(pos, 0, id);
  parent.children = arr;
  for (let i=0;i<arr.length;i++){ state.nodes[arr[i]].order = i; }
}

function nodeUpdate(state, op) {
  const n = state.nodes[op.nodeId];
  if (!n) return;
  if (op.title !== undefined) n.title = op.title;
  if (op.description !== undefined) n.description = op.description;
  if (op.content !== undefined) n.content = op.content;
  if (op.collapsed !== undefined) n.collapsed = Boolean(op.collapsed);
  if (op.props !== undefined) n.props = sanitizeProps(op.props);
  if (op.sys && typeof op.sys === 'object') {
    const shape = op.sys.shape ?? n.sys?.shape ?? 'leaf';
    n.sys = { shape };
    if (shape === 'leaf') {
      n.children = [];
    }
  }
  n.updatedAt = Date.now();
}

function nodeMove(state, op) {
  const n = state.nodes[op.nodeId];
  const p = state.nodes[op.toParentId];
  if (!n || !p) return;
  const fromPid = n.parentId;
  if (fromPid && state.nodes[fromPid]) {
    const arr = state.nodes[fromPid].children || [];
    const idx = arr.indexOf(n.id);
    if (idx !== -1) arr.splice(idx,1);
    state.nodes[fromPid].children = arr;
    for (let i=0;i<arr.length;i++){ state.nodes[arr[i]].order = i; }
  }
  n.parentId = p.id;
  const arr = p.children || [];
  const pos = Math.max(0, Math.min(op.toIndex, arr.length));
  arr.splice(pos,0,n.id);
  p.children = arr;
  for (let i=0;i<arr.length;i++){ state.nodes[arr[i]].order = i; }
  n.updatedAt = Date.now();
}

function nodeDelete(state, op) {
  const id = op.nodeId;
  const n = state.nodes[id];
  if (!n) return;
  const pid = n.parentId;
  if (pid && state.nodes[pid]) {
    const arr = state.nodes[pid].children || [];
    const idx = arr.indexOf(id);
    if (idx !== -1) arr.splice(idx,1);
    state.nodes[pid].children = arr;
    for (let i=0;i<arr.length;i++){ state.nodes[arr[i]].order = i; }
  }
  (function removeSubtree(x){
    const node = state.nodes[x];
    if (!node) return;
    for (const cid of (node.children||[])) removeSubtree(cid);
    delete state.nodes[x];
  })(id);
}
