import { requestJson } from '../services/http.js';
import { formatDateTime } from '../utils/format.js';

export function defaultActivityState() {
  return { open: false, loading: false, items: [], lastFetched: 0, unseenCount: 0 };
}

export function createActivityStore(config = {}) {
  const { request = requestJson, getState = null, setState = null, onChange = null, initialState = null } = config;
  const fetchList = (since = 0) => request(`/api/activity?since=${since}`, { method: 'GET' });
  const cmdMarkSeen = (ids) => request('/api/commands', { method: 'POST', body: { type: 'Activity.MarkSeen', payload: { ids } } });
  let local = finalize(initialState);
  const read = () => (typeof getState === 'function' ? finalize(getState()) : local);
  const write = (next) => { const s = finalize(next); if (typeof setState === 'function') setState(s); else local = s; if (typeof onChange === 'function') onChange(s); return s; };
  const update = (fn) => write(fn(finalize(read())) ?? read());

  function toggle(force = null) { const shouldOpen = typeof force === 'boolean' ? force : !read().open; return shouldOpen ? open() : (close(), Promise.resolve(read())); }
  function open() { if (!read().open) update(d => (d.open = true, d)); return load(); }
  function close() { update(d => (d.open = false, d)); }

  async function load() {
    const since = read().lastFetched ?? 0;
    update(d => (d.loading = true, d));
    const payload = await fetchList(since);
    const items = Array.isArray(payload?.items) ? payload.items.map(normalizeItem).filter(Boolean) : [];
    const now = Number(payload?.now ?? Date.now());
    const next = update(d => { d.loading = false; d.items = items; d.lastFetched = Number.isFinite(now) ? now : Date.now(); d.unseenCount = items.filter(i => !i.seen && !i.deleted && !i.archived).length; return d; });
    if (next.open) { const unseen = next.items.filter(i => !i.seen).map(i => i.id); if (unseen.length) markSeen(unseen).catch(console.error); }
    return next;
  }

  async function markSeen(ids) {
    const scoped = uniq(ids);
    if (!scoped.length) return;
    update(d => (d.items = d.items.map(i => scoped.includes(i.id) ? { ...i, seen: true } : i), d.unseenCount = d.items.filter(i => !i.seen && !i.archived && !i.deleted).length, d));
    try { await cmdMarkSeen(scoped); } catch (e) { console.error('ACTIVITY_SEEN_SYNC_FAILED', e); }
  }

  return { getState: read, toggle, open, close, load, markSeen, render: (s, options) => renderOverlay(s ?? read(), options) };
}

export function renderOverlay(state, options = {}) {
  const s = finalize(state);
  if (!s.open) return '';
  const loader = s.loading ? '<div class="notifications-loading">Chargement…</div>' : '';
  const list = s.items.map(renderItem).join('') || '<li class="empty">Aucune activité récente.</li>';
  const panel = `
    <div class="notifications-panel" data-stop-propagation>
      <header class="notifications-header">
        <div class="notifications-header__titles">
          <strong>Activité</strong>
          <span class="notifications-subtitle">${s.unseenCount > 0 ? `${s.unseenCount} non lues` : 'À jour'}</span>
        </div>
        <div class="notifications-actions">
          <button class="btn ghost" data-action="close-activity">Fermer</button>
        </div>
      </header>
      ${loader}
      <ul class="notifications-list">${list}</ul>
    </div>`;
  if (options?.wrap === false) return panel;
  return `<div class="notifications-overlay" data-action="close-activity">${panel}</div>`;
}

function renderItem(entry) {
  const date = formatDateTime(entry.createdAt);
  const dt = date?.label || '';
  const meta = [entry.kind, dt].filter(Boolean).join(' · ');
  const payload = entry.payload && typeof entry.payload === 'object' ? JSON.stringify(entry.payload) : '';
  return `<li class="notification-card${entry.seen ? ' is-seen' : ''}">
    <div class="notification-card__header">
      <div class="notification-card__titles">
        <div class="notification-card__title">${escapeHtml(meta)}</div>
      </div>
    </div>
    ${payload ? `<pre class="notification-card__body" style="white-space:pre-wrap">${escapeHtml(payload)}</pre>` : ''}
  </li>`;
}

function normalizeItem(n) {
  const id = Number(n?.id ?? 0); if (!Number.isFinite(id) || id <= 0) return null;
  return { id, boardId: Number(n.boardId ?? 0) || null, actorId: Number(n.actorId ?? 0) || null, kind: String(n.kind ?? ''), payload: n.payload ?? null, severity: Number(n.severity ?? 0), createdAt: Number(n.createdAt ?? 0), seen: !!n.seen, archived: !!n.archived, deleted: !!n.deleted };
}

function finalize(state) { if (!state) return defaultActivityState(); const s = defaultActivityState(); s.open=!!state.open; s.loading=!!state.loading; s.items=Array.isArray(state.items)?state.items.map(normalizeItem).filter(Boolean):[]; s.lastFetched=Number.isFinite(state.lastFetched)?Number(state.lastFetched):0; s.unseenCount=Number(state.unseenCount??0); return s; }
function uniq(ids){ if(!Array.isArray(ids))return[]; const scoped=ids.map(n=>Number(n)).filter(n=>Number.isFinite(n)&&n>0); return Array.from(new Set(scoped)); }
function escapeHtml(str){ return String(str ?? '').replace(/[&<>"']/g, s => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[s])); }

