// Dialog Orchestrator (headless) — Itération 1
// - Single active modal policy; embed optional (manual activation)
// - Cancellable close via sb-modal-request-close
// - Fallback host injection when mount.host missing

let _instanceSeq = 0;
const MAX_FIELDS = 20;
const MAX_ACTIONS = 6;
const MAX_OPTIONS = 100;
const DEFAULT_TTL_MS = 10 * 60 * 1000;

function uid(prefix = 'dlg_') {
  _instanceSeq += 1;
  return `${prefix}${Date.now().toString(36)}_${_instanceSeq.toString(36)}`;
}

function nowMs() { return Date.now(); }

function ensureGlobalModalHost() {
  let node = document.getElementById('sb-global-modal-host');
  if (!node) {
    node = document.createElement('div');
    node.id = 'sb-global-modal-host';
    node.setAttribute('data-sb-dialog-host', 'global-modal');
    node.style.position = 'fixed';
    node.style.inset = '0';
    // Ensure dialogs (sb-modal) appear above other modals like the file-library (z-index:1000)
    // Set a higher stacking context so nested sb-modal panels are on top
    node.style.zIndex = '3000';
    node.style.pointerEvents = 'none';
    document.body.appendChild(node);
  }
  return node;
}

function resolveHost(mount) {
  const host = mount && typeof mount.host === 'string' ? mount.host : 'global-modal';
  if (host === 'global-modal') return ensureGlobalModalHost();
  if (host.startsWith('container:')) {
    const sel = host.slice('container:'.length);
    const el = document.querySelector(sel);
    if (el instanceof HTMLElement) return el;
    console.warn('[dialog] mount.host introuvable; fallback global-modal', { host });
    return ensureGlobalModalHost();
  }
  return ensureGlobalModalHost();
}

function createContainer(def) {
  const container = document.createElement('div');
  container.className = 'sb-dialog';
  container.setAttribute('data-sb-dialog-id', def.id);
  const title = String(def.title || '');
  const description = def.description ? String(def.description) : '';
  const mode = def.mode || 'confirm';
  const actions = Array.isArray(def.actions) && def.actions.length
    ? def.actions
    : (mode === 'confirm'
        ? [ { id: 'cancel', label: 'Annuler', cancel: true }, { id: 'ok', label: 'Valider', submit: true, primary: true } ]
        : [ { id: 'cancel', label: 'Annuler', cancel: true }, { id: 'ok', label: 'Valider', submit: true, primary: true } ]);

  const bodyHtml = mode === 'form' ? renderForm(def) : renderConfirm(def);
  const footerHtml = `
    <div class="sb-dialog-actions">
      ${actions.map(a => {
        const classes = ['btn'];
        if (a.primary) classes.push('primary');
        if (a.cancel) classes.push('ghost');
        const type = a.submit ? 'submit' : 'button';
        return `<button class="${classes.join(' ')}" data-action-id="${escapeAttr(String(a.id || ''))}" data-action-submit="${a.submit? '1':'0'}" data-action-cancel="${a.cancel? '1':'0'}" type="${type}">${escapeHtml(String(a.label || 'OK'))}</button>`;
      }).join('')}
    </div>`;

  container.innerHTML = `
    <div class="sb-dialog-panel">
      <header class="sb-dialog-header">
        ${title ? `<h2 class="sb-dialog-title">${escapeHtml(title)}</h2>` : ''}
        ${description ? `<p class="sb-dialog-desc">${escapeHtml(description)}</p>` : ''}
      </header>
      <div class="sb-dialog-body">${bodyHtml}</div>
      <footer class="sb-dialog-footer">${footerHtml}</footer>
    </div>`;
  return container;
}

function renderConfirm(def) {
  const message = String(def.message || def.description || '');
  return message ? `<div class="sb-dialog-message">${escapeHtml(message)}</div>` : '';
}

function renderForm(def) {
  const fields = Array.isArray(def.fields) ? def.fields : [];
  const defaults = def.defaults && typeof def.defaults === 'object' ? def.defaults : {};
  const rows = fields.map(f => {
    const id = String(f.id || f.name || '');
    const label = String(f.label || id);
    const type = String(f.type || 'text').toLowerCase();
    const required = !!f.required;
    const init = defaults[id] != null ? String(defaults[id]) : '';
    const common = `name="${escapeAttr(id)}" ${required ? 'required' : ''}`;
    if (type === 'textarea') {
      return `<label class="form-field">${escapeHtml(label)}<textarea ${common}>${escapeHtml(init)}</textarea></label>`;
    }
    if (type === 'select' && Array.isArray(f.options)) {
      const opts = f.options.map(o => {
        const v = String(o.value ?? '');
        const t = String(o.label ?? v);
        const sel = init === v ? ' selected' : '';
        return `<option value="${escapeAttr(v)}"${sel}>${escapeHtml(t)}</option>`;
      }).join('');
      return `<label class="form-field">${escapeHtml(label)}<select ${common}>${opts}</select></label>`;
    }
    // default input
    const inputType = ['text','email','number','password','date'].includes(type) ? type : 'text';
    return `<label class="form-field">${escapeHtml(label)}<input type="${escapeAttr(inputType)}" value="${escapeAttr(init)}" ${common} /></label>`;
  }).join('');
  return `<form class="sb-dialog-form" novalidate>${rows}</form>`;
}

function collectFormValues(container) {
  const form = container.querySelector('.sb-dialog-form');
  if (!form) return {};
  const inputs = form.querySelectorAll('input[name], select[name], textarea[name]');
  const values = {};
  inputs.forEach(el => {
    const name = el.getAttribute('name');
    if (!name) return;
    if (el instanceof HTMLInputElement && (el.type === 'checkbox' || el.type === 'radio')) {
      if (el.checked) values[name] = el.value === '' ? true : el.value;
      else if (!(name in values)) values[name] = false;
    } else {
      values[name] = el.value;
    }
  });
  return values;
}

function escapeHtml(str) {
  return String(str ?? '').replace(/[&<>"']/g, s => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[s]));
}
function escapeAttr(str) { return escapeHtml(str); }

export function createDialogOrchestrator(config = {}) {
  const instances = new Map(); // id -> instance
  let activeModalId = null;

  function spawn(def) {
    const id = String(def.id || uid());
    const layout = (def.layout || 'modal') === 'embed' ? 'embed' : 'modal';
    const ttlMs = Number(def?.policy?.ttlMs ?? DEFAULT_TTL_MS);
    const check = validateAndNormalize({ ...def, id, layout, policy: { ...(def.policy || {}), ttlMs } });
    if (!check.ok) {
      console.warn('[dialog] invalid schema', check);
      // Harden: attempt to proceed with a safe default confirm dialog
      def = safeFallbackDef(check.message || 'SCHEMA_INVALID');
    } else {
      def = check.schema;
    }
    const createdAt = nowMs();
    const mountHost = resolveHost(def.mount || { host: 'global-modal' });

    let dom = null;
    let modal = null;
    let container = null;

    if (layout === 'modal') {
      modal = document.createElement('sb-modal');
      modal.setAttribute('data-sb-dialog', '');
      container = createContainer({ ...def, id });
      modal.appendChild(container);
      mountHost.appendChild(modal);
      // Enable interactions on host while a modal is present
      try { mountHost.style.pointerEvents = 'auto'; } catch (_) {}
      // Apply width based on size
      const autoSize = (() => {
        if (def.size) return String(def.size);
        if (def.mode === 'form') return 'md';
        const msgLen = typeof def.message === 'string' ? def.message.length : 0;
        return msgLen <= 80 ? 'xs' : 'sm';
      })();
      const size = String(autoSize);
      const sizeMap = { xs: 'min(360px, 90vw)', sm: 'min(420px, 92vw)', md: 'min(720px, 95vw)', lg: 'min(960px, 96vw)' };
      const width = sizeMap[size] || sizeMap.md;
      try { modal.style.setProperty('--sb-modal-width', width); } catch (_) {}
      // Close requests
      modal.addEventListener('sb-modal-request-close', async (e) => {
        const inst = instances.get(id);
        if (!inst) return;
        if (typeof inst.beforeClose === 'function') {
          try {
            const allow = await inst.beforeClose('REQUEST');
            if (allow === false) {
              e.preventDefault();
              return;
            }
          } catch (_) {
            // in doubt, block closing
            e.preventDefault();
            return;
          }
        }
      });
      // When closed, cancel if pending
      modal.addEventListener('sb-modal-close', () => {
        const inst = instances.get(id);
        if (!inst) return;
        if (inst._resolver) {
          inst._resolver({ ok: false, code: 'USER_CANCELLED' });
        }
        cleanup(id);
      });
      dom = modal;
    } else {
      // embed inline (inert until activation)
      container = createContainer({ ...def, id });
      container.setAttribute('tabindex', '-1');
      container.setAttribute('aria-hidden', 'true');
      mountHost.appendChild(container);
      dom = container;
    }

    // Wire actions
  if (container) {
      container.addEventListener('click', (e) => {
        const btn = e.target instanceof HTMLElement ? e.target.closest('button[data-action-id]') : null;
        if (!btn) return;
        const isSubmit = btn.getAttribute('data-action-submit') === '1';
        const isCancel = btn.getAttribute('data-action-cancel') === '1';
        if (isCancel) {
          if (layout === 'modal' && dom) dom.removeAttribute('open');
          const inst = instances.get(id);
          if (inst && inst._resolver) inst._resolver({ ok: false, code: 'USER_CANCELLED' });
          cleanup(id);
          return;
        }
        if (isSubmit) {
          const inst = instances.get(id);
          const defForValidation = inst?.def || def;
          const result = validateAndMaybeRender(container, defForValidation);
          if (!result.ok) {
            // Keep dialog open; focus first invalid field
            try { focusFirstInvalid(container); } catch (_) {}
            return;
          }
          if (inst && inst._resolver) inst._resolver({ ok: true, values: result.values });
          if (layout === 'modal' && dom) dom.removeAttribute('open');
          cleanup(id);
        }
      });
      // Anywhere click submit: alerts (single OK) and confirmations (binary)
      const acts = Array.isArray(def.actions) ? def.actions : [];
      const hasSubmit = acts.some(a => !!a.submit);
      const hasCancel = acts.some(a => !!a.cancel);
      const anywhereOk = def.mode === 'confirm' && hasSubmit && !hasCancel; // alerts-like
      const anywhereConfirm = def.mode === 'confirm' && hasSubmit; // confirmations (with or without cancel)
      if (anywhereOk || anywhereConfirm) {
        container.classList.add(anywhereOk ? 'sb-dialog--anywhere-ok' : 'sb-dialog--anywhere-confirm');
        try { if (modal) modal.classList.add(anywhereOk ? 'sb-modal-anywhere-ok' : 'sb-modal-anywhere-confirm'); } catch (_) {}
        const panelEl = container.querySelector('.sb-dialog-panel');
        if (panelEl) {
          panelEl.addEventListener('click', (ev) => {
            // Ignore clicks on explicit controls to avoid double handling
            const target = ev.target instanceof HTMLElement ? ev.target : null;
            if (target && (target.closest('button, a, [data-action-id]'))) return;
            const inst = instances.get(id);
            if (inst && inst._resolver) inst._resolver({ ok: true, values: {} });
            if (layout === 'modal' && dom) dom.removeAttribute('open');
            cleanup(id);
          });
        }
      }
      // Handle Enter key submission on form
      const formEl = container.querySelector('.sb-dialog-form');
      if (formEl instanceof HTMLFormElement) {
        formEl.addEventListener('submit', (ev) => {
          ev.preventDefault();
          const inst = instances.get(id);
          const defForValidation = inst?.def || def;
          const result = validateAndMaybeRender(container, defForValidation);
          if (!result.ok) { try { focusFirstInvalid(container); } catch (_) {} return; }
          if (inst && inst._resolver) inst._resolver({ ok: true, values: result.values });
          if (layout === 'modal' && dom) dom.removeAttribute('open');
          cleanup(id);
        });
      }
    }

    const expireAt = createdAt + Math.max(1000, ttlMs);
    const timer = setTimeout(() => {
      const inst = instances.get(id);
      if (!inst) return;
      if (layout === 'modal' && dom) dom.removeAttribute('open');
      if (inst._resolver) inst._resolver({ ok: false, code: 'INTERACTION_EXPIRED' });
      cleanup(id);
    }, Math.max(1000, ttlMs));

    const instance = {
      id, def: { ...def, id }, layout, dom, createdAt, expireAt, timer,
      activate() {
        if (layout === 'modal' && dom instanceof HTMLElement) {
          // Ensure single active modal
          if (activeModalId && activeModalId !== id) {
            const prev = instances.get(activeModalId);
            if (prev && prev.dom instanceof HTMLElement) prev.dom.removeAttribute('open');
          }
          dom.setAttribute('open', '');
          try { if (typeof dom.show === 'function') dom.show(); } catch (_) {}
          activeModalId = id;
        } else if (layout === 'embed' && dom instanceof HTMLElement) {
          dom.removeAttribute('aria-hidden');
          dom.removeAttribute('tabindex');
        }
      },
      close(reason = 'CLOSE') {
        if (layout === 'modal' && dom instanceof HTMLElement) {
          dom.removeAttribute('open');
        }
        const inst = instances.get(id);
        if (inst && inst._resolver) inst._resolver({ ok: false, code: reason === 'EXPIRED' ? 'INTERACTION_EXPIRED' : 'USER_CANCELLED' });
        cleanup(id);
      },
      beforeClose: null,
      _resolver: null,
    };
    instances.set(id, instance);
    return instance;
  }

  function cleanup(id) {
    const inst = instances.get(id);
    if (!inst) return;
    if (inst.timer) try { clearTimeout(inst.timer); } catch (_) {}
    if (inst.dom && inst.dom.parentNode) {
      const parent = inst.dom.parentNode;
      parent.removeChild(inst.dom);
      // If host empty, disable pointer events so it doesn't block UI
      if (parent instanceof HTMLElement && parent.childElementCount === 0 && parent.id === 'sb-global-modal-host') {
        try { parent.style.pointerEvents = 'none'; } catch (_) {}
      }
    }
    if (activeModalId === id) activeModalId = null;
    instances.delete(id);
  }

  function list() {
    return Array.from(instances.values()).map(i => ({ id: i.id, layout: i.layout, createdAt: i.createdAt, expireAt: i.expireAt }));
  }

  async function openDialog(def) {
    const inst = spawn(def);
    inst.activate();
    return new Promise((resolve) => {
      inst._resolver = resolve;
    });
  }

  function confirmDialog({ title, message, okLabel = 'Valider' } = {}) {
    const def = {
      id: uid('confirm_'),
      schemaVersion: 1,
      layout: 'modal',
      mode: 'confirm',
      title: title || 'Confirmer',
      description: '',
      message: message || '',
      actions: [ { id: 'ok', label: okLabel, submit: true, primary: true } ],
      mount: { host: 'global-modal', activation: 'active' },
      policy: { scope: 'global', concurrency: 'stack', maxOpenPerScope: 5, ttlMs: DEFAULT_TTL_MS },
    };
    return openDialog(def).then(res => !!(res && res.ok));
  }

  function formDialog({ title, fields = [], defaults = {}, okLabel = 'Valider', cancelLabel = 'Annuler' } = {}) {
    const def = {
      id: uid('form_'),
      schemaVersion: 1,
      layout: 'modal',
      mode: 'form',
      title: title || 'Formulaire',
      fields,
      defaults,
      actions: [
        { id: 'cancel', label: cancelLabel, cancel: true },
        { id: 'ok', label: okLabel, submit: true, primary: true }
      ],
      mount: { host: 'global-modal', activation: 'active' },
      policy: { scope: 'global', concurrency: 'stack', maxOpenPerScope: 5, ttlMs: DEFAULT_TTL_MS },
    };
    return openDialog(def).then(res => res && res.ok ? { ok: true, values: res.values || {} } : { ok: false });
  }

  function alertDialog({ title = 'Information', message = '', okLabel = 'OK' } = {}) {
    const def = {
      id: uid('alert_'),
      schemaVersion: 1,
      layout: 'modal',
      mode: 'confirm',
      title,
      description: '',
      message,
      actions: [ { id: 'ok', label: okLabel, submit: true, primary: true } ],
      mount: { host: 'global-modal', activation: 'active' },
      policy: { scope: 'global', concurrency: 'stack', maxOpenPerScope: 5, ttlMs: DEFAULT_TTL_MS },
    };
    return openDialog(def).then(() => true);
  }

  function promptDialog({ title = 'Entrée requise', label = 'Valeur', defaultValue = '' , okLabel = 'Valider', cancelLabel = 'Annuler' } = {}) {
    const def = {
      id: uid('prompt_'),
      schemaVersion: 1,
      layout: 'modal',
      mode: 'form',
      title,
      fields: [ { id: 'value', label, type: 'text', required: false } ],
      defaults: { value: String(defaultValue ?? '') },
      actions: [
        { id: 'cancel', label: cancelLabel, cancel: true },
        { id: 'ok', label: okLabel, submit: true, primary: true }
      ],
      mount: { host: 'global-modal', activation: 'active' },
      policy: { scope: 'global', concurrency: 'stack', maxOpenPerScope: 5, ttlMs: DEFAULT_TTL_MS },
    };
    return openDialog(def).then(res => res && res.ok ? String((res.values || {}).value ?? '') : null);
  }

  return { spawn, list, openDialog, confirmDialog, formDialog, alertDialog, promptDialog };
}

// Default singleton orchestrator
export const dialog = createDialogOrchestrator();
export const openDialog = (...args) => dialog.openDialog(...args);
export const confirmDialog = (...args) => dialog.confirmDialog(...args);
export const formDialog = (...args) => dialog.formDialog(...args);
export const alertDialog = (...args) => dialog.alertDialog(...args);
export const promptDialog = (...args) => dialog.promptDialog(...args);

// ——— Validation & sanitization ———
function safeFallbackDef(message) {
  return {
    id: uid('invalid_'),
    schemaVersion: 1,
    layout: 'modal',
    mode: 'confirm',
    title: 'Dialogue non disponible',
    description: '',
    message: typeof message === 'string' ? message : 'Une erreur est survenue.',
    actions: [ { id: 'ok', label: 'OK', submit: true, primary: true } ],
    mount: { host: 'global-modal', activation: 'active' },
    policy: { scope: 'global', concurrency: 'stack', maxOpenPerScope: 5, ttlMs: DEFAULT_TTL_MS },
  };
}

function validateAndNormalize(schema) {
  try {
    const errors = [];
    const warn = (code, message, details = {}) => errors.push({ code, message, details });
    // version
    const version = Number(schema.schemaVersion ?? 1);
    if (version !== 1) return { ok: false, code: 'SCHEMA_UNSUPPORTED_VERSION', message: 'Version non supportée' };
    // layout/mode
    const layout = String(schema.layout || 'modal');
    if (!['modal', 'embed'].includes(layout)) warn('SCHEMA_INVALID_FIELD', 'layout invalide');
    const mode = String(schema.mode || 'confirm');
    if (!['confirm','form'].includes(mode)) return { ok: false, code: 'SCHEMA_INVALID_FIELD', message: 'mode invalide' };
    // title/desc
    if (schema.title != null && String(schema.title).length > 160) warn('SCHEMA_TOO_LARGE', 'title trop long');
    if (schema.description != null && String(schema.description).length > 400) warn('SCHEMA_TOO_LARGE', 'description trop longue');
    // actions
    const actions = Array.isArray(schema.actions) ? schema.actions : [];
    if (actions.length > MAX_ACTIONS) warn('SCHEMA_TOO_LARGE', 'trop d\'actions');
    const normalizedActions = actions.slice(0, MAX_ACTIONS).map(a => ({
      id: String(a.id || 'action'),
      label: String(a.label || a.id || 'Action'),
      submit: !!a.submit,
      cancel: !!a.cancel,
      primary: !!a.primary,
    }));
    // fields (form only)
    let normalizedFields = undefined;
    let normalizedDefaults = undefined;
    if (mode === 'form') {
      const fields = Array.isArray(schema.fields) ? schema.fields : [];
      if (fields.length > MAX_FIELDS) warn('SCHEMA_TOO_LARGE', 'trop de champs');
      normalizedFields = fields.slice(0, MAX_FIELDS).map(f => normalizeField(f, warn));
      normalizedDefaults = schema.defaults && typeof schema.defaults === 'object' ? schema.defaults : {};
    }
    // mount
    const mount = schema.mount && typeof schema.mount === 'object' ? schema.mount : { host: 'global-modal', activation: 'active' };
    const host = String(mount.host || 'global-modal');
    const activation = String(mount.activation || 'active');
    const normalizedMount = { host, activation };
    // policy
    const pol = schema.policy && typeof schema.policy === 'object' ? schema.policy : {};
    let ttlMs = Number(pol.ttlMs ?? DEFAULT_TTL_MS);
    if (!Number.isFinite(ttlMs) || ttlMs < 1000) ttlMs = DEFAULT_TTL_MS;
    if (ttlMs > 30 * 60 * 1000) { ttlMs = 30 * 60 * 1000; warn('SCHEMA_TOO_LARGE', 'ttlMs réduit'); }
    const normalizedPolicy = {
      scope: String(pol.scope || 'global'),
      concurrency: ['stack','coexist','replace'].includes(String(pol.concurrency)) ? String(pol.concurrency) : 'stack',
      maxOpenPerScope: Number.isInteger(pol.maxOpenPerScope) ? Math.max(1, Math.min(10, pol.maxOpenPerScope)) : 5,
      ttlMs,
    };
    const normalized = {
      id: String(schema.id || uid()),
      schemaVersion: 1,
      layout,
      mode,
      size: ['sm','md','lg'].includes(String(schema.size || 'md')) ? String(schema.size || 'md') : 'md',
      title: schema.title != null ? String(schema.title) : '',
      description: schema.description != null ? String(schema.description) : '',
      message: schema.message != null ? String(schema.message) : undefined,
      ctx: schema.ctx && typeof schema.ctx === 'object' ? schema.ctx : undefined,
      defaults: normalizedDefaults,
      data: schema.data && typeof schema.data === 'object' ? schema.data : undefined,
      fields: normalizedFields,
      actions: normalizedActions,
      mount: normalizedMount,
      policy: normalizedPolicy,
    };
    if (errors.length) {
      try { console.warn('[dialog] schema warnings', errors); } catch (_) {}
    }
    return { ok: true, schema: normalized, warnings: errors };
  } catch (e) {
    return { ok: false, code: 'SCHEMA_INVALID', message: String(e?.message || 'invalid') };
  }
}

function normalizeField(f, warn) {
  const id = String(f?.id || f?.name || '').trim();
  if (!id) { warn('SCHEMA_INVALID_FIELD', 'field.id requis'); }
  const type = String(f?.type || 'text').toLowerCase();
  const allowed = ['text','email','number','password','date','textarea','select'];
  const finalType = allowed.includes(type) ? type : 'text';
  const out = {
    id: id || 'field',
    label: String(f?.label || id || 'Champ'),
    type: finalType,
    required: !!f?.required,
  };
  if (finalType === 'select') {
    const options = Array.isArray(f?.options) ? f.options.slice(0, MAX_OPTIONS) : [];
    out.options = options.map(o => ({ value: String(o?.value ?? ''), label: String(o?.label ?? o?.value ?? '') }));
    if (!Array.isArray(f?.options)) warn('SCHEMA_INVALID_FIELD', 'select.options manquants');
  }
  // Basic validation constraints
  const min = Number.isFinite(f?.min) ? Number(f.min) : undefined;
  const max = Number.isFinite(f?.max) ? Number(f.max) : undefined;
  const minLength = Number.isFinite(f?.minLength) ? Math.max(0, Number(f.minLength)) : undefined;
  const maxLength = Number.isFinite(f?.maxLength) ? Math.max(0, Number(f.maxLength)) : undefined;
  const pattern = typeof f?.pattern === 'string' ? String(f.pattern) : undefined;
  const message = typeof f?.message === 'string' ? String(f.message) : undefined;
  if (typeof min !== 'undefined') out.min = min;
  if (typeof max !== 'undefined') out.max = max;
  if (typeof minLength !== 'undefined') out.minLength = minLength;
  if (typeof maxLength !== 'undefined') out.maxLength = maxLength;
  if (typeof pattern !== 'undefined') out.pattern = pattern;
  if (typeof message !== 'undefined') out.message = message;
  return out;
}

// ——— Client-side validation and rendering ———
function validateAndMaybeRender(container, def) {
  if (!def || def.mode !== 'form') {
    // No validation for non-form
    return { ok: true, values: {} };
  }
  const values = collectFormValues(container);
  const errors = {};
  const fields = Array.isArray(def.fields) ? def.fields : [];
  for (const f of fields) {
    const id = String(f.id || '');
    if (!id) continue;
    const raw = values[id];
    const type = String(f.type || 'text');
    const v = type === 'number' ? (raw === '' ? '' : Number(raw)) : String(raw ?? '');
    let err = null;
    if (f.required && (v === '' || v === false)) {
      err = f.message || 'Champ requis';
    }
    if (!err && type === 'number' && v !== '' && !Number.isFinite(v)) {
      err = f.message || 'Nombre invalide';
    }
    if (!err && typeof f.min === 'number' && type === 'number' && Number.isFinite(v) && v < f.min) {
      err = f.message || `Min ${f.min}`;
    }
    if (!err && typeof f.max === 'number' && type === 'number' && Number.isFinite(v) && v > f.max) {
      err = f.message || `Max ${f.max}`;
    }
    if (!err && typeof f.minLength === 'number' && type !== 'number' && String(v).length < f.minLength) {
      err = f.message || `Min ${f.minLength} caractères`;
    }
    if (!err && typeof f.maxLength === 'number' && type !== 'number' && String(v).length > f.maxLength) {
      err = f.message || `Max ${f.maxLength} caractères`;
    }
    if (!err && typeof f.pattern === 'string' && v !== '') {
      try {
        const re = new RegExp(f.pattern);
        if (!re.test(String(v))) err = f.message || 'Format invalide';
      } catch (_) {
        // ignore invalid pattern
      }
    }
    if (!err && type === 'email' && v !== '') {
      const simpleEmail = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
      if (!simpleEmail.test(String(v))) err = f.message || 'Email invalide';
    }
    if (err) errors[id] = err;
  }
  renderFieldErrors(container, errors);
  const ok = Object.keys(errors).length === 0;
  return { ok, values };
}

function renderFieldErrors(container, errors) {
  const form = container.querySelector('.sb-dialog-form');
  if (!form) return;
  const wrappers = Array.from(form.querySelectorAll('.form-field'));
  // Reset all
  wrappers.forEach(w => {
    w.classList.remove('has-error');
    const input = w.querySelector('input,select,textarea');
    if (input) input.setAttribute('aria-invalid', 'false');
    const err = w.querySelector('.sb-field-error');
    if (err) err.remove();
  });
  // Apply errors
  Object.entries(errors || {}).forEach(([id, message]) => {
    const input = form.querySelector(`[name="${cssEscape(id)}"]`);
    if (!input) return;
    const wrapper = input.closest('.form-field') || input.parentElement;
    if (!wrapper) return;
    wrapper.classList.add('has-error');
    input.setAttribute('aria-invalid', 'true');
    const div = document.createElement('div');
    div.className = 'sb-field-error';
    div.textContent = String(message || 'Champ invalide');
    wrapper.appendChild(div);
  });
}

function focusFirstInvalid(container) {
  const first = container.querySelector('.form-field.has-error input, .form-field.has-error select, .form-field.has-error textarea');
  if (first && first.focus) {
    try { first.focus({ preventScroll: true }); } catch (_) {}
    try { first.scrollIntoView({ block: 'center', behavior: 'smooth' }); } catch (_) {}
  }
}

function cssEscape(value) {
  if (typeof CSS !== 'undefined' && typeof CSS.escape === 'function') return CSS.escape(value);
  return String(value).replace(/[^a-zA-Z0-9_\-]/g, s => `\\${s}`);
}
// Ensure custom element <sb-modal> is defined before usage across browsers
import './modal.js';
