import { capitalize, escapeAttr, escapeHtml } from '../utils/format.js';

const HOME_URL = '/boards.html';

export function renderTopbar(options = {}) {
  const {
    brand = '',
    actions = [],
    className = '',
    attributes = {},
  } = options || {};
  const classes = ['app-topbar'];
  if (className) {
    classes.push(String(className));
  }
  const attrs = Object.entries(attributes || {})
    .filter(([key, value]) => typeof key === 'string' && value !== false && value != null)
    .map(([key, value]) => {
      if (value === true) {
        return `${escapeAttr(key)}`;
      }
      return `${escapeAttr(key)}="${escapeAttr(String(value))}"`;
    });
  const attrString = attrs.length ? ` ${attrs.join(' ')}` : '';
  const actionsHtml = (Array.isArray(actions) ? actions : []).filter(Boolean).join('');
  const actionsBlock = actionsHtml ? `<div class="app-topbar__actions">${actionsHtml}</div>` : '';
  return `
    <header class="${classes.join(' ')}"${attrString}>
      ${brand || ''}
      ${actionsBlock}
    </header>
  `;
}

export function renderActionsGroup(content, options = {}) {
  const { className = '' } = options || {};
  const classes = ['app-topbar__actions-group'];
  if (className) {
    classes.push(String(className));
  }
  return `<div class="${classes.join(' ')}">${content || ''}</div>`;
}

export function renderBrand(options = {}) {
  const {
    subtitle = '',
    title = 'Skyboard',
    icon = '☁️',
    href = HOME_URL,
  } = options;
  const safeHref = escapeAttr(href || HOME_URL);
  const safeTitle = escapeHtml(title || 'Skyboard');
  const subtitleHtml = subtitle ? `<small>${escapeHtml(subtitle)}</small>` : '';
  return `
    <a class="app-topbar__brand" href="${safeHref}" aria-label="Skyboard – retour aux boards">
      <span class="app-topbar__logo">${escapeHtml(icon)}</span>
      <span class="app-topbar__brand-labels">
        <span class="app-topbar__title">${safeTitle}</span>
        ${subtitleHtml}
      </span>
    </a>
  `;
}

export function renderNotificationButton(options = {}) {
  const {
    count = 0,
    open = false,
    action = null,
    href = null,
    title = 'Notifications',
  } = options;
  const badge = count > 0 ? `<span class="app-topbar__badge" aria-hidden="true">${Math.min(count, 99)}</span>` : '';
  const baseAttrs = [
    `class="app-topbar__icon-button focus-ring app-topbar__notifications${open ? ' is-open' : ''}"`,
    `aria-label="${escapeAttr(title)}"`,
    `data-count="${escapeAttr(String(count > 0 ? Math.min(count, 99) : ''))}"`,
    `data-open="${open ? 'true' : 'false'}"`,
  ];
  if (action) {
    baseAttrs.push(`data-action="${escapeAttr(action)}"`);
  }
  const attrString = baseAttrs.join(' ');
  if (href) {
    return `<a ${attrString} href="${escapeAttr(href)}">🔔${badge}</a>`;
  }
  return `<button ${attrString} type="button">🔔${badge}</button>`;
}

export function renderUserMenu(user, options = {}) {
  const identity = resolveUserIdentity(user);
  const {
    items = [],
    footer = [],
  } = options;
  const primaryItems = Array.isArray(items) ? [...items] : [];
  const footerItems = Array.isArray(footer) ? [...footer] : [];
  const role = typeof user?.role === 'string' ? user.role.trim().toLowerCase() : '';
  const userIsAdmin = role === 'admin' || role === 'superadmin';
  if (userIsAdmin) {
    const hasAdminEntry = primaryItems.some(item => {
      if (!item || typeof item !== 'object') return false;
      const action = typeof item.id === 'string' ? item.id : (typeof item.action === 'string' ? item.action : null);
      return action === 'open-admin';
    });
    if (!hasAdminEntry) {
      primaryItems.push({
        id: 'open-admin',
        label: 'Administration',
        icon: '🛠️',
      });
    }
  }

  const trigger = `
    <button class="app-user-chip focus-ring" slot="trigger" type="button" aria-haspopup="true" aria-expanded="false">
      <span class="app-user-chip__avatar" aria-hidden="true">${escapeHtml(identity.avatar)}</span>
      <span class="app-user-chip__labels">
        <span class="app-user-chip__name">${escapeHtml(identity.displayName)}</span>
        <span class="app-user-chip__role">${escapeHtml(identity.roleLabel)}</span>
      </span>
    </button>
  `;

  const header = `
    <div class="app-user-menu__identity">
      <span class="app-user-menu__avatar" aria-hidden="true">${escapeHtml(identity.avatar)}</span>
      <div>
        <div class="app-user-menu__name">${escapeHtml(identity.displayName)}</div>
        <div class="app-user-menu__meta">
          ${identity.email ? `<span>${escapeHtml(identity.email)}</span>` : ''}
          <span>${escapeHtml(identity.roleLabel)}</span>
        </div>
      </div>
    </div>
  `;

  const renderButton = (item) => {
    if (!item || typeof item !== 'object') {
      return '';
    }
    const action = typeof item.id === 'string' ? item.id : (typeof item.action === 'string' ? item.action : null);
    const label = typeof item.label === 'string' ? item.label : '';
    if (!action || !label) {
      return '';
    }
    const kind = item.kind === 'danger' ? ' app-user-menu__item--danger' : '';
    const icon = item.icon ? `<span class="app-user-menu__item-icon">${escapeHtml(item.icon)}</span>` : '';
    return `<button type="button" class="app-user-menu__item${kind}" data-action="${escapeAttr(action)}">${icon}${escapeHtml(label)}</button>`;
  };

  const menuInner = [
    header,
    '<hr />',
    ...primaryItems.map(renderButton),
    footerItems.length ? '<hr />' : '',
    ...footerItems.map(renderButton),
  ].filter(Boolean).join('');

  return `
    <sb-menu class="app-user-menu">
      ${trigger}
      <div class="app-user-menu__panel" slot="menu">
        ${menuInner}
      </div>
    </sb-menu>
  `;
}

export function resolveUserIdentity(user) {
  const pseudo = typeof user?.pseudo === 'string' ? user.pseudo.trim() : '';
  const name = typeof user?.name === 'string' ? user.name.trim() : '';
  const email = typeof user?.email === 'string' ? user.email.trim() : '';
  const displayName = pseudo || name || email || 'Utilisateur';
  const roleRaw = typeof user?.role === 'string' ? user.role.trim() : 'standard';
  const role = roleRaw === '' ? 'standard' : roleRaw;
  const roleLabel = capitalize(role);
  const initialSource = displayName || email || 'U';
  const initial = initialSource.trim().charAt(0).toUpperCase() || 'U';
  const avatar = /^[A-Z0-9]$/i.test(initial) ? initial : '👤';
  return {
    displayName,
    email,
    role,
    roleLabel,
    avatar,
  };
}
