const modalTemplate = document.createElement('template');
modalTemplate.innerHTML = `
  <style>
    :host {
      display: none;
    }
    :host([open]) {
      display: block;
      position: fixed;
      inset: 0;
      z-index: 1000; /* ensure above toasts/overlays */
    }
    .backdrop {
      position: absolute;
      inset: 0;
      background: rgba(15, 23, 42, 0.45);
      backdrop-filter: blur(3px);
    }
    .panel {
      position: relative;
      background: white;
      border-radius: 16px;
      width: var(--sb-modal-width, min(1100px, 95vw));
      margin: 10vh auto;
      padding: 24px;
      box-shadow: 0 25px 50px -12px rgba(15, 23, 42, 0.25);
    }
    /* Hover feedback only when host carries anywhere-* class */
    :host(.sb-modal-anywhere-ok) .panel,
    :host(.sb-modal-anywhere-confirm) .panel {
      transition: background-color .12s ease, box-shadow .12s ease;
      cursor: pointer;
    }
    :host(.sb-modal-anywhere-ok) .panel:hover,
    :host(.sb-modal-anywhere-confirm) .panel:hover {
      background: #f0fdf4; /* white, slightly green tinted */
      box-shadow: 0 25px 60px -15px rgba(16, 185, 129, 0.25), 0 10px 28px -12px rgba(15, 23, 42, 0.25);
    }
  </style>
  <div class="backdrop" part="backdrop"></div>
  <div class="panel" role="dialog" aria-modal="true" part="panel">
    <slot></slot>
  </div>
`;

export class SBModal extends HTMLElement {
  constructor() {
    super();
    this.attachShadow({ mode: 'open' });
    this.shadowRoot.appendChild(modalTemplate.content.cloneNode(true));
    this._handleKey = this._handleKey.bind(this);
    this._handleFocusIn = this._handleFocusIn.bind(this);
    this._focusInitial = this._focusInitial.bind(this);
    this._focusing = false;
    this._previousFocus = null;
    this._panel = this.shadowRoot.querySelector('.panel');
    if (this._panel) {
      this._panel.tabIndex = -1;
    }
    const backdrop = this.shadowRoot.querySelector('.backdrop');
    if (backdrop) {
      backdrop.addEventListener('click', () => this._requestClose('BACKDROP'));
    }
  }

  connectedCallback() {
    if (this.hasAttribute('open')) {
      this._enable();
    }
  }

  static get observedAttributes() {
    return ['open'];
  }

  attributeChangedCallback(name, oldValue, newValue) {
    if (name === 'open') {
      if (this.hasAttribute('open')) {
        this._enable();
      } else {
        this._disable();
      }
    }
  }

  show() {
    this.setAttribute('open', '');
  }

  close() {
    this.removeAttribute('open');
    this.dispatchEvent(new CustomEvent('sb-modal-close', { bubbles: true }));
  }

  _handleKey(event) {
    if (event.key === 'Escape') {
      this._requestClose('ESC');
      return;
    }
    if (event.key === 'Tab') {
      this._cycleFocus(event);
    }
  }

  _requestClose(reason = 'REQUEST') {
    const ev = new CustomEvent('sb-modal-request-close', { bubbles: true, cancelable: true, detail: { reason } });
    this.dispatchEvent(ev);
    if (!ev.defaultPrevented) {
      this.close();
    }
  }

  _enable() {
    this._previousFocus = document.activeElement instanceof HTMLElement ? document.activeElement : null;
    document.addEventListener('keydown', this._handleKey);
    document.addEventListener('focusin', this._handleFocusIn);
    queueMicrotask(this._focusInitial);
  }

  _disable() {
    document.removeEventListener('keydown', this._handleKey);
    document.removeEventListener('focusin', this._handleFocusIn);
    if (this._previousFocus && typeof this._previousFocus.focus === 'function') {
      try { this._previousFocus.focus(); } catch (_) {}
    }
    this._previousFocus = null;
  }

  _handleFocusIn(event) {
    if (!event || !event.target) return;
    const target = event.target;
    // Avoid re-entrancy loops when we programmatically focus
    if (this._focusing) {
      return;
    }
    if (this.contains(target) || this.shadowRoot.contains(target)) {
      return;
    }
    this._focusInitial();
  }

  _focusInitial() {
    const focusable = this._getFocusable();
    try {
      this._focusing = true;
      if (focusable.length) {
        try { focusable[0].focus({ preventScroll: true }); } catch (_) {}
        return;
      }
      if (this._panel) {
        try { this._panel.focus({ preventScroll: true }); } catch (_) {}
      }
    } finally {
      // Release flag in a microtask to swallow the immediate focusin cycle
      queueMicrotask(() => { this._focusing = false; });
    }
  }

  _cycleFocus(event) {
    const focusable = this._getFocusable();
    if (!focusable.length) {
      if (this._panel) {
        event.preventDefault();
        try { this._panel.focus({ preventScroll: true }); } catch (_) {}
      }
      return;
    }
    const first = focusable[0];
    const last = focusable[focusable.length - 1];
    const active = document.activeElement;
    if (event.shiftKey) {
      if (active === first || !this.contains(active)) {
        event.preventDefault();
        try { last.focus({ preventScroll: true }); } catch (_) {}
      }
      return;
    }
    if (active === last) {
      event.preventDefault();
      try { first.focus({ preventScroll: true }); } catch (_) {}
    }
  }

  _getFocusable() {
    const selectors = [
      'a[href]:not([tabindex="-1"])',
      'button:not([disabled]):not([tabindex="-1"])',
      'input:not([disabled]):not([tabindex="-1"])',
      'textarea:not([disabled]):not([tabindex="-1"])',
      'select:not([disabled]):not([tabindex="-1"])',
      '[tabindex]:not([tabindex="-1"])'
    ];
    const nodes = Array.from(this.querySelectorAll(selectors.join(',')));
    return nodes.filter(el => this._isVisible(el));
  }

  _isVisible(el) {
    if (!(el instanceof HTMLElement)) return false;
    return !!(el.offsetWidth || el.offsetHeight || el.getClientRects().length);
  }
}

customElements.define('sb-modal', SBModal);
