import { requestJson } from '../services/http.js';
import { openDialog } from './dialog.js';
import { showToast } from './toast.js';

// Bridge postMessage (frames + nonces + manifests)
const _sbFrames = new Set();
const _sbNonces = new Map();
const _sbManifests = new Map();
let _sbBridgeBooted = false;

function bootPostMessageBridge() {
  if (_sbBridgeBooted) return;
  _sbBridgeBooted = true;
  window.addEventListener('message', (e) => {
    try {
      const m = typeof e.data === 'string' ? JSON.parse(e.data) : e.data;
      if (!m || typeof m !== 'object') return;
      if (m.kind === 'sb:hello') {
        _sbFrames.add(e.source);
        _sbNonces.set(e.source, m.nonce);
        return;
      }
      if (m.kind === 'sb:action') {
        if (!_sbFrames.has(e.source)) return;
        const expect = _sbNonces.get(e.source);
        if (!expect || m.nonce !== expect) return;
        const notificationId = Number(m.notificationId);
        const actionId = String(m.actionId || '');
        if (!Number.isFinite(notificationId) || !actionId) return;
        dispatchAction(notificationId, actionId, e.source).catch(console.error);
      }
    } catch (_) {}
  });
}

async function dispatchAction(notificationId, actionId, sourceWin) {
  let mf = _sbManifests.get(notificationId);
  // Lazy-load manifest if not yet available to avoid ACTION_UNKNOWN on fast clicks
  if (!mf || !Array.isArray(mf.actions)) {
    try {
      const full = await requestJson(`/api/notifications/rich/${notificationId}`, { method: 'GET' });
      const actions = Array.isArray(full?.actions) ? full.actions : (Array.isArray(full?.item?.actions) ? full.item.actions : null);
      if (actions) {
        const manifestVersion = Number(full?.manifestVersion ?? full?.item?.manifestVersion ?? 0) || 0;
        const etag = String(full?.manifestEtag ?? full?.item?.manifestEtag ?? '');
        _sbManifests.set(notificationId, { actions, manifestVersion, etag });
        mf = _sbManifests.get(notificationId);
      }
    } catch (_) {}
  }
  if (!mf || !Array.isArray(mf.actions)) throw new Error('ACTION_UNKNOWN');
  const def = mf.actions.find(a => a && (a.id === actionId || a.ref === actionId));
  if (!def) throw new Error('ACTION_UNKNOWN');
  const headers = { 'X-Action-Correlation-Id': genCorrelationId(notificationId, actionId) };
  try {
    if (def.kind === 'OpenDialog') {
      const dlg = openDialog;
      if (typeof dlg !== 'function') return;
      const values = await dlg(def.schema || def.schemaRef);
      if (!values) return;
      await requestJson('/api/commands', { method: 'POST', headers, body: { type: def.followUpType, payload: { ...(def.params || {}), ...(values || {}) } } });
    } else {
      await requestJson('/api/commands', { method: 'POST', headers, body: { type: def.kind, payload: def.params || {} } });
    }
    try { showToast('Action effectuée', { kind: 'success' }); } catch (_) {}
    if (sourceWin && _sbFrames.has(sourceWin)) {
      const nonce = _sbNonces.get(sourceWin);
      try { sourceWin.postMessage({ kind: 'sb:action:result', nonce, actionId, ok: true }, '*'); } catch (_) {}
    }
  } catch (e) {
    const status = Number(e?.status ?? 0) || 0;
    const code = e?.payload?.code || e?.payload?.error || e?.message || 'REQUEST_FAILED';
    // Idempotence: treat 409 CONFLICT_ALREADY_SET as success
    if (status === 409 || String(code) === 'CONFLICT_ALREADY_SET') {
      try { showToast('Déjà à jour', { kind: 'success' }); } catch (_) {}
      if (sourceWin && _sbFrames.has(sourceWin)) {
        const nonce = _sbNonces.get(sourceWin);
        try { sourceWin.postMessage({ kind: 'sb:action:result', nonce, actionId, ok: true }, '*'); } catch (_) {}
      }
      return; // swallow as success
    }
    // Other errors: notify iframe and show a minimal toast
    if (sourceWin && _sbFrames.has(sourceWin)) {
      const nonce = _sbNonces.get(sourceWin);
      try { sourceWin.postMessage({ kind: 'sb:action:result', nonce, actionId, ok: false, code }, '*'); } catch (_) {}
    }
    try { showToast(mapActionError(code, status), { kind: 'error' }); } catch (_) {}
    throw e;
  }
}

function genCorrelationId(notificationId, actionId) {
  const rnd = Math.random().toString(36).slice(2);
  const ts = Date.now().toString(36);
  return `act_${notificationId}_${actionId}_${ts}_${rnd}`;
}

function mapActionError(code, status) {
  const c = String(code || '').toUpperCase();
  if (c === 'UNAUTHORIZED') return 'Authentification requise.';
  if (c === 'FORBIDDEN' || c === 'ACTION_NOT_ALLOWED') return 'Action interdite.';
  if (c === 'ACTION_DISABLED') return 'Action désactivée.';
  // V4 canonical code is INVALID_PAYLOAD
  if (c === 'INVALID_PAYLOAD') return 'Données invalides.';
  if (c === 'RATE_LIMITED') return 'Trop d’actions, réessayez plus tard.';
  if ((Number(status)||0) >= 500) return 'Erreur serveur.';
  return 'Action impossible.';
}

export function defaultRichState() {
  return {
    open: false,
    loading: false,
    categories: [],
    items: [],
    selectedCategoryId: null,
    selectedId: null,
    lastFetched: 0,
    mode: 'unread',
  };
}

export function createRichNotificationsStore(config = {}) {
  const {
    request = requestJson,
    getState = null,
    setState = null,
    onChange = null,
    initialState = null,
  } = config;

  const fetchList = (mode = 'unread') => request(`/api/notifications/rich?mode=${encodeURIComponent(mode)}`, { method: 'GET' });
  const fetchOne = (id) => request(`/api/notifications/rich/${id}`, { method: 'GET' });
  const cmdMarkSeen = (ids) => request('/api/commands', { method: 'POST', body: { type: 'NotificationRich.MarkSeen', payload: { ids } } });
  const cmdArchive = (ids) => request('/api/commands', { method: 'POST', body: { type: 'Notification.Archive', payload: { ids } } });
  const cmdArchiveByType = (categoryId) => request('/api/commands', { method: 'POST', body: { type: 'Notification.ArchiveByType', payload: { categoryId } } });
  const cmdDeleteForUser = (id) => request('/api/commands', { method: 'POST', body: { type: 'Notification.DeleteForUser', payload: { id } } });

  let localState = finalizeState(initialState);
  try { bootPostMessageBridge(); } catch (_) {}

  function read() {
    if (typeof getState === 'function') return finalizeState(getState());
    return localState;
  }
  function write(next) {
    const normalized = finalizeState(next);
    if (typeof setState === 'function') setState(normalized); else localState = normalized;
    if (typeof onChange === 'function') onChange(normalized);
    return normalized;
  }
  function update(updater) {
    const draft = cloneState(read());
    const next = updater(draft) ?? draft;
    return write(next);
  }

  function toggle(force = null) {
    const shouldOpen = typeof force === 'boolean' ? force : !read().open;
    if (shouldOpen) return open();
    close();
    return Promise.resolve(read());
  }

  function open() {
    if (!read().open) update(d => { d.open = true; return d; });
    return load('unread');
  }
  function close() { update(d => { d.open = false; return d; }); }

  async function load(mode = 'unread') {
    update(d => { d.loading = true; return d; });
    const payload = await fetchList(mode);
    const cats = Array.isArray(payload?.categories) ? payload.categories.map(normalizeCategory).filter(Boolean) : [];
    const items = Array.isArray(payload?.items) ? payload.items.map(normalizeItem).filter(Boolean) : [];
    const now = Number(payload?.now ?? Date.now());
    let selectedId = read().selectedId;
    if (!selectedId && items.length) selectedId = items[0].id;
    let selectedCategoryId = read().selectedCategoryId;
    if (!selectedCategoryId && cats.length) selectedCategoryId = cats[0].id;
    const next = update(d => {
      d.loading = false;
      d.categories = cats;
      d.items = items;
      d.selectedId = selectedId;
      d.selectedCategoryId = selectedCategoryId;
      d.lastFetched = Number.isFinite(now) ? now : Date.now();
      d.mode = mode === 'history' ? 'history' : 'unread';
      return d;
    });
    // Auto mark seen currently selected
    if (next.open && next.selectedId) markSeen([next.selectedId]).catch(console.error);
    return next;
  }

  async function markSeen(ids) {
    const scoped = uniqueIds(ids);
    if (!scoped.length) return;
    update(d => {
      d.items = d.items.map(n => scoped.includes(n.id) ? { ...n, seen: true } : n);
      return d;
    });
    try { await cmdMarkSeen(scoped); } catch (e) { console.error('RICH_SEEN_SYNC_FAILED', e); }
  }
  async function archiveMany(ids) {
    const scoped = uniqueIds(ids);
    if (!scoped.length) return;
    update(d => {
      d.items = d.items.filter(n => !scoped.includes(n.id));
      return d;
    });
    try { await cmdArchive(scoped); await load(read().mode); } catch (e) { console.error('RICH_ARCHIVE_FAILED', e); }
  }
  async function archiveCategory(categoryId) {
    if (!Number.isFinite(categoryId) || categoryId <= 0) return;
    update(d => {
      d.items = d.items.filter(n => n.category?.id !== categoryId);
      return d;
    });
    try { await cmdArchiveByType(categoryId); await load(read().mode); } catch (e) { console.error('RICH_ARCHIVE_BY_TYPE_FAILED', e); }
  }
  async function deleteForUser(id) {
    if (!Number.isFinite(id) || id <= 0) return;
    update(d => {
      d.items = d.items.filter(n => n.id !== id);
      return d;
    });
    try { await cmdDeleteForUser(id); } catch (e) { console.error('RICH_DELETE_FOR_USER_FAILED', e); }
  }

  async function select(id) {
    let item = read().items.find(n => n.id === id);
    if (!item) return;
    update(d => { d.selectedId = id; return d; });
    // Lazy-load full content if missing
    if (!item.content || typeof item.content.html !== 'string') {
      try {
        const full = await fetchOne(id);
        if (full && typeof full === 'object') {
          update(d => {
            d.items = d.items.map(n => n.id === id ? normalizeItem({ ...n, content: full.item?.content ?? full.content ?? null }) : n);
            return d;
          });
          try {
            const actions = Array.isArray(full.actions) ? full.actions : (Array.isArray(full.item?.actions) ? full.item.actions : []);
            const manifestVersion = Number(full.manifestVersion ?? full.item?.manifestVersion ?? 0) || 0;
            const etag = String(full.manifestEtag ?? full.item?.manifestEtag ?? '');
            _sbManifests.set(id, { actions, manifestVersion, etag });
          } catch (e) { console.error('RICH_ACTIONS_MANIFEST_INVALID', e); }
        }
      } catch (e) { console.error('RICH_FETCH_ONE_FAILED', e); }
    }
    if (!item.seen) await markSeen([id]);
  }

  async function selectCategory(categoryId) {
    update(d => {
      d.selectedCategoryId = categoryId;
      const candidates = d.items.filter(n => !categoryId || (n.category && n.category.id === categoryId));
      const stillVisible = candidates.some(n => n.id === d.selectedId);
      if (!stillVisible) {
        d.selectedId = candidates.length ? candidates[0].id : null;
      }
      return d;
    });
    const s = read();
    if (s.open && s.selectedId) markSeen([s.selectedId]).catch(console.error);
  }

  return {
    getState: read,
    toggle,
    open,
    close,
    load,
    select,
    selectCategory,
    markSeen,
    archiveMany,
    archiveCategory,
    deleteForUser,
    render(state, options) { return renderRichModal(state ?? read(), options); },
  };
}

export function renderRichModal(state, options = {}) {
  const s = finalizeState(state);
  if (!s.open) return '';
  const cats = s.categories;
  const items = s.items;
  const selected = items.find(n => n.id === s.selectedId) || null;
  const tabs = cats.map(c => {
    const badge = s.mode === 'history' ? '' : (c.unread>0?` <span class=\\\"badge\\\">${c.unread}</span>`:'');
    const closeBtn = s.mode === 'history' ? '' : `<span class=\"close\" data-action=\"rich-archive-cat\" data-category=\"${c.id}\">×</span>`;
    return `<button class=\"notifrich-tab${s.selectedCategoryId===c.id?' is-active':''}\" data-action=\"rich-sel-cat\" data-category=\"${c.id}\">${escapeHtml(c.name)}${badge}${closeBtn}</button>`;
  }).join('');
  const visible = items.filter(n => !s.selectedCategoryId || n.category?.id === s.selectedCategoryId);
  const list = visible
    .map(n => {
      const actionBtn = s.mode === 'history'
        ? `<button class=\"btn ghost danger\" title=\"Supprimer définitivement\" data-action=\"rich-delete\" data-id=\"${n.id}\">🗑️</button>`
        : `<button class=\"btn ghost\" title=\"Archiver cette notification\" data-action=\"rich-archive\" data-id=\"${n.id}\">🗄️</button><button class=\"btn ghost danger\" title=\"Supprimer définitivement\" data-action=\"rich-delete\" data-id=\"${n.id}\">🗑️</button>`;
      return `<li class=\"notifrich-item${n.seen?' is-seen':''}\" data-action=\"rich-open\" data-id=\"${n.id}\"><div class=\"title\">${escapeHtml(n.title)}</div><div class=\"actions\">${actionBtn}</div></li>`;
    })
    .join('');
  const preview = selected ? renderPreview(selected) : '<div class="notifrich-empty">Aucune notification</div>';
  const loader = s.loading ? '<div class="notifrich-loading">Chargement…</div>' : '';
  const isEmpty = visible.length === 0;
  const emptyTitle = s.mode === 'history' ? 'Aucune notification dans l\u2019historique' : 'Aucune nouvelle notification';
  const emptyHint = s.mode === 'history' ? 'Revenez plus tard pour de nouvelles annonces.' : 'Vous êtes à jour \u2014 revenez plus tard ou consultez l\u2019historique.';
  const emptyCta = s.mode === 'history'
    ? '<button class="btn ghost" data-action="rich-load-unread">Nouvelles</button>'
    : '<button class="btn ghost" data-action="rich-load-history">Voir l\u2019historique</button>';
  const body = isEmpty
    ? `<div class="notifrich-body is-empty">
         <div class="notifrich-empty-state">
           <div class="icon" aria-hidden="true">🔔✨</div>
           <div class="title">${emptyTitle}</div>
           <div class="hint">${emptyHint}</div>
           <div class="actions">${emptyCta}</div>
         </div>
       </div>`
    : `<div class=\"notifrich-body\">\n      <main class=\"notifrich-preview\">${preview}</main>\n      <aside class=\"notifrich-list\"><ul>${list || '<li class=\"empty\">Rien à afficher</li>'}</ul></aside>\n    </div>`;
  const panel = `
  <div class="notifrich-modal" data-stop-propagation>
    <header class="notifrich-header">
      <div class="tabs">${tabs}</div>
      <div class="actions">
        ${s.mode === 'history' ? '<button class="btn ghost" data-action="rich-load-unread">Nouvelles</button>' : '<button class="btn ghost" data-action="rich-load-history">Historique</button>'}
        <button class="btn ghost" data-action="rich-close">Fermer</button>
      </div>
    </header>
    ${loader}
    ${body}
  </div>`;
  if (options?.wrap === false) return panel;
  return `<div class="notifications-overlay" data-action="rich-overlay">${panel}</div>`;
}

function renderPreview(item) {
  const html = String(item.content?.html ?? '');
  const css = String(item.content?.css ?? '');
  // Author JS is not executed per V4
  const origin = (typeof window !== 'undefined' && window.location && window.location.origin) ? window.location.origin : '';
  let csp = `default-src 'none'; img-src ${origin} data:; media-src 'none'; style-src 'unsafe-inline'; script-src 'none'; connect-src 'none';`;
  let srcdoc = `<!doctype html><html><head><meta http-equiv=\"Content-Security-Policy\" content=\"${escapeAttr(csp)}\"><style>${css}</style></head><body>${html}</body></html>`;
  try {
    const nonce = Math.random().toString(36).slice(2) + Date.now().toString(36);
    const notifId = Number(item.id || 0);
    csp = `default-src 'none'; img-src ${origin} data:; media-src 'none'; style-src 'unsafe-inline'; script-src 'nonce-${nonce}'; connect-src 'none';`;
    // Build a small preamble that registers hello and captures clicks
    const pre = `
      const __SB_NONCE__='${nonce}';
      const __SB_NOTIF_ID__=${Number.isFinite(notifId)?notifId:0};
      function __sb_post(m){try{parent.postMessage(m,'*');}catch(e){}}
      function __sb_expand(){ try {var body=document.body; if(!body) return; var html=body.innerHTML; html=html.replace(/\\[sb:action\\s+id=\\\"([^\\\"]+)\\\"\\]([\\s\\S]*?)\\[\\/sb:action\\]/g,function(_,id,txt){ var lab=String(txt).replace(/<[^>]+>/g,' ').replace(/\s+/g,' ').trim(); return '<button data-sb-action-id="'+id+'" aria-label="'+lab+'">'+txt+'</button>';}); body.innerHTML=html;} catch(_){} }
      (function(){
        document.addEventListener('DOMContentLoaded', function(){
          __sb_expand();
          __sb_post({kind:'sb:hello', nonce: __SB_NONCE__, notificationId: __SB_NOTIF_ID__});
          document.body.addEventListener('click', function(e){ var t=e.target && e.target.closest ? e.target.closest('[data-sb-action-id]') : null; if(!t) return; if(t.hasAttribute('disabled')) return; var aid=t.getAttribute('data-sb-action-id')||''; if(!aid) return; try{ t.setAttribute('aria-busy','true'); t.setAttribute('disabled',''); }catch(_){} __sb_post({ kind:'sb:action', nonce: __SB_NONCE__, notificationId: __SB_NOTIF_ID__, actionId: aid }); });
          window.addEventListener('message', function(ev){ var m = (typeof ev.data==='string'? (function(){try{return JSON.parse(ev.data)}catch(_){return null}})(): ev.data); if(!m||m.kind!=='sb:action:result') return; if(m.nonce!==__SB_NONCE__) return; try{ var btn=document.querySelector('[data-sb-action-id=\"'+(m.actionId||'')+'\"]'); if(btn){ btn.removeAttribute('aria-busy'); if(m.ok){ btn.setAttribute('disabled',''); btn.setAttribute('data-sb-action-done','true'); try{ var hasMark=(btn.textContent||'').indexOf('✓')>=0; if(!hasMark){ btn.textContent=(btn.textContent||'')+' ✓'; } }catch(_){} } else { btn.removeAttribute('disabled'); } } }catch(_){} });
        });
      })();
    `;
    srcdoc = `<!doctype html><html><head><meta http-equiv=\"Content-Security-Policy\" content=\"${escapeAttr(csp)}\"><style>${css}</style></head><body>${html}<script nonce=\"${escapeAttr(nonce)}\">${pre}</script></body></html>`;
  } catch (_) {}
  return `<iframe class="notifrich-iframe" sandbox="allow-scripts" referrerpolicy="no-referrer" srcdoc="${escapeAttr(srcdoc)}"></iframe>`;
}

function normalizeCategory(c) {
  const id = Number(c?.id ?? 0);
  if (!Number.isFinite(id) || id <= 0) return null;
  return { id, slug: String(c.slug ?? ''), name: String(c.name ?? ''), unread: Number(c.unread ?? 0) };
}
function normalizeItem(n) {
  const id = Number(n?.id ?? 0);
  if (!Number.isFinite(id) || id <= 0) return null;
  return {
    id,
    category: n.category ? { id: Number(n.category.id ?? 0), slug: String(n.category.slug ?? ''), name: String(n.category.name ?? '') } : null,
    title: String(n.title ?? ''),
    emitter: n.emitter ?? null,
    weight: Number(n.weight ?? 0),
    createdAt: Number(n.createdAt ?? 0),
    seen: !!n.seen,
    archived: !!n.archived,
    deleted: !!n.deleted,
    content: n.content ?? null,
  };
}

function finalizeState(state) {
  if (!state) return defaultRichState();
  const base = defaultRichState();
  base.open = !!state.open;
  base.loading = !!state.loading;
  base.categories = Array.isArray(state.categories) ? state.categories.map(normalizeCategory).filter(Boolean) : [];
  base.items = Array.isArray(state.items) ? state.items.map(normalizeItem).filter(Boolean) : [];
  base.selectedId = Number(state.selectedId ?? 0) || null;
  base.selectedCategoryId = Number(state.selectedCategoryId ?? 0) || null;
  base.lastFetched = Number.isFinite(state.lastFetched) ? Number(state.lastFetched) : 0;
  base.mode = state.mode === 'history' ? 'history' : 'unread';
  return base;
}

function cloneState(state) {
  const s = finalizeState(state);
  return { ...s, categories: s.categories.map(c => ({ ...c })), items: s.items.map(n => ({ ...n })) };
}

function uniqueIds(ids) {
  if (!Array.isArray(ids)) return [];
  const scoped = ids.map(n => Number(n)).filter(n => Number.isFinite(n) && n > 0);
  return Array.from(new Set(scoped));
}

function escapeHtml(str) { return String(str ?? '').replace(/[&<>"']/g, s => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[s])); }
function escapeAttr(str) { return escapeHtml(str); }
