const template = document.createElement('template');
template.innerHTML = `
  <style>
    :host {
      display: block;
    }
    .tablist {
      display: flex;
      gap: 6px;
    }
    button {
      border: none;
      background: transparent;
      font: inherit;
      padding: 6px 10px;
      border-radius: 8px;
      cursor: pointer;
    }
    button[aria-selected="true"] {
      background: rgba(59, 130, 246, 0.12);
      color: #1e3a8a;
      font-weight: 600;
    }
  </style>
  <div class="tablist" role="tablist"></div>
  <slot></slot>
`;

export class SBTabs extends HTMLElement {
  constructor() {
    super();
    this.attachShadow({ mode: 'open' });
    this.shadowRoot.appendChild(template.content.cloneNode(true));
    this._list = this.shadowRoot.querySelector('.tablist');
    this._observer = new MutationObserver(() => this._refresh());
  }

  connectedCallback() {
    this._observer.observe(this, { childList: true });
    this._refresh();
  }

  disconnectedCallback() {
    this._observer.disconnect();
  }

  _refresh() {
    const panels = Array.from(this.querySelectorAll('sb-tab'));
    this._list.innerHTML = '';
    let hasActive = panels.some(panel => panel.hasAttribute('active'));
    panels.forEach((panel, index) => {
      const id = panel.getAttribute('tab-id') || `sb-tab-${index}`;
      panel.id = `${id}-panel`;
      panel.setAttribute('role', 'tabpanel');
      if (!hasActive && index === 0) {
        panel.setAttribute('active', '');
        hasActive = true;
      }
      const btn = document.createElement('button');
      btn.type = 'button';
      btn.textContent = panel.getAttribute('label') || `Tab ${index + 1}`;
      btn.setAttribute('role', 'tab');
      btn.dataset.panel = panel.id;
      btn.addEventListener('click', () => this.activate(panel.id));
      this._list.appendChild(btn);
    });
    const active = panels.find(p => p.hasAttribute('active'));
    if (active) {
      this.activate(active.id);
    }
  }

  activate(panelId) {
    const panels = Array.from(this.querySelectorAll('sb-tab'));
    const buttons = Array.from(this._list.querySelectorAll('button'));
    panels.forEach(panel => {
      const isActive = panel.id === panelId;
      panel.toggleAttribute('active', isActive);
      panel.hidden = !isActive;
    });
    buttons.forEach(btn => {
      const isActive = btn.dataset.panel === panelId;
      btn.setAttribute('aria-selected', isActive ? 'true' : 'false');
      btn.tabIndex = isActive ? 0 : -1;
    });
    this.dispatchEvent(new CustomEvent('sb-tab-change', {
      bubbles: true,
      detail: { panelId },
    }));
  }
}

export class SBTabPanel extends HTMLElement {}

customElements.define('sb-tabs', SBTabs);
customElements.define('sb-tab', SBTabPanel);
