let panelStylesInjected = false;

function injectPanelStyles() {
  if (panelStylesInjected) return;
  panelStylesInjected = true;
  const style = document.createElement('style');
  style.dataset.orgSettingsStyles = '1';
  style.textContent = `
.org-settings-panel__actions { margin-bottom: 16px; }
.org-settings-panel__list { display: flex; flex-direction: column; gap: 16px; }
.org-settings-panel__row { display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 12px; align-items: end; padding: 12px; border: 1px solid rgba(148,163,184,0.25); border-radius: 12px; background: rgba(248,250,252,0.8); position: relative; }
.org-settings-panel__field { display: flex; flex-direction: column; gap: 6px; font-size: 13px; }
.org-settings-panel__field input[type="text"] { padding: 8px 10px; border-radius: 8px; border: 1px solid rgba(148,163,184,0.35); }
.org-settings-panel__field input[type="color"] { width: 48px; height: 32px; border: none; padding: 0; }
.org-settings-panel__actions-row { display: flex; gap: 8px; justify-content: flex-end; }
.org-settings-panel__empty { margin: 0; font-size: 14px; color: rgba(100,116,139,0.9); }
.org-settings-panel__preview { display: flex; align-items: center; gap: 10px; grid-column: 1 / -1; margin-top: 4px; }
.org-settings-panel__preview-chip { width: 34px; height: 34px; border-radius: 12px; display: inline-flex; align-items: center; justify-content: center; font-weight: 700; font-size: 16px; color: #0f172a; background: rgba(148,163,184,0.2); box-shadow: inset 0 0 0 1px rgba(148,163,184,0.25); }
.org-settings-panel__preview-chip[data-kind="image"]::after { content: '\\1F5BC'; }
`;
  document.head.appendChild(style);
}

async function settings(el, { registries }) {
  injectPanelStyles();
  el.innerHTML = `<div class="sb-panel org-settings-panel">
    <h3>Organisation — Catégories</h3>
    <div class="org-settings-panel__actions">
      <button type="button" class="btn" data-add>Ajouter une catégorie</button>
    </div>
    <div class="org-settings-panel__list" data-list></div>
  </div>`;
  const list = el.querySelector('[data-list]');
  const btnAdd = el.querySelector('[data-add]');

  let currentDataset = null;

  async function refresh() {
    list.innerHTML = '';
    const dataset = await registries?.datasets?.get?.('org:categories');
    currentDataset = dataset && typeof dataset === 'object'
      ? dataset
      : { id: 'org:categories', version: 0, items: [] };
    const items = Array.isArray(currentDataset.items) ? currentDataset.items : [];
    if (!items.length) {
      const empty = document.createElement('p');
      empty.className = 'org-settings-panel__empty';
      empty.textContent = 'Aucune catégorie définie. Ajoutez vos thématiques pour afficher des badges.';
      list.appendChild(empty);
      return;
    }

    for (const def of items) {
      const key = String(def?.key ?? '');
      const slug = key.includes('/') ? key.split('/')[1] : key;
      const row = document.createElement('div');
      row.className = 'org-settings-panel__row';
      row.innerHTML = `
        <label class="org-settings-panel__field">
          <span>Slug</span>
          <input data-slug type="text" value="${slug}" />
        </label>
        <label class="org-settings-panel__field">
          <span>Libellé</span>
          <input data-label type="text" value="${def?.label ?? ''}" />
        </label>
        <label class="org-settings-panel__field">
          <span>Icône</span>
          <input data-icon type="text" value="${def?.icon ?? ''}" placeholder="💡 ou /assets/icones/..." />
        </label>
        <label class="org-settings-panel__field">
          <span>Couleur</span>
          <input data-color type="color" value="${(def?.themeColor ?? '#64748b') || '#64748b'}" />
        </label>
        <div class="org-settings-panel__preview" data-preview>
          <span class="org-settings-panel__preview-chip" data-preview-chip></span>
          <span class="org-settings-panel__preview-label" data-preview-label></span>
        </div>
        <div class="org-settings-panel__actions-row">
          <button type="button" class="btn" data-save>Enregistrer</button>
          <button type="button" class="btn ghost" data-del>Supprimer</button>
        </div>
      `;
      list.appendChild(row);

      const slugInput = row.querySelector('[data-slug]');
      const labelInput = row.querySelector('[data-label]');
      const iconInput = row.querySelector('[data-icon]');
      const colorInput = row.querySelector('[data-color]');
      const previewChip = row.querySelector('[data-preview-chip]');
      const previewLabel = row.querySelector('[data-preview-label]');

      const updatePreview = () => {
        if (!previewChip || !previewLabel) return;
        const themeColor = (colorInput?.value || '#64748b').trim() || '#64748b';
        const iconValue = iconInput?.value?.trim?.() ?? '';
        const iconKind = detectIconKind(iconValue);
        const labelValue = labelInput?.value?.trim?.() || (slugInput?.value?.trim?.() || slug || '').replace(/-/g, ' ');
        previewLabel.textContent = labelValue ? labelValue : 'Catégorie';
        previewChip.dataset.kind = iconKind;
        previewChip.style.background = applyAlpha(themeColor, 0.22);
        previewChip.style.boxShadow = `inset 0 0 0 1px ${applyAlpha(themeColor, 0.35)}`;
        previewChip.style.color = getContrastingColor(themeColor);
        if (iconKind === 'image') {
          previewChip.textContent = '';
          previewChip.style.backgroundImage = `url(${iconValue})`;
          previewChip.style.backgroundSize = 'cover';
          previewChip.style.backgroundPosition = 'center';
        } else {
          previewChip.style.backgroundImage = 'none';
          const glyph = iconKind === 'glyph' ? iconValue : (labelValue.charAt(0) || 'C');
          previewChip.textContent = glyph.slice(0, 3);
        }
      };

      updatePreview();

      [slugInput, labelInput, iconInput, colorInput].forEach(input => {
        input?.addEventListener('input', () => updatePreview());
        input?.addEventListener('change', () => updatePreview());
      });

      row.querySelector('[data-save]').onclick = async () => {
        const nextSlug = (slugInput.value || slug || '').trim().toLowerCase().replace(/\s+/g, '-');
        if (!nextSlug) {
          slugInput.value = slug;
          return;
        }
        const nextKey = `category/${nextSlug}`;
        const themeColor = colorInput.value || '#64748b';
        const nextItem = {
          key: nextKey,
          label: labelInput.value.trim() || nextSlug,
          icon: iconInput.value.trim() || '',
          themeColor,
        };
        const baseItems = Array.isArray(currentDataset.items) ? currentDataset.items.slice() : [];
        const filtered = baseItems.filter(item => (item?.key ?? '') !== key);
        filtered.push(nextItem);
        await registries?.datasets?.patch?.('org:categories', {
          id: currentDataset.id ?? 'org:categories',
          version: (currentDataset.version ?? 0) + 1,
          items: filtered,
        });
        try { window.dispatchEvent(new CustomEvent('org:categories:updated')); } catch (_) {}
        await refresh();
      };

      row.querySelector('[data-del]').onclick = async () => {
        const baseItems = Array.isArray(currentDataset.items) ? currentDataset.items.slice() : [];
        const filtered = baseItems.filter(item => (item?.key ?? '') !== key);
        await registries?.datasets?.patch?.('org:categories', {
          id: currentDataset.id ?? 'org:categories',
          version: (currentDataset.version ?? 0) + 1,
          items: filtered,
        });
        try { window.dispatchEvent(new CustomEvent('org:categories:updated')); } catch (_) {}
        await refresh();
      };
    }

    try { window.dispatchEvent(new CustomEvent('board-config-panels:updated')); } catch (_) {}
  }

  btnAdd.onclick = async () => {
    const slug = `categorie-${Math.random().toString(36).slice(2, 7)}`;
    const baseItems = Array.isArray(currentDataset?.items) ? currentDataset.items.slice() : [];
    baseItems.push({
      key: `category/${slug}`,
      label: 'Nouvelle catégorie',
      icon: '',
      themeColor: '#64748b'
    });
    await registries?.datasets?.patch?.('org:categories', {
      id: currentDataset?.id ?? 'org:categories',
      version: (currentDataset?.version ?? 0) + 1,
      items: baseItems,
    });
    try { window.dispatchEvent(new CustomEvent('org:categories:updated')); } catch (_) {}
    await refresh();
  };

  await refresh();
  try { window.dispatchEvent(new CustomEvent('board-config-panels:updated')); } catch (_) {}
}

export default [
  { id: 'org:settings', slot: 'board.configPanels', order: 40, label: 'Catégories', render: settings }
];

function detectIconKind(icon) {
  if (!icon || typeof icon !== 'string') {
    return 'none';
  }
  const trimmed = icon.trim();
  if (!trimmed) return 'none';
  if (/^(https?:)?\//.test(trimmed) || /\.(svg|png|jpg|jpeg|webp)(\?.*)?$/i.test(trimmed)) {
    return 'image';
  }
  if (trimmed.length <= 6) {
    return 'glyph';
  }
  return 'glyph';
}

function applyAlpha(color, alpha) {
  if (!color || typeof color !== 'string') return `rgba(100,116,139,${alpha})`;
  const trimmed = color.trim();
  if (trimmed.startsWith('rgba')) {
    return trimmed;
  }
  if (trimmed.startsWith('rgb')) {
    return trimmed.replace('rgb', 'rgba').replace(')', `, ${alpha})`);
  }
  if (trimmed.startsWith('#')) {
    let hex = trimmed.slice(1);
    if (hex.length === 3) {
      hex = hex.split('').map(ch => ch + ch).join('');
    }
    if (hex.length !== 6) {
      return `rgba(100,116,139,${alpha})`;
    }
    const r = parseInt(hex.slice(0, 2), 16);
    const g = parseInt(hex.slice(2, 4), 16);
    const b = parseInt(hex.slice(4, 6), 16);
    if ([r, g, b].some(n => Number.isNaN(n))) {
      return `rgba(100,116,139,${alpha})`;
    }
    return `rgba(${r}, ${g}, ${b}, ${alpha})`;
  }
  return `rgba(100,116,139,${alpha})`;
}

function getContrastingColor(color) {
  if (!color || typeof color !== 'string') {
    return '#0f172a';
  }
  let hex = color.trim();
  if (hex.startsWith('#')) {
    hex = hex.slice(1);
  }
  if (hex.length === 3) {
    hex = hex.split('').map(ch => ch + ch).join('');
  }
  if (hex.length !== 6) {
    return '#0f172a';
  }
  const r = parseInt(hex.slice(0, 2), 16);
  const g = parseInt(hex.slice(2, 4), 16);
  const b = parseInt(hex.slice(4, 6), 16);
  if ([r, g, b].some(n => Number.isNaN(n))) {
    return '#0f172a';
  }
  const luminance = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
  return luminance > 0.6 ? '#0f172a' : '#f8fafc';
}
