<?php

declare(strict_types=1);

use Skyboard\Application\Commands\CommandBus;
use Skyboard\Application\Commands\CreateColumnCommand;
use Skyboard\Application\Commands\CreateWorkspaceCommand;
use Skyboard\Application\Commands\CreateNodeCommand;
use Skyboard\Application\Commands\UpdateNodeCommand;
use Skyboard\Application\Commands\MoveNodeCommand;
use Skyboard\Application\Commands\DeleteNodeCommand;
use Skyboard\Application\Commands\AddTagV3Command;
use Skyboard\Application\Commands\RemoveTagV3Command;
use Skyboard\Application\Commands\MoveColumnCommand;
use Skyboard\Application\Commands\RenameColumnCommand;
use Skyboard\Application\Commands\RenameWorkspaceCommand;
use Skyboard\Application\Commands\UpdateTagFilterCommand;
use Skyboard\Application\Handlers\AddTagHandler;
use Skyboard\Application\Handlers\CreateColumnHandler;
use Skyboard\Application\Handlers\CreateNodeHandler;
use Skyboard\Application\Handlers\UpdateNodeHandler;
use Skyboard\Application\Handlers\MoveNodeHandler;
use Skyboard\Application\Handlers\DeleteNodeHandler;
use Skyboard\Application\Handlers\CreateWorkspaceHandler;
use Skyboard\Application\Handlers\MoveColumnHandler;
use Skyboard\Application\Handlers\RemoveTagHandler;
use Skyboard\Application\Handlers\RenameColumnHandler;
use Skyboard\Application\Handlers\RenameWorkspaceHandler;
use Skyboard\Application\Handlers\UpdateTagFilterHandler;
use Skyboard\Application\Services\Admin\LicenseAdminService;
use Skyboard\Application\Services\Admin\NotificationCategoryAdminService;
use Skyboard\Application\Services\Admin\NotificationRichAdminService;
use Skyboard\Application\Services\Admin\SecurityAdminService;
use Skyboard\Application\Services\AccountService;
use Skyboard\Application\Services\Admin\UserAdminService;
use Skyboard\Application\Services\AuthService;
use Skyboard\Application\Services\BoardCatalog;
use Skyboard\Application\Services\BoardRulesService;
use Skyboard\Application\Services\EmailVerificationService;
use Skyboard\Infrastructure\Mail\LogMailer;
use Skyboard\Application\Services\LicenseService;
use Skyboard\Application\Services\ActivityReadService;
use Skyboard\Application\Services\NotificationRichReadService;
use Skyboard\Application\Services\ActionResolver;
use Skyboard\Application\Services\TagSuggestionService;
use Skyboard\Application\Services\StructureTagReactor;
use Skyboard\Application\Services\UserFileService;
use Skyboard\Application\Services\UserFileFolderRepository;
use Skyboard\Application\NonBoard\Handlers\BoardSetThumbnailHandler;
use Skyboard\Infrastructure\Http\HttpKernel;
use Skyboard\Infrastructure\Http\Response;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Router;
use Skyboard\Infrastructure\Rules\SystemRules;
use Skyboard\Infrastructure\Packs\SystemTagRegistry;
use Skyboard\Infrastructure\Packs\UiSlotRegistry;
use Skyboard\Infrastructure\Packs\DatasetStore;
use Skyboard\Infrastructure\Packs\RulesCatalog;
use Skyboard\Infrastructure\Packs\PackLoader;
use Skyboard\Infrastructure\Persistence\MySqlBoardRepository;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;
use Skyboard\Infrastructure\Persistence\MySqlRulesProvider;
use Skyboard\Infrastructure\Persistence\MySqlUserRepository;
use Skyboard\Infrastructure\Persistence\MySqlUserFileRepository;
use Skyboard\Infrastructure\Persistence\MySqlUserFileFolderRepository;
use Skyboard\Infrastructure\Persistence\SchemaManager;
use Skyboard\Infrastructure\Logging\CommandAuditLogger;
use Skyboard\Infrastructure\Security\PasswordHasher;
use Skyboard\Infrastructure\Security\RateLimiter;
use Skyboard\Infrastructure\Security\SessionManager;
use Skyboard\Domain\Rules\CompositeRulesProvider;
use Skyboard\Domain\Rules\RuleContextBuilder;
use Skyboard\Domain\Rules\RuleDictionary;
use Skyboard\Domain\Rules\RulesEngine;
use Skyboard\Interfaces\Http\Controllers\Admin\LicenseAdminController;
use Skyboard\Interfaces\Http\Controllers\Admin\NotificationCategoryAdminController;
use Skyboard\Interfaces\Http\Controllers\Admin\NotificationRichAdminController;
use Skyboard\Interfaces\Http\Controllers\Admin\SecurityAdminController;
use Skyboard\Interfaces\Http\Controllers\Admin\ActionCatalogAdminController;
use Skyboard\Interfaces\Http\Controllers\ActivationController;
use Skyboard\Interfaces\Http\Controllers\Admin\UserAdminController;
use Skyboard\Interfaces\Http\Controllers\AuthController;
use Skyboard\Interfaces\Http\Controllers\AccountController;
use Skyboard\Interfaces\Http\Controllers\BoardController;
use Skyboard\Interfaces\Http\Controllers\BoardRulesController;
use Skyboard\Interfaces\Http\Controllers\CommandController;
use Skyboard\Interfaces\Http\Controllers\ActivityController;
use Skyboard\Interfaces\Http\Controllers\NotificationRichController;
use Skyboard\Interfaces\Http\Controllers\DevAdminController;
use Skyboard\Interfaces\Http\Controllers\PacksController;
use Skyboard\Interfaces\Http\Controllers\RulesExplainController;
use Skyboard\Interfaces\Http\Controllers\UserFilesController;
use Skyboard\Interfaces\Http\Middleware\AuthMiddleware;
use Skyboard\Interfaces\Http\Middleware\CsrfMiddleware;
use Skyboard\Application\NonBoard\NonBoardBus;
use Skyboard\Application\NonBoard\Handlers\BoardCreateHandler;
use Skyboard\Application\NonBoard\Handlers\BoardRenameHandler;
use Skyboard\Application\NonBoard\Handlers\BoardDeleteHandler;
use Skyboard\Application\NonBoard\Handlers\AccountLogoutHandler;
use Skyboard\Application\NonBoard\Handlers\AccountUpdateProfileHandler;
use Skyboard\Application\NonBoard\Handlers\AccountRequestVerificationHandler;
use Skyboard\Application\NonBoard\Handlers\AccountVerifyHandler;
use Skyboard\Application\NonBoard\Handlers\AccountRedeemLicenseHandler;
use Skyboard\Application\NonBoard\Handlers\FileComposeHandler;
use Skyboard\Application\NonBoard\Handlers\FileUpdateContentHandler;
use Skyboard\Application\NonBoard\Handlers\FileRenameHandler;
use Skyboard\Application\NonBoard\Handlers\FileDeleteHandler;
use Skyboard\Application\NonBoard\Handlers\FileUploadHandler;
use Skyboard\Application\NonBoard\Handlers\FileCreateFolderHandler;
use Skyboard\Application\NonBoard\Handlers\FileRenameFolderHandler;
use Skyboard\Application\NonBoard\Handlers\FileDeleteFolderHandler;
use Skyboard\Application\NonBoard\Handlers\FileDeleteFolderCascadeHandler;
use Skyboard\Application\NonBoard\Handlers\PackDatasetUpsertHandler;
use Skyboard\Application\NonBoard\Handlers\PackDatasetDeleteHandler;
use Skyboard\Application\NonBoard\Handlers\BoardImportHandler;
use Skyboard\Application\NonBoard\Handlers\BoardAutosaveHandler;
use Skyboard\Application\NonBoard\Handlers\BoardRulesImportHandler;
use Skyboard\Interfaces\Http\Middleware\RateLimitMiddleware;
use Skyboard\Interfaces\Http\Middleware\ErrorHandlingMiddleware;
use Skyboard\Application\NonBoard\Handlers\ActivityMarkSeenHandler;
use Skyboard\Application\NonBoard\Handlers\NotificationArchiveHandler;
use Skyboard\Application\NonBoard\Handlers\NotificationArchiveByTypeHandler;
use Skyboard\Application\NonBoard\Handlers\NotificationDeleteForUserHandler;
use Skyboard\Application\NonBoard\Handlers\NotificationRichMarkSeenHandler;
use Skyboard\Application\NonBoard\Handlers\UserSubscribeCategoryHandler;
use Skyboard\Application\NonBoard\Handlers\UserUnsubscribeCategoryHandler;
use Skyboard\Application\NonBoard\Handlers\UserSetCategoryFrequencyOverrideHandler;

require __DIR__ . '/../bootstrap.php';

$config = \Skyboard\config();
$appEnv = $config['app_env'] ?? 'dev';
ob_start();

$connection = new DatabaseConnection($config['dsn'], $config['db_user'] ?? null, $config['db_password'] ?? null);
$schema = new SchemaManager($connection);
$schema->migrate();

$userRepository = new MySqlUserRepository($connection);
$boardRepository = new MySqlBoardRepository($connection);
$boardCatalog = new BoardCatalog($connection);
$passwordHasher = new PasswordHasher();
$sessionManager = new SessionManager($connection);
$authService = new AuthService($userRepository, $passwordHasher, $sessionManager);

// Chargement du système MCC
$systemTagRegistry = new SystemTagRegistry();
$uiSlotRegistry = new UiSlotRegistry();
$datasetStore = new DatasetStore();
$rulesCatalog = new RulesCatalog();

// Configuration des packs (activation explicite)
$enabledFile = \ROOT_PATH . '/config/packs/enabled.json';
$enabledRaw = @file_get_contents($enabledFile);
$enabledJson = $enabledRaw !== false ? json_decode($enabledRaw, true) : null;
$enabledPacks = is_array($enabledJson) && isset($enabledJson['enabled']) && is_array($enabledJson['enabled'])
    ? $enabledJson['enabled']
    : [];
$packsDir = \ROOT_PATH . '/packs';

// Chargement des packs
$packLoader = new PackLoader($systemTagRegistry, $uiSlotRegistry, $datasetStore, $rulesCatalog);
$packLoader->load($enabledPacks, $packsDir);

// Règles système
$ruleDictionary = new RuleDictionary();
$systemRules = SystemRules::load(__DIR__ . '/../config/rules/core-system.rules.json');
foreach ($systemRules as $definition) {
    $ruleDictionary->registerDefinition($definition);
}

// Composite des règles
$databaseRulesProvider = new MySqlRulesProvider($connection, $ruleDictionary);
$rulesProvider = new CompositeRulesProvider($systemRules, [$databaseRulesProvider], $rulesCatalog, $enabledPacks);
$ruleContextBuilder = new RuleContextBuilder();
$rulesEngine = new RulesEngine($rulesProvider, $ruleContextBuilder, $ruleDictionary);

$structureTagReactor = new StructureTagReactor($config['structure_tag_reactions_enabled'] ?? false);
$commandBus = new CommandBus(
    $boardRepository,
    $rulesEngine,
    $config['rules_engine_enabled'],
    $config['rules_version'],
    $structureTagReactor
);
$commandBus->registerHandler(AddTagV3Command::class, new AddTagHandler());
$commandBus->registerHandler(RemoveTagV3Command::class, new RemoveTagHandler());
$commandBus->registerHandler(CreateNodeCommand::class, new CreateNodeHandler());
$commandBus->registerHandler(UpdateNodeCommand::class, new UpdateNodeHandler());
$commandBus->registerHandler(DeleteNodeCommand::class, new DeleteNodeHandler());
$commandBus->registerHandler(MoveNodeCommand::class, new MoveNodeHandler());
$commandBus->registerHandler(CreateColumnCommand::class, new CreateColumnHandler());
$commandBus->registerHandler(RenameColumnCommand::class, new RenameColumnHandler());
$commandBus->registerHandler(MoveColumnCommand::class, new MoveColumnHandler());
$commandBus->registerHandler(CreateWorkspaceCommand::class, new CreateWorkspaceHandler());
$commandBus->registerHandler(RenameWorkspaceCommand::class, new RenameWorkspaceHandler());
$commandBus->registerHandler(UpdateTagFilterCommand::class, new UpdateTagFilterHandler());

$tagSuggestions = new TagSuggestionService($systemTagRegistry, $rulesEngine);
$packsController = new PacksController($systemTagRegistry, $uiSlotRegistry, $datasetStore, $rulesCatalog);
$activityReadService = new ActivityReadService($connection);
$actionResolver = new ActionResolver($connection);
$notificationRichReadService = new NotificationRichReadService($connection, $actionResolver);
$activityController = new ActivityController($activityReadService);
$notificationRichController = new NotificationRichController($notificationRichReadService);
$userFileRepository = new MySqlUserFileRepository($connection);
$userFileFolderRepository = new MySqlUserFileFolderRepository($connection);
$userFileService = new UserFileService($userFileRepository, null, $userFileFolderRepository);
$userFilesController = new UserFilesController($userFileService);

// Instantiate dependent services BEFORE controllers that use them
$licenseService = new LicenseService($connection, $config['license_secret']);
$mailTransport = \Skyboard\sb_env('MAIL_TRANSPORT', 'log');
$mailer = $mailTransport === 'mail' ? new \Skyboard\Infrastructure\Mail\PhpMailMailer() : new LogMailer();
$emailVerificationService = new EmailVerificationService($connection, $mailer);
$accountService = new AccountService($connection, $licenseService);

$accountController = new AccountController($accountService, $emailVerificationService);
$activationController = new ActivationController($licenseService);

$authController = new AuthController($authService);
$boardRulesService = new BoardRulesService($connection, $boardCatalog);
$boardController = new BoardController($boardCatalog, $boardRepository, $tagSuggestions, $boardRulesService, $config['rules_version']);
$boardRulesController = new BoardRulesController($boardRulesService);
$rulesExplainController = new RulesExplainController($ruleDictionary, $appEnv === 'dev');
$notificationCategoryAdminService = new NotificationCategoryAdminService($connection);
$notificationRichAdminService = new NotificationRichAdminService($connection);
$userAdminService = new UserAdminService($connection, $passwordHasher);
$licenseAdminService = new LicenseAdminService($connection);
$securityAdminService = new SecurityAdminService($connection, $schema);
$notificationCategoryAdminController = new NotificationCategoryAdminController($notificationCategoryAdminService);
$notificationRichAdminController = new NotificationRichAdminController($notificationRichAdminService);
$actionCatalogAdminController = new ActionCatalogAdminController(new \Skyboard\Application\Services\Admin\ActionCatalogAdminService($connection));
$userAdminController = new UserAdminController($userAdminService);
$licenseAdminController = new LicenseAdminController($licenseAdminService);
$securityAdminController = new SecurityAdminController($securityAdminService);
$commandLogger = new CommandAuditLogger();

// Non-board bus (Account, Files, Pack datasets, Board CRUD & tools)
$nonBoardBus = new NonBoardBus();
$nonBoardBus->register('Board.Create', new BoardCreateHandler($boardCatalog));
$nonBoardBus->register('Board.Rename', new BoardRenameHandler($boardCatalog));
$nonBoardBus->register('Board.Delete', new BoardDeleteHandler($boardCatalog));
$nonBoardBus->register('Board.Import', new BoardImportHandler($boardCatalog, $boardRepository, $boardRulesService));
$nonBoardBus->register('Board.Autosave', new BoardAutosaveHandler($boardRepository));
$nonBoardBus->register('Board.SetThumbnail', new BoardSetThumbnailHandler($boardCatalog, $userFileRepository));
$nonBoardBus->register('BoardRules.Import', new BoardRulesImportHandler($boardRulesService));
$nonBoardBus->register('Account.Logout', new AccountLogoutHandler($authService));
$nonBoardBus->register('Account.UpdateProfile', new AccountUpdateProfileHandler($accountService));
$nonBoardBus->register('Account.RequestVerification', new AccountRequestVerificationHandler($emailVerificationService));
$nonBoardBus->register('Account.Verify', new AccountVerifyHandler($emailVerificationService));
$nonBoardBus->register('Account.RedeemLicense', new AccountRedeemLicenseHandler($accountService));
$nonBoardBus->register('File.Compose', new FileComposeHandler($userFileService));
$nonBoardBus->register('File.UpdateContent', new FileUpdateContentHandler($userFileService));
$nonBoardBus->register('File.Rename', new FileRenameHandler($userFileService));
$nonBoardBus->register('File.Delete', new FileDeleteHandler($userFileService));
$nonBoardBus->register('File.Upload', new FileUploadHandler($userFileService));
// Folders / move commands
$nonBoardBus->register('File.CreateFolder', new FileCreateFolderHandler($userFileService));
$nonBoardBus->register('File.RenameFolder', new FileRenameFolderHandler($userFileService));
$nonBoardBus->register('File.DeleteFolder', new FileDeleteFolderHandler($userFileService));
$nonBoardBus->register('File.DeleteFolderCascade', new FileDeleteFolderCascadeHandler($userFileService));
$nonBoardBus->register('File.Move', new \Skyboard\Application\NonBoard\Handlers\FileMoveHandler($userFileService));
$nonBoardBus->register('PackDataset.Upsert', new PackDatasetUpsertHandler($datasetStore));
$nonBoardBus->register('PackDataset.Delete', new PackDatasetDeleteHandler($datasetStore));
// V4 rich notifications & activity state handlers
$nonBoardBus->register('Activity.MarkSeen', new ActivityMarkSeenHandler($connection));
$nonBoardBus->register('Notification.Archive', new NotificationArchiveHandler($connection));
$nonBoardBus->register('Notification.ArchiveByType', new NotificationArchiveByTypeHandler($connection));
$nonBoardBus->register('Notification.DeleteForUser', new NotificationDeleteForUserHandler($connection));
$nonBoardBus->register('NotificationRich.MarkSeen', new NotificationRichMarkSeenHandler($connection));
// User subscriptions (notifications)
$nonBoardBus->register('User.SubscribeCategory', new UserSubscribeCategoryHandler($connection));
$nonBoardBus->register('User.UnsubscribeCategory', new UserUnsubscribeCategoryHandler($connection));
$nonBoardBus->register('User.SetCategoryFrequencyOverride', new UserSetCategoryFrequencyOverrideHandler($connection));
// User data & composite preference actions
$nonBoardBus->register('UserData.SetKey', new \Skyboard\Application\NonBoard\Handlers\UserDataSetKeyHandler($connection));
$nonBoardBus->register('User.SubscribeAndSetPref', new \Skyboard\Application\NonBoard\Handlers\UserSubscribeAndSetPrefHandler($connection));

$commandController = new CommandController(
    $commandBus,
    $commandLogger,
    $nonBoardBus,
);
$devAdminController = $appEnv === 'prod' ? null : new DevAdminController($connection, $appEnv);

$router = new Router();

// Human-friendly entry points (serve static app shells)
// If no specific file requested, redirect to the appropriate HTML shell
$router->add('GET', '/', function (Request $req) {
    $user = $req->getAttribute('user');
    $target = $user ? '/boards.html' : '/auth.html';
    return new Response(302, ['Location' => $target], '');
});
// In case the host is accessed via /public path, normalize to root
$router->add('GET', '/public', fn(Request $req) => new Response(302, ['Location' => '/'], ''));
$router->add('GET', '/public/', fn(Request $req) => new Response(302, ['Location' => '/'], ''));
$router->add('POST', '/api/auth/register', fn(Request $req) => $authController->register($req));
$router->add('POST', '/api/auth/login', fn(Request $req) => $authController->login($req));
$router->add('GET', '/api/session', fn(Request $req) => $authController->session($req));
$router->add('GET', '/api/account', fn(Request $req) => $accountController->get($req));
$router->add('GET', '/api/activation', fn(Request $req) => $activationController->consume($req));

$router->add('GET', '/api/boards', fn(Request $req) => $boardController->list($req));
$router->add('GET', '/api/boards/:boardId/state', fn(Request $req) => $boardController->state($req));
$router->add('GET', '/api/boards/:boardId/allowed-tags', fn(Request $req) => $boardController->allowedTags($req));
$router->add('GET', '/api/boards/:boardId/export', fn(Request $req) => $boardController->export($req));
$router->add('GET', '/api/boards/rules', fn(Request $req) => $boardRulesController->export($req));
$router->add('GET', '/api/rules/explain', fn(Request $req) => $rulesExplainController->explain($req));
$router->add('GET', '/api/rules/dictionary', fn(Request $req) => $rulesExplainController->index($req));


// V4 rich notifications admin (new)
$router->add('GET', '/api/admin/notification-categories', fn(Request $req) => $notificationCategoryAdminController->list($req));
$router->add('POST', '/api/admin/notification-categories', fn(Request $req) => $notificationCategoryAdminController->create($req));
$router->add('POST', '/api/admin/notification-categories/update', fn(Request $req) => $notificationCategoryAdminController->update($req));
$router->add('POST', '/api/admin/notification-categories/delete', fn(Request $req) => $notificationCategoryAdminController->delete($req));

$router->add('GET', '/api/admin/notifications/rich', fn(Request $req) => $notificationRichAdminController->list($req));
$router->add('GET', '/api/admin/notifications/rich/:id', fn(Request $req) => $notificationRichAdminController->get($req));
$router->add('POST', '/api/admin/notifications/rich', fn(Request $req) => $notificationRichAdminController->create($req));
$router->add('POST', '/api/admin/notifications/rich/update', fn(Request $req) => $notificationRichAdminController->update($req));
$router->add('POST', '/api/admin/notifications/rich/delete', fn(Request $req) => $notificationRichAdminController->delete($req));

// Admin — Actions catalogue
$router->add('GET', '/api/admin/actions/catalog', fn(Request $req) => $actionCatalogAdminController->list($req));
$router->add('GET', '/api/admin/actions/catalog/:id', fn(Request $req) => $actionCatalogAdminController->get($req));
$router->add('POST', '/api/admin/actions/catalog', fn(Request $req) => $actionCatalogAdminController->createAction($req));
$router->add('POST', '/api/admin/actions/catalog/version', fn(Request $req) => $actionCatalogAdminController->createVersion($req));
$router->add('POST', '/api/admin/actions/catalog/version/status', fn(Request $req) => $actionCatalogAdminController->updateVersionStatus($req));
// Delete a specific version (guarded: not active, not in use)
$router->add('POST', '/api/admin/actions/catalog/version/delete', fn(Request $req) => $actionCatalogAdminController->deleteVersion($req));
// Delete an action (guarded: not referenced by any notification)
$router->add('POST', '/api/admin/actions/catalog/delete', fn(Request $req) => $actionCatalogAdminController->deleteAction($req));

$router->add('GET', '/api/admin/users', fn(Request $req) => $userAdminController->list($req));
$router->add('GET', '/api/admin/users/:id/data', fn(Request $req) => $userAdminController->data($req));
$router->add('POST', '/api/admin/users', fn(Request $req) => $userAdminController->create($req));
$router->add('POST', '/api/admin/users/update', fn(Request $req) => $userAdminController->update($req));
$router->add('POST', '/api/admin/users/delete', fn(Request $req) => $userAdminController->delete($req));

$router->add('GET', '/api/admin/licenses', fn(Request $req) => $licenseAdminController->list($req));
$router->add('POST', '/api/admin/licenses', fn(Request $req) => $licenseAdminController->create($req));
$router->add('POST', '/api/admin/licenses/assign', fn(Request $req) => $licenseAdminController->assign($req));
$router->add('POST', '/api/admin/licenses/unassign', fn(Request $req) => $licenseAdminController->unassign($req));
$router->add('POST', '/api/admin/licenses/link', fn(Request $req) => $licenseAdminController->link($req));
$router->add('POST', '/api/admin/licenses/delete', fn(Request $req) => $licenseAdminController->delete($req));

$router->add('GET', '/api/admin/security/ip', fn(Request $req) => $securityAdminController->ipList($req));
$router->add('POST', '/api/admin/security/ip/clear', fn(Request $req) => $securityAdminController->clearIp($req));
$router->add('POST', '/api/admin/security/ip/clear-all', fn(Request $req) => $securityAdminController->clearAll($req));
$router->add('POST', '/api/admin/security/reset-schema', fn(Request $req) => $securityAdminController->resetSchema($req));

$router->add('POST', '/api/commands', fn(Request $req) => $commandController->execute($req));
if ($devAdminController) {
    $router->add('POST', '/api/dev/elevate-admin', fn(Request $req) => $devAdminController->elevate($req));
}
$router->add('GET', '/api/packs/enabled', fn(Request $req) => $packsController->enabled());
$router->add('GET', '/api/packs/slots', fn(Request $req) => $packsController->slots());
$router->add('GET', '/api/packs/manifest', fn(Request $req) => $packsController->manifest());
$router->add('GET', '/api/packs/config', fn(Request $req) => $packsController->config());
$router->add('PUT', '/api/packs/config', function(Request $req) use ($packsController) {
    $data = json_decode($req->rawBody ?? 'null', true) ?? [];
    return $packsController->updateConfig($data);
});
$router->add('GET', '/api/packs/datasets/:id', function(Request $req) use ($packsController) {
    $rawId = $req->getAttribute('id');
    $id = is_string($rawId) ? rawurldecode($rawId) : '';
    return $packsController->getDataset($id);
});
// V4: new read-only endpoints for activity & rich notifications
$router->add('GET', '/api/activity', fn(Request $req) => $activityController->list($req));
$router->add('GET', '/api/notifications/rich', fn(Request $req) => $notificationRichController->list($req));
$router->add('GET', '/api/notifications/rich/:id', fn(Request $req) => $notificationRichController->get($req));
$router->add('GET', '/api/notifications/categories', fn(Request $req) => $notificationRichController->categories($req));
$router->add('GET', '/api/files', fn(Request $req) => $userFilesController->index($req));
$router->add('GET', '/api/files/folders', fn(Request $req) => $userFilesController->folders($req));
$router->add('GET', '/api/files/:id/content', fn(Request $req) => $userFilesController->content($req));

$kernel = new HttpKernel($router);
$kernel->addMiddleware(new ErrorHandlingMiddleware($appEnv));
$limiter = new RateLimiter($connection);
$kernel->addMiddleware(new RateLimitMiddleware($limiter));
$kernel->addMiddleware(new AuthMiddleware($sessionManager, $userRepository));
$kernel->addMiddleware(new CsrfMiddleware());

$request = Request::fromGlobals();
$response = $kernel->handle($request);
$response->send();
if (ob_get_level() > 0) {
    // Flush buffered output so the response body is not discarded
    ob_end_flush();
}
exit;
