<?php

declare(strict_types=1);

namespace Skyboard\Application\Commands;

use InvalidArgumentException;
use Skyboard\Application\Handlers\CommandHandler;
use Skyboard\Application\Services\BoardRepository;
use Skyboard\Application\Services\StructureTagReactor;
use Skyboard\Application\Services\StructureTagReactionRefused;
use Skyboard\Domain\Boards\BoardConflictException;
use Skyboard\Domain\Boards\BoardInvariantViolation;
use Skyboard\Domain\Boards\BoardState;
use RuntimeException;
use Skyboard\Domain\Rules\RuleEvaluationTrace;
use Skyboard\Domain\Rules\RulesEngine;

final class CommandBus
{
    /** @var array<string,CommandHandler> */
    private array $handlers = [];

    public function __construct(
        private readonly BoardRepository $boards,
        private readonly RulesEngine $rulesEngine,
        private readonly bool $rulesEngineEnabled,
        private readonly string $rulesVersion,
        private readonly ?StructureTagReactor $tagReactor = null
    ) {
    }

    public function registerHandler(string $commandType, CommandHandler $handler): void
    {
        $this->handlers[$commandType] = $handler;
    }

    /**
     * @return array{
     *     status:string,
     *     trace?:RuleEvaluationTrace|null,
     *     error?:array<string,mixed>,
     *     autosave?:array<string,mixed>,
     *     events?:array{pre:list<array<string,mixed>>,post:list<array<string,mixed>>},
     *     finalPatch?:list<array<string,mixed>>,
     *     patch?:list<array<string,mixed>>,
     *     revision?:int|null,
     *     lastSavedAt?:string|null,
     *     rulesVersion?:string
     * }
     */
    public function dispatch(Command $command, ?int $expectedRevision = null): array
    {
        $type = $command::class;
        $handler = $this->handlers[$type] ?? null;
        if ($handler === null) {
            throw new InvalidArgumentException('No handler for command ' . $type);
        }

        $boardId = $command->boardId();
        $actorId = (int) $command->actorId();
        $snapshot = $boardId ? $this->boards->getSnapshot($boardId, $actorId) : null;
        $board = $snapshot?->state();
        $currentRevision = $snapshot?->revision();
        $currentUpdatedAt = $snapshot?->updatedAt();
        $currentHistory = $snapshot?->history() ?? [];

        if ($boardId !== null && $board === null) {
            return [
                'status' => 'error',
                'error' => ['code' => 'board.not_found'],
                'autosave' => [
                    'status' => 'aborted',
                    'occurredAt' => gmdate('c'),
                    'reasonId' => 'board.not_found',
                    'history' => $currentHistory,
                ],
                'events' => ['pre' => [], 'post' => []],
                'finalPatch' => [],
                'patch' => [],
                'trace' => null,
                'revision' => $currentRevision,
                'lastSavedAt' => $currentUpdatedAt,
                'rulesVersion' => $this->rulesVersion,
            ];
        }

        $startedAt = microtime(true);
        $proposed = $handler->handle($command);
        $basePatch = $proposed->patch();
        $preEvents = $proposed->events();
        $finalPatch = $basePatch;
        $trace = null;

        if ($this->rulesEngineEnabled) {
            $result = $this->rulesEngine->evaluate($board, $command, $proposed);
            if ($result->isRejected()) {
                $occurredAt = gmdate('c');
                return [
                    'status' => 'policy_veto',
                    'error' => [
                        'code' => '409-POLICY',
                        'reasonId' => $result->reasonId(),
                        'message' => $result->message(),
                    ],
                    'trace' => $result->trace(),
                    'autosave' => [
                        'status' => 'blocked',
                        'occurredAt' => $occurredAt,
                        'reasonId' => $result->reasonId(),
                        'history' => $currentHistory,
                    ],
                    'events' => [
                        'pre' => $preEvents,
                        'post' => [[
                            'name' => 'BoardAutosaveBlocked',
                            'boardId' => $boardId,
                            'occurredAt' => $occurredAt,
                            'reasonId' => $result->reasonId(),
                            'history' => $currentHistory,
                        ]],
                    ],
                'finalPatch' => [],
                'revision' => $currentRevision,
                'lastSavedAt' => null,
                'rulesVersion' => $this->rulesVersion,
                'patch' => $basePatch->operations(),
            ];
        }
            $finalPatch = $result->patch();
            $trace = $result->trace();
        }

        if ($this->tagReactor !== null) {
            try {
                $finalPatch = $this->tagReactor->augment($board, $finalPatch);
            } catch (StructureTagReactionRefused $refused) {
                $occurredAt = gmdate('c');
                return [
                    'status' => 'error',
                    'error' => [
                        'code' => '422-STRUCTURE',
                        'message' => $refused->getMessage(),
                        'reasonId' => $refused->reason(),
                        'details' => $refused->details(),
                    ],
                    'trace' => $trace,
                    'autosave' => [
                        'status' => 'aborted',
                        'occurredAt' => $occurredAt,
                        'reasonId' => $refused->reason(),
                        'history' => $currentHistory,
                    ],
                    'events' => [
                        'pre' => $preEvents,
                        'post' => [],
                    ],
                    'finalPatch' => [],
                    'patch' => $basePatch->operations(),
                    'revision' => $currentRevision,
                    'lastSavedAt' => $currentUpdatedAt,
                    'rulesVersion' => $this->rulesVersion,
                ];
            }
        }

        if ($board === null) {
            throw new \RuntimeException('Command requires an existing board state');
        }

        try {
            $writeResult = $this->boards->applyPatch($board, $finalPatch, $actorId, $expectedRevision ?? $currentRevision);
            $board = $writeResult->board();
        } catch (BoardInvariantViolation $violation) {
            $occurredAt = gmdate('c');
            return [
                'status' => 'invariant_violation',
                'error' => [
                    'code' => '409-INVARIANT',
                    'reasonId' => $violation->reason(),
                    'message' => $violation->getMessage(),
                ],
                'trace' => $trace,
                'autosave' => [
                    'status' => 'failed',
                    'occurredAt' => $occurredAt,
                    'reasonId' => $violation->reason(),
                    'history' => $currentHistory,
                ],
                'events' => [
                    'pre' => $preEvents,
                    'post' => [[
                        'name' => 'BoardAutosaveFailed',
                            'boardId' => $finalPatch->boardId(),
                            'occurredAt' => $occurredAt,
                            'reasonId' => $violation->reason(),
                            'history' => $currentHistory,
                        ]],
                ],
                'finalPatch' => $finalPatch->operations(),
                'patch' => $finalPatch->operations(),
                'revision' => $currentRevision,
                'lastSavedAt' => null,
                'rulesVersion' => $this->rulesVersion,
            ];
        } catch (BoardConflictException) {
            $occurredAt = gmdate('c');
            return [
                'status' => 'conflict',
                'error' => [
                    'code' => '409-CONFLICT',
                    'reasonId' => 'board.conflict',
                    'message' => 'Le board a été modifié par ailleurs. Rechargez avant de réessayer.',
                ],
                'trace' => $trace,
                'autosave' => [
                    'status' => 'blocked',
                    'occurredAt' => $occurredAt,
                    'reasonId' => 'board.conflict',
                    'history' => $currentHistory,
                ],
                'events' => [
                    'pre' => $preEvents,
                    'post' => [[
                        'name' => 'BoardAutosaveBlocked',
                            'boardId' => $finalPatch->boardId(),
                            'occurredAt' => $occurredAt,
                            'reasonId' => 'board.conflict',
                            'history' => $currentHistory,
                        ]],
                ],
                'finalPatch' => $finalPatch->operations(),
                'patch' => $finalPatch->operations(),
                'revision' => $currentRevision,
                'lastSavedAt' => $currentUpdatedAt,
                'rulesVersion' => $this->rulesVersion,
            ];
        } catch (RuntimeException $e) {
            $occurredAt = gmdate('c');
            return [
                'status' => 'error',
                'error' => [
                    'code' => 'board.not_found',
                    'message' => $e->getMessage(),
                ],
                'trace' => $trace,
                'autosave' => [
                    'status' => 'aborted',
                    'occurredAt' => $occurredAt,
                    'reasonId' => 'board.not_found',
                    'history' => $currentHistory,
                ],
                'events' => [
                    'pre' => $preEvents,
                    'post' => [[
                        'name' => 'BoardAutosaveBlocked',
                            'boardId' => $finalPatch->boardId(),
                            'occurredAt' => $occurredAt,
                            'reasonId' => 'board.not_found',
                            'history' => $currentHistory,
                        ]],
                ],
                'finalPatch' => $finalPatch->operations(),
                'patch' => $finalPatch->operations(),
                'revision' => $currentRevision,
                'lastSavedAt' => $currentUpdatedAt,
                'rulesVersion' => $this->rulesVersion,
            ];
        }

        $durationMs = (int) round((microtime(true) - $startedAt) * 1000);
        $savedAt = gmdate('c');
        $revision = isset($writeResult) ? $writeResult->revision() : ($currentRevision ?? null);
        $updatedAt = isset($writeResult) ? $writeResult->updatedAt() : null;
        $history = isset($writeResult) ? $writeResult->history() : $currentHistory;
        $postEvents = array_merge(
            $proposed->postCommitEvents(),
            [[
                'name' => 'BoardAutosaveSucceeded',
                'boardId' => $finalPatch->boardId(),
                'savedAt' => $savedAt,
                'durationMs' => $durationMs,
                'revision' => $revision,
                'history' => $history,
            ]]
        );

        return [
            'status' => 'ok',
            'trace' => $trace,
            'autosave' => [
                'status' => 'dirty',
                'savedAt' => $savedAt,
                'durationMs' => $durationMs,
                'revision' => $revision,
                'updatedAt' => $updatedAt,
                'history' => $history,
            ],
            'events' => [
                'pre' => $preEvents,
                'post' => $postEvents,
            ],
            'finalPatch' => $finalPatch->operations(),
            'patch' => $finalPatch->operations(),
            'revision' => $revision,
            'lastSavedAt' => $savedAt,
            'rulesVersion' => $this->rulesVersion,
        ];
    }
}
