<?php

declare(strict_types=1);

namespace Skyboard\Application\NonBoard\Handlers;

use Skyboard\Application\NonBoard\NonBoardHandler;
use Skyboard\Application\Services\BoardRepository;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class BoardAutosaveHandler implements NonBoardHandler
{
    public function __construct(private readonly BoardRepository $boards)
    {
    }

    public function handle(int $userId, array $payload, Request $request): Response
    {
        $boardId = (string) ($payload['boardId'] ?? '');
        if ($boardId === '') {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        $snapshot = $this->boards->getSnapshot($boardId, $userId);
        if ($snapshot === null) {
            return Response::error('BOARD_NOT_FOUND', 'Board introuvable.', [], 404);
        }

        $expected = $payload['revision'] ?? null;
        $expected = is_numeric($expected) ? (int) $expected : null;

        $currentRevision = $snapshot->revision();
        $updatedAt = $snapshot->updatedAt();
        $history = $snapshot->history();
        if ($expected !== null && $expected !== $currentRevision) {
            return Response::error(
                'BOARD_CONFLICT',
                'Le board a été modifié par ailleurs. Rechargez avant de réessayer.',
                [
                    'revision' => $currentRevision,
                    'updatedAt' => $updatedAt,
                    'history' => $history,
                ],
                409
            );
        }

        return Response::ok([
            'status' => 'saved',
            'revision' => $currentRevision,
            'updatedAt' => $updatedAt,
            'confirmedAt' => gmdate('c'),
            'history' => $history,
        ]);
    }
}

