<?php

declare(strict_types=1);

namespace Skyboard\Application\NonBoard\Handlers;

use Skyboard\Application\NonBoard\NonBoardHandler;
use Skyboard\Application\Services\BoardCatalog;
use Skyboard\Application\Services\BoardRepository;
use Skyboard\Application\Services\BoardRulesService;
use Skyboard\Domain\Boards\BoardConflictException;
use Skyboard\Domain\Boards\BoardState;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class BoardImportHandler implements NonBoardHandler
{
    public function __construct(
        private readonly BoardCatalog $catalog,
        private readonly BoardRepository $boards,
        private readonly BoardRulesService $rules
    ) {
    }

    public function handle(int $userId, array $payload, Request $request): Response
    {
        $boardId = (int) ($payload['boardId'] ?? 0);
        $bodyBoard = $payload['board'] ?? null;
        $title = (string) ($payload['title'] ?? '');
        $rules = $payload['rules'] ?? null;

        if (!is_array($bodyBoard) || !isset($bodyBoard['state']) || !is_array($bodyBoard['state'])) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        $state = BoardState::fromArray($bodyBoard['state']);

        if ($boardId <= 0) {
            $titleToUse = $title !== '' ? $title : (string) ($bodyBoard['title'] ?? 'Board');
            $created = $this->catalog->createFromState($userId, $titleToUse, $state);
            $exportedRules = [];
            if (is_array($rules)) {
                // Import rules for the newly created board id
                $this->rules->replace((int) $created['id'], $userId, $rules);
                $exportedRules = $this->rules->export((int) $created['id'], $userId);
            }

            return Response::ok([
                'status' => 'created',
                'board' => [
                    'id' => $created['id'],
                    'title' => $created['title'],
                    'state' => $created['state'],
                    'updatedAt' => $created['updatedAt'] ?? null,
                    'createdAt' => $created['createdAt'] ?? ($created['updatedAt'] ?? null),
                    'revision' => $created['revision'] ?? ($created['updatedAt'] ?? null),
                    'history' => $created['history'] ?? [],
                    'rules' => $exportedRules,
                ],
            ], 201);
        }

        // Replace existing board
        try {
            $writeResult = $this->boards->overwrite(
                (string) $boardId,
                $userId,
                $state,
                isset($payload['revision']) && is_numeric($payload['revision']) ? (int) $payload['revision'] : null,
            );
            if (is_array($rules)) {
                $this->rules->replace($boardId, $userId, $rules);
            }
        } catch (BoardConflictException $e) {
            return Response::error('BOARD_CONFLICT', 'Le board a été modifié par ailleurs. Rechargez avant de réessayer.', [], 409);
        } catch (\RuntimeException $e) {
            return Response::error('BOARD_NOT_FOUND', $e->getMessage(), [], 404);
        } catch (\Throwable $e) {
            return Response::error('IMPORT_FAILED', $e->getMessage(), [], 422);
        }

        $meta = $this->catalog->getMetadata($boardId, $userId) ?? [];
        $resolvedTitle = $title !== '' ? $title : (string) ($meta['title'] ?? 'Board');
        $exportedRules = $this->rules->export($boardId, $userId);

        return Response::ok([
            'status' => 'replaced',
            'board' => [
                'id' => $boardId,
                'title' => $resolvedTitle,
                'state' => $writeResult->board()->toArray(),
                'updatedAt' => $writeResult->updatedAt(),
                'revision' => $writeResult->revision(),
                'history' => $writeResult->history(),
                'rules' => $exportedRules,
            ],
        ]);
    }
}

