<?php

declare(strict_types=1);

namespace Skyboard\Application\NonBoard\Handlers;

use Skyboard\Application\NonBoard\NonBoardHandler;
use Skyboard\Application\Services\BoardCatalog;
use Skyboard\Application\Services\UserFileRepository;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class BoardSetThumbnailHandler implements NonBoardHandler
{
    public function __construct(
        private readonly BoardCatalog $boards,
        private readonly UserFileRepository $files,
    ) {
    }

    public function handle(int $userId, array $payload, Request $request): Response
    {
        $boardId = (int) ($payload['boardId'] ?? 0);
        $publicIdRaw = $payload['publicId'] ?? null;
        $publicId = is_string($publicIdRaw) ? trim($publicIdRaw) : null;

        if ($boardId <= 0) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        // Ensure the board exists and belongs to the user
        if (!$this->boards->exists($boardId, $userId)) {
            return Response::error('BOARD_NOT_FOUND', 'Board introuvable.', [], 404);
        }

        if ($publicId !== null && $publicId !== '') {
            $file = $this->files->findByPublicId($userId, $publicId);
            if ($file === null) {
                return Response::error('FILE_NOT_FOUND', 'Fichier introuvable.', [], 404);
            }
            $mime = $file->mimeType();
            if (!is_string($mime) || !str_starts_with(strtolower($mime), 'image/')) {
                return Response::error('INVALID_FILE_TYPE', 'La vignette doit être une image.', [], 422);
            }
        }

        $this->boards->setThumbnail($boardId, $userId, ($publicId !== '') ? $publicId : null);
        return Response::ok();
    }
}

