<?php

declare(strict_types=1);

namespace Skyboard\Application\NonBoard\Handlers;

use Skyboard\Application\NonBoard\NonBoardHandler;
use Skyboard\Application\Services\UserFileService;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class FileUploadHandler implements NonBoardHandler
{
    public function __construct(private readonly UserFileService $files)
    {
    }

    public function handle(int $userId, array $payload, Request $request): Response
    {
        $specs = $this->resolveUploadSpecs($request);
        if ($specs === []) {
            return Response::error('NO_FILE', 'Aucun fichier fourni.', [], 422);
        }
        $destFolderId = is_string($payload['folderId'] ?? null) && $payload['folderId'] !== '' ? (string) $payload['folderId'] : null;
        $created = [];
        foreach ($specs as $spec) {
            $file = $this->files->uploadFromSpec($userId, $spec);
            if ($destFolderId) {
                try {
                    $moved = $this->files->moveFile($userId, (string) ($file['id'] ?? ''), $destFolderId);
                    $file = $moved;
                } catch (\Throwable) {
                    // Si le déplacement échoue, on renvoie quand même le fichier importé à la racine
                }
            }
            $created[] = $file;
        }
        $out = count($created) === 1 ? ['file' => $created[0]] : ['files' => $created];
        return Response::ok($out, 201);
    }

    /**
     * @return list<array{name:string,type:string,tmp_name:string,error:int,size:int}>
     */
    private function resolveUploadSpecs(Request $request): array
    {
        $fields = ['file', 'files', 'upload', 'uploads'];
        foreach ($fields as $field) {
            $raw = $request->files[$field] ?? null;
            if (!is_array($raw)) {
                continue;
            }
            $normalized = [];
            foreach ($raw as $spec) {
                if (!is_array($spec)) {
                    continue;
                }
                $error = (int) ($spec['error'] ?? 0);
                $skipCode = \defined('UPLOAD_ERR_NO_FILE') ? UPLOAD_ERR_NO_FILE : 4;
                if ($error === $skipCode) {
                    continue;
                }
                $normalized[] = [
                    'name' => (string) ($spec['name'] ?? ''),
                    'type' => (string) ($spec['type'] ?? ''),
                    'tmp_name' => (string) ($spec['tmp_name'] ?? ''),
                    'error' => $error,
                    'size' => (int) ($spec['size'] ?? 0),
                ];
            }
            if ($normalized !== []) {
                return $normalized;
            }
        }
        return [];
    }
}
