<?php

declare(strict_types=1);

namespace Skyboard\Application\NonBoard\Handlers;

use PDO;
use Skyboard\Application\NonBoard\NonBoardHandler;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class NotificationArchiveByTypeHandler implements NonBoardHandler
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    public function handle(int $userId, array $payload, Request $request): Response
    {
        $categoryId = isset($payload['categoryId']) ? (int) $payload['categoryId'] : 0;
        if ($categoryId <= 0) {
            return Response::error('INVALID_PAYLOAD', 'categoryId requis.', [], 422);
        }
        $pdo = $this->connection->pdo();
        $ts = time();

        // Archive all active, non-deleted and non-archived notifications for this category for the user.
        // We insert rows for missing ones, and update archived_at only if NULL.
        $sql = 'INSERT INTO notification_user_state(user_id, notification_id, archived_at)
                SELECT :user, nr.id, :ts
                FROM notifications_rich nr
                LEFT JOIN notification_user_state nus
                  ON nus.user_id = :user AND nus.notification_id = nr.id
                WHERE nr.category_id = :category AND nr.active = 1
                  AND (nus.deleted_at IS NULL)
                  AND (nus.archived_at IS NULL)';
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['user' => $userId, 'ts' => $ts, 'category' => $categoryId]);

        // For existing rows (already present), ensure archived_at is set where NULL
        $upd = $pdo->prepare('UPDATE notification_user_state nus
                              JOIN notifications_rich nr ON nr.id = nus.notification_id
                              SET nus.archived_at = IF(nus.archived_at IS NULL, :ts, nus.archived_at)
                              WHERE nus.user_id = :user AND nr.category_id = :category AND nr.active = 1 AND nus.deleted_at IS NULL');
        $upd->execute(['ts' => $ts, 'user' => $userId, 'category' => $categoryId]);

        return Response::ok();
    }
}

