<?php

declare(strict_types=1);

namespace Skyboard\Application\NonBoard\Handlers;

use Skyboard\Application\NonBoard\NonBoardHandler;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;
use Skyboard\Infrastructure\Packs\DatasetStore;

final class PackDatasetDeleteHandler implements NonBoardHandler
{
    public function __construct(private readonly DatasetStore $datasets)
    {
    }

    public function handle(int $userId, array $payload, Request $request): Response
    {
        $id = (string) ($payload['id'] ?? '');
        $key = (string) ($payload['key'] ?? '');
        if ($id === '' || $key === '') {
            return Response::error('INVALID_PAYLOAD', 'Identifiants requis.', [], 422);
        }
        $current = $this->datasets->load($id) ?? [];
        $updated = is_array($current) ? $current : [];
        $currentEtag = $this->datasets->getEtag($id);
        $ifMatch = isset($payload['ifMatch']) && is_string($payload['ifMatch']) ? $payload['ifMatch'] : null;
        if ($currentEtag !== '') {
            if ($ifMatch === null) {
                return Response::error('PRECONDITION_FAILED', 'ETag requis pour modifier ce dataset.', [], 412, ['ETag' => $currentEtag]);
            }
            if ($ifMatch !== $currentEtag) {
                return Response::error('PRECONDITION_FAILED', 'ETag requis non satisfaisant.', [], 412, ['ETag' => $currentEtag]);
            }
        }
        $modified = false;
        if (is_array($updated) && array_key_exists($key, $updated)) {
            unset($updated[$key]);
            $modified = true;
        } elseif (isset($updated['items']) && is_array($updated['items'])) {
            $items = [];
            foreach ($updated['items'] as $entry) {
                $entryKey = is_array($entry) ? ($entry['key'] ?? null) : null;
                if ($entryKey === $key) {
                    $modified = true;
                    continue;
                }
                $items[] = $entry;
            }
            if ($modified) {
                $updated['items'] = array_values($items);
                if (array_key_exists('version', $updated)) {
                    $updated['version'] = (int) ($updated['version'] ?? 0) + 1;
                }
            }
        }
        if (!$modified) {
            $etag = $this->datasets->getEtag($id);
            $headers = $etag !== '' ? ['ETag' => $etag] : [];
            return Response::ok(['unchanged' => true, 'data' => $current, 'etag' => $etag], 200, $headers);
        }
        $errors = $this->datasets->validateData($id, $updated);
        if (!empty($errors)) {
            return Response::error('INVALID_PAYLOAD', 'Données de dataset invalides.', ['errors' => $errors], 422);
        }
        $ok = $this->datasets->save($id, $updated);
        if (!$ok) return Response::error('DATASET_WRITE_FAILED', 'Impossible de sauvegarder le dataset.', [], 500);
        $etag = $this->datasets->getEtag($id);
        $headers = $etag !== '' ? ['ETag' => $etag] : [];
        return Response::ok(['data' => $this->datasets->load($id) ?? $updated, 'etag' => $etag], 200, $headers);
    }
}

