<?php

declare(strict_types=1);

namespace Skyboard\Application\NonBoard\Handlers;

use Skyboard\Application\NonBoard\NonBoardHandler;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;
use Skyboard\Infrastructure\Packs\DatasetStore;

final class PackDatasetUpsertHandler implements NonBoardHandler
{
    public function __construct(private readonly DatasetStore $datasets)
    {
    }

    public function handle(int $userId, array $payload, Request $request): Response
    {
        $id = (string) ($payload['id'] ?? '');
        if ($id === '') return Response::error('INVALID_PAYLOAD', 'Identifiant dataset requis.', [], 422);
        $data = is_array($payload['data'] ?? null) ? $payload['data'] : [];
        $ifMatch = isset($payload['ifMatch']) && is_string($payload['ifMatch']) ? $payload['ifMatch'] : null;
        $errors = $this->datasets->validateData($id, $data);
        if (!empty($errors)) {
            return Response::error('INVALID_PAYLOAD', 'Données de dataset invalides.', ['errors' => $errors], 422);
        }
        $currentEtag = $this->datasets->getEtag($id);
        if ($currentEtag !== '') {
            if ($ifMatch !== null && $ifMatch !== $currentEtag) {
                return Response::error('PRECONDITION_FAILED', 'ETag requis non satisfaisant.', [], 412, ['ETag' => $currentEtag]);
            }
        }
        $ok = $this->datasets->save($id, $data);
        if (!$ok) return Response::error('DATASET_WRITE_FAILED', 'Impossible de sauvegarder le dataset.', [], 500);
        $etag = $this->datasets->getEtag($id);
        $headers = $etag !== '' ? ['ETag' => $etag] : [];
        return Response::ok(['data' => $this->datasets->load($id) ?? $data, 'etag' => $etag], 200, $headers);
    }
}

