<?php

declare(strict_types=1);

namespace Skyboard\Application\NonBoard\Handlers;

use DateTimeImmutable;
use DateTimeZone;
use PDO;
use Skyboard\Application\NonBoard\NonBoardHandler;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class UserSubscribeAndSetPrefHandler implements NonBoardHandler
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    public function handle(int $userId, array $payload, Request $request): Response
    {
        $categoryId = isset($payload['categoryId']) ? (int) $payload['categoryId'] : 0;
        $namespace = trim((string) ($payload['namespace'] ?? 'prefs'));
        $key = trim((string) ($payload['key'] ?? ''));
        $value = $payload['value'] ?? true;

        if ($namespace === '' || $key === '') {
            return Response::error('INVALID_PAYLOAD', 'namespace et key requis.', [], 422);
        }

        $pdo = $this->connection->pdo();
        $pdo->beginTransaction();
        try {
            // Resolve category strictly by id (IDs only, no slug allowed)
            if ($categoryId <= 0) {
                $pdo->rollBack();
                return Response::error('INVALID_PAYLOAD', 'categoryId requis.', [], 422);
            }
            $stmt = $pdo->prepare('SELECT dispatch_mode, active FROM notification_categories WHERE id = :id');
            $stmt->execute(['id' => $categoryId]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$row || empty($row['active'])) {
                $pdo->rollBack();
                return Response::error('NOT_FOUND', 'Catégorie introuvable ou inactive.', [], 404);
            }
            $dispatchMode = strtoupper((string) ($row['dispatch_mode'] ?? 'BROADCAST'));

            // Subscribe (idempotent upsert)
            $now = time();
            $stmt = $pdo->prepare('INSERT INTO user_subscriptions(user_id, category_id, subscribed, created_at)
                                   VALUES(:user, :cat, 1, :now)
                                   ON DUPLICATE KEY UPDATE subscribed = VALUES(subscribed)');
            $stmt->execute(['user' => $userId, 'cat' => $categoryId, 'now' => $now]);

            // If PERSONALIZED, ensure cycle_anchor_ts set
            if ($dispatchMode === 'PERSONALIZED') {
                $q = $pdo->prepare('SELECT cycle_anchor_ts FROM user_subscriptions WHERE user_id = :user AND category_id = :cat');
                $q->execute(['user' => $userId, 'cat' => $categoryId]);
                $cur = $q->fetch(PDO::FETCH_ASSOC) ?: [];
                $anchor = isset($cur['cycle_anchor_ts']) ? (int) $cur['cycle_anchor_ts'] : 0;
                if ($anchor <= 0) {
                    $utcMidnight = self::utcMidnightEuropeParis($now);
                    $upd = $pdo->prepare('UPDATE user_subscriptions SET cycle_anchor_ts = :anchor WHERE user_id = :user AND category_id = :cat');
                    $upd->execute(['anchor' => $utcMidnight, 'user' => $userId, 'cat' => $categoryId]);
                }
            }

            // Set user preference (idempotent upsert)
            $stmt = $pdo->prepare(
                'INSERT INTO user_data(user_id, namespace, `key`, value_json, updated_at)
                 VALUES(:u, :ns, :k, :v, :ts)
                 ON DUPLICATE KEY UPDATE value_json = VALUES(value_json), updated_at = VALUES(updated_at)'
            );
            $stmt->execute([
                'u' => $userId,
                'ns' => $namespace,
                'k' => $key,
                'v' => json_encode($value, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
                'ts' => $now,
            ]);

            $pdo->commit();
            return Response::ok();
        } catch (\Throwable $e) {
            try { if ($pdo->inTransaction()) $pdo->rollBack(); } catch (\Throwable) {}
            return Response::error('INTERNAL_SERVER_ERROR', 'Erreur serveur.', ['reason' => 'TX_FAILED'], 500);
        }
    }

    private static function utcMidnightEuropeParis(int $nowUtc): int
    {
        $tz = new DateTimeZone('Europe/Paris');
        $utc = new DateTimeZone('UTC');
        $local = (new DateTimeImmutable('@' . $nowUtc))->setTimezone($tz)->setTime(0, 0, 0);
        return $local->setTimezone($utc)->getTimestamp();
    }
}
