<?php

declare(strict_types=1);

namespace Skyboard\Application\NonBoard\Handlers;

use DateTimeImmutable;
use DateTimeZone;
use Skyboard\Application\NonBoard\NonBoardHandler;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class UserSubscribeCategoryHandler implements NonBoardHandler
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    public function handle(int $userId, array $payload, Request $request): Response
    {
        $categoryId = isset($payload['categoryId']) ? (int) $payload['categoryId'] : 0;
        if ($categoryId <= 0) {
            return Response::error('INVALID_PAYLOAD', 'categoryId requis.', [], 422);
        }

        $pdo = $this->connection->pdo();
        $cat = $pdo->prepare("SELECT id, dispatch_mode FROM notification_categories WHERE id = :id AND active = 1");
        $cat->execute(['id' => $categoryId]);
        $row = $cat->fetch(\PDO::FETCH_ASSOC);
        if (!$row) {
            return Response::error('NOT_FOUND', 'Catégorie introuvable ou inactive.', [], 404);
        }

        $now = time();
        $stmt = $pdo->prepare('INSERT INTO user_subscriptions(user_id, category_id, subscribed, created_at)
                               VALUES(:user, :cat, 1, :now)
                               ON DUPLICATE KEY UPDATE subscribed = VALUES(subscribed)');
        $stmt->execute(['user' => $userId, 'cat' => $categoryId, 'now' => $now]);

        // If PERSONALIZED and cycle_anchor_ts is null, set to midnight Europe/Paris in UTC
        if (strtoupper((string) $row['dispatch_mode']) === 'PERSONALIZED') {
            $q = $pdo->prepare('SELECT cycle_anchor_ts FROM user_subscriptions WHERE user_id = :user AND category_id = :cat');
            $q->execute(['user' => $userId, 'cat' => $categoryId]);
            $cur = $q->fetch(\PDO::FETCH_ASSOC) ?: [];
            $anchor = isset($cur['cycle_anchor_ts']) ? (int) $cur['cycle_anchor_ts'] : 0;
            if ($anchor <= 0) {
                $utcMidnight = self::utcMidnightEuropeParis($now);
                $upd = $pdo->prepare('UPDATE user_subscriptions SET cycle_anchor_ts = :anchor WHERE user_id = :user AND category_id = :cat');
                $upd->execute(['anchor' => $utcMidnight, 'user' => $userId, 'cat' => $categoryId]);
            }
        }

        return Response::ok();
    }

    private static function utcMidnightEuropeParis(int $nowUtc): int
    {
        $tz = new DateTimeZone('Europe/Paris');
        $utc = new DateTimeZone('UTC');
        $local = (new DateTimeImmutable('@' . $nowUtc))->setTimezone($tz)->setTime(0, 0, 0);
        return $local->setTimezone($utc)->getTimestamp();
    }
}

