<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class AccountService
{
    public function __construct(
        private readonly DatabaseConnection $connection,
        private readonly LicenseService $licenses
    ) {
    }

    /**
     * @return array{email:string,emailVerified:bool,pseudo:string,role:string,licenses:list<array<string,mixed>>}
     */
    public function getAccount(int $userId): array
    {
        $userStmt = $this->connection->pdo()->prepare('SELECT email, email_verified_at FROM users WHERE id = :id');
        $userStmt->execute(['id' => $userId]);
        $user = $userStmt->fetch(PDO::FETCH_ASSOC);
        if (!$user) {
            throw new \RuntimeException('USER_NOT_FOUND');
        }

        $this->ensureProfile($userId);

        $profileStmt = $this->connection->pdo()->prepare('SELECT pseudo, role FROM user_profiles WHERE user_id = :id');
        $profileStmt->execute(['id' => $userId]);
        $profile = $profileStmt->fetch(PDO::FETCH_ASSOC) ?: ['pseudo' => 'utilisateur', 'role' => 'standard'];

        $licenses = $this->licenses->listForUser($userId);
        $effectiveRole = $this->licenses->computeEffectiveRole($userId) ?: (string) ($profile['role'] ?? 'standard');

        return [
            'email' => (string) ($user['email'] ?? ''),
            'emailVerified' => !empty($user['email_verified_at']),
            'pseudo' => (string) ($profile['pseudo'] ?? 'utilisateur'),
            'role' => $effectiveRole,
            'licenses' => $licenses,
        ];
    }

    public function updatePseudo(int $userId, string $pseudo): void
    {
        $pseudo = trim($pseudo);
        if ($pseudo === '') {
            throw new \InvalidArgumentException('PSEUDO_REQUIRED');
        }

        $this->ensureProfile($userId);
        $stmt = $this->connection->pdo()->prepare('UPDATE user_profiles SET pseudo = :pseudo WHERE user_id = :id');
        $stmt->execute([
            'pseudo' => mb_substr($pseudo, 0, 60),
            'id' => $userId,
        ]);
    }

    public function applyLicense(int $userId, string $code): array
    {
        return $this->licenses->redeemCode($userId, trim($code));
    }

    private function ensureProfile(int $userId): void
    {
        $stmt = $this->connection->pdo()->prepare(
            'INSERT IGNORE INTO user_profiles(user_id, pseudo, role) VALUES(:id, :pseudo, :role)'
        );
        $stmt->execute([
            'id' => $userId,
            'pseudo' => 'utilisateur',
            'role' => 'standard',
        ]);
    }
}
