<?php

declare(strict_types=1);

namespace Skyboard\Application\Services\Admin;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class ActionCatalogAdminService
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function list(): array
    {
        $pdo = $this->connection->pdo();
        $stmt = $pdo->query('SELECT action_id, kind, created_at FROM action_catalog ORDER BY action_id ASC');
        $rows = $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
        return array_map(fn(array $r) => [
            'action_id' => (string) $r['action_id'],
            'kind' => (string) $r['kind'],
            'created_at' => isset($r['created_at']) ? (int) $r['created_at'] : 0,
        ], $rows ?: []);
    }

    /**
     * @return array<string,mixed>
     */
    public function get(string $actionId): array
    {
        $pdo = $this->connection->pdo();
        $stmt = $pdo->prepare('SELECT action_id, kind, created_at, created_by FROM action_catalog WHERE action_id = :id');
        $stmt->execute(['id' => $actionId]);
        $cat = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$cat) throw new \RuntimeException('ACTION_NOT_FOUND');
        $stmtV = $pdo->prepare('SELECT version, status, etag, definition_json, updated_at, updated_by FROM action_catalog_versions WHERE action_id = :id ORDER BY version DESC');
        $stmtV->execute(['id' => $actionId]);
        $versions = $stmtV->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return [
            'action_id' => (string) $cat['action_id'],
            'kind' => (string) $cat['kind'],
            'created_at' => isset($cat['created_at']) ? (int) $cat['created_at'] : 0,
            'created_by' => isset($cat['created_by']) ? (int) $cat['created_by'] : null,
            'versions' => array_map(fn($v) => [
                'version' => (int) $v['version'],
                'status' => (string) $v['status'],
                'etag' => (string) $v['etag'],
                'definition' => (function($def){ $d = json_decode((string) ($def ?? '{}'), true); return is_array($d) ? $d : []; })($v['definition_json'] ?? '{}'),
                'updated_at' => isset($v['updated_at']) ? (int) $v['updated_at'] : 0,
                'updated_by' => isset($v['updated_by']) ? (int) $v['updated_by'] : null,
            ], $versions)
        ];
    }

    /**
     * @return array<string,mixed>
     */
    public function createAction(array $input): array
    {
        $actionId = (string) ($input['action_id'] ?? '');
        $kind = (string) ($input['kind'] ?? '');
        $createdBy = isset($input['created_by']) ? (int) $input['created_by'] : null;
        if ($actionId === '' || !preg_match('/^[a-z0-9_.:-]{2,64}$/', $actionId)) {
            throw new \InvalidArgumentException('FIELD_ACTION_ID_INVALID');
        }
        if (!in_array($kind, [
            'User.SubscribeCategory',
            'User.UnsubscribeCategory',
            'User.SetCategoryFrequencyOverride',
            'UserData.SetKey',
            'User.SubscribeAndSetPref',
            'OpenDialog'
        ], true)) {
            throw new \InvalidArgumentException('FIELD_KIND_INVALID');
        }
        $pdo = $this->connection->pdo();
        $stmt = $pdo->prepare('INSERT INTO action_catalog(action_id, kind, created_at, created_by) VALUES(:id, :kind, :ts, :by)');
        $stmt->execute(['id' => $actionId, 'kind' => $kind, 'ts' => time(), 'by' => $createdBy]);
        return $this->get($actionId);
    }

    /**
     * @return array<string,mixed>
     */
    public function createVersion(array $input): array
    {
        $actionId = (string) ($input['action_id'] ?? '');
        $version = isset($input['version']) ? (int) $input['version'] : 0;
        $status = (string) ($input['status'] ?? 'draft');
        $definition = $input['definition'] ?? [];
        $updatedBy = isset($input['updated_by']) ? (int) $input['updated_by'] : null;
        if ($actionId === '' || $version <= 0) throw new \InvalidArgumentException('INVALID_PAYLOAD');
        if (!in_array($status, ['draft','active','deprecated','disabled'], true)) throw new \InvalidArgumentException('FIELD_STATUS_INVALID');
        if (!is_array($definition)) throw new \InvalidArgumentException('FIELD_DEFINITION_INVALID');
        // Validate payloadSchema presence (optional at V1), ensure additionalProperties default false if provided.
        if (isset($definition['payloadSchema']) && !is_array($definition['payloadSchema'])) {
            throw new \InvalidArgumentException('FIELD_SCHEMA_INVALID');
        }
        // Harden: enforce additionalProperties:false on payloadSchema when missing (+defaults)
        $definition = $this->normalizeDefinition($definition);
        $etag = 'sha256:' . hash('sha256', $this->canonicalJson($definition));
        $pdo = $this->connection->pdo();
        // Ensure action exists
        $check = $pdo->prepare('SELECT 1 FROM action_catalog WHERE action_id = :id');
        $check->execute(['id' => $actionId]);
        if (!$check->fetchColumn()) throw new \RuntimeException('ACTION_NOT_FOUND');

        $stmt = $pdo->prepare('INSERT INTO action_catalog_versions(action_id, version, status, definition_json, etag, updated_at, updated_by) VALUES(:id, :ver, :status, :def, :etag, :ts, :by)');
        $stmt->execute([
            'id' => $actionId,
            'ver' => $version,
            'status' => $status,
            'def' => json_encode($definition, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
            'etag' => $etag,
            'ts' => time(),
            'by' => $updatedBy,
        ]);
        return $this->get($actionId);
    }

    public function updateVersionStatus(string $actionId, int $version, string $status, ?int $updatedBy = null): array
    {
        if (!in_array($status, ['draft','active','deprecated','disabled'], true)) throw new \InvalidArgumentException('FIELD_STATUS_INVALID');
        $pdo = $this->connection->pdo();
        $stmt = $pdo->prepare('UPDATE action_catalog_versions SET status = :status, updated_at = :ts, updated_by = :by WHERE action_id = :id AND version = :ver');
        $stmt->execute(['status' => $status, 'ts' => time(), 'by' => $updatedBy, 'id' => $actionId, 'ver' => $version]);
        return $this->get($actionId);
    }

    /**
     * Delete a specific version of an action (if not active and not referenced by any notification).
     * @return array<string,mixed>
     */
    public function deleteVersion(string $actionId, int $version): array
    {
        if ($actionId === '' || $version <= 0) throw new \InvalidArgumentException('INVALID_PAYLOAD');
        $pdo = $this->connection->pdo();
        // Check version exists and is not active
        $chk = $pdo->prepare('SELECT status FROM action_catalog_versions WHERE action_id = :id AND version = :ver');
        $chk->execute(['id' => $actionId, 'ver' => $version]);
        $row = $chk->fetch(PDO::FETCH_ASSOC);
        if (!$row) throw new \RuntimeException('NOT_FOUND');
        $status = strtolower((string) ($row['status'] ?? ''));
        if ($status === 'active') {
            throw new \InvalidArgumentException('INVALID_OPERATION_ACTIVE_VERSION');
        }
        // Check if referenced in any notification (strict decode after coarse LIKE)
        $like = $pdo->prepare('SELECT id, actions_ref_json FROM notifications_rich WHERE actions_ref_json IS NOT NULL AND actions_ref_json != "" AND actions_ref_json LIKE :needle');
        $like->execute(['needle' => '%"ref":"' . $actionId . '"%']);
        $refs = $like->fetchAll(PDO::FETCH_ASSOC) ?: [];
        foreach ($refs as $r) {
            $json = (string) ($r['actions_ref_json'] ?? '');
            $data = json_decode($json, true);
            if (is_array($data) && isset($data['actions']) && is_array($data['actions'])) {
                foreach ($data['actions'] as $a) {
                    if (is_array($a) && (string) ($a['ref'] ?? '') === $actionId && (int) ($a['version'] ?? 0) === $version) {
                        throw new \RuntimeException('ACTION_VERSION_IN_USE');
                    }
                }
            }
        }
        $del = $pdo->prepare('DELETE FROM action_catalog_versions WHERE action_id = :id AND version = :ver');
        $del->execute(['id' => $actionId, 'ver' => $version]);
        return $this->get($actionId);
    }

    /**
     * Delete an action (catalog entry) if no notification references any of its versions.
     * @return array<string,mixed>
     */
    public function deleteAction(string $actionId): array
    {
        if ($actionId === '') throw new \InvalidArgumentException('INVALID_PAYLOAD');
        $pdo = $this->connection->pdo();
        // Check references in notifications (strict decode after coarse LIKE)
        $like = $pdo->prepare('SELECT id, actions_ref_json FROM notifications_rich WHERE actions_ref_json IS NOT NULL AND actions_ref_json != "" AND actions_ref_json LIKE :needle');
        $like->execute(['needle' => '%"ref":"' . $actionId . '"%']);
        $refs = $like->fetchAll(PDO::FETCH_ASSOC) ?: [];
        foreach ($refs as $r) {
            $json = (string) ($r['actions_ref_json'] ?? '');
            $data = json_decode($json, true);
            if (is_array($data) && isset($data['actions']) && is_array($data['actions'])) {
                foreach ($data['actions'] as $a) {
                    if (is_array($a) && (string) ($a['ref'] ?? '') === $actionId) {
                        throw new \RuntimeException('ACTION_IN_USE');
                    }
                }
            }
        }
        // Safe to delete; versions cascade delete
        $del = $pdo->prepare('DELETE FROM action_catalog WHERE action_id = :id');
        $del->execute(['id' => $actionId]);
        return ['deleted' => true];
    }

    private function canonicalJson(mixed $value): string
    {
        if (is_array($value)) {
            if ($this->isAssoc($value)) {
                ksort($value);
                $out = [];
                foreach ($value as $k => $v) {
                    $out[$k] = json_decode($this->canonicalJson($v), true);
                }
                return json_encode($out, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
            }
            $out = array_map(fn($v) => json_decode($this->canonicalJson($v), true), $value);
            return json_encode($out, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
        }
        if (is_bool($value) || is_null($value) || is_numeric($value)) {
            return json_encode($value, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
        }
        return json_encode((string) $value, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
    }

    private function isAssoc(array $arr): bool
    {
        if ($arr === []) return false;
        return array_keys($arr) !== range(0, count($arr) - 1);
    }

    /**
     * @param array<string,mixed> $definition
     * @return array<string,mixed>
     */
    private function normalizeDefinition(array $definition): array
    {
        $def = $definition;
        $schema = $def['payloadSchema'] ?? null;
        if (is_array($schema)) {
            if (!array_key_exists('additionalProperties', $schema)) {
                $schema['additionalProperties'] = false;
            }
            if (!isset($schema['type'])) {
                $schema['type'] = 'object';
            }
            if (!isset($schema['properties']) || !is_array($schema['properties'])) {
                $schema['properties'] = [];
            }
            if (!isset($schema['required']) || !is_array($schema['required'])) {
                $schema['required'] = [];
            }
            $def['payloadSchema'] = $schema;
        }
        return $def;
    }
}
