<?php

declare(strict_types=1);

namespace Skyboard\Application\Services\Admin;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class LicenseAdminService
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function list(): array
    {
        $sql = 'SELECT l.id, l.code, l.role, l.packs_json, l.expires_at, l.assigned_user_id, l.redeemed_at, l.created_at, u.email AS assigned_email
                FROM licenses l
                LEFT JOIN users u ON u.id = l.assigned_user_id
                ORDER BY l.created_at DESC';
        $stmt = $this->connection->pdo()->query($sql);
        $rows = $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
        return array_map([$this, 'hydrate'], $rows ?: []);
    }

    /**
     * @return array<string,mixed>
     */
    public function create(array $input): array
    {
        $code = trim((string) ($input['code'] ?? $this->generateCode()));
        if ($code === '') {
            $code = $this->generateCode();
        }
        $role = trim((string) ($input['role'] ?? 'standard'));
        if ($role === '') {
            throw new \InvalidArgumentException('ROLE_REQUIRED');
        }
        $packs = $input['packs'] ?? [];
        if (is_string($packs)) {
            $packs = $packs === '' ? [] : explode(',', $packs);
        }
        if (!is_array($packs)) {
            throw new \InvalidArgumentException('PACKS_INVALID');
        }
        $expiresAt = $input['expiresAt'] ?? null;
        $expiresAt = $expiresAt !== null ? (int) $expiresAt : null;

        $stmt = $this->connection->pdo()->prepare('INSERT INTO licenses(code, role, packs_json, expires_at, assigned_user_id, redeemed_at, created_at) VALUES(:code, :role, :packs, :expiresAt, NULL, NULL, :createdAt)');
        $stmt->execute([
            'code' => strtoupper($code),
            'role' => $role,
            'packs' => json_encode(array_values(array_map('strval', $packs)), JSON_UNESCAPED_SLASHES),
            'expiresAt' => $expiresAt,
            'createdAt' => time(),
        ]);

        return $this->get((int) $this->connection->pdo()->lastInsertId());
    }

    /**
     * @return array<string,mixed>
     */
    public function assign(int $licenseId, int $userId): array
    {
        $license = $this->get($licenseId);
        $now = time();

        $pdo = $this->connection->pdo();
        $pdo->beginTransaction();
        try {
            $pdo->prepare('UPDATE licenses SET assigned_user_id = :userId, redeemed_at = :redeemedAt WHERE id = :id')
                ->execute([
                    'userId' => $userId,
                    'redeemedAt' => $now,
                    'id' => $licenseId,
                ]);

            $sql = $this->insertIgnoreSql('user_licenses', 'user_id, license_id', ':userId, :licenseId');
            $pdo->prepare($sql)->execute(['userId' => $userId, 'licenseId' => $licenseId]);

            $pdo->commit();
        } catch (\Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }

        return $this->get($licenseId);
    }

    public function unassign(int $licenseId): array
    {
        $pdo = $this->connection->pdo();
        $pdo->beginTransaction();
        try {
            $pdo->prepare('UPDATE licenses SET assigned_user_id = NULL, redeemed_at = NULL WHERE id = :id')
                ->execute(['id' => $licenseId]);
            $pdo->prepare('DELETE FROM user_licenses WHERE license_id = :id')
                ->execute(['id' => $licenseId]);
            $pdo->commit();
        } catch (\Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }

        return $this->get($licenseId);
    }

    /**
     * @return array<string,mixed>
     */
    public function generateActivationLink(int $licenseId, int $userId, int $ttlSeconds = 86400): array
    {
        $license = $this->get($licenseId);
        $token = bin2hex(random_bytes(24));
        $expiresAt = time() + max(60, $ttlSeconds);

        $this->connection->pdo()->prepare('INSERT INTO license_links(user_id, token, code, expires_at, created_at) VALUES(:userId, :token, :code, :expiresAt, :createdAt)')
            ->execute([
                'userId' => $userId,
                'token' => $token,
                'code' => $license['code'],
                'expiresAt' => $expiresAt,
                'createdAt' => time(),
            ]);

        return [
            'token' => $token,
            'expiresAt' => $expiresAt,
            'url' => sprintf('/activate-license?token=%s', $token),
        ];
    }

    public function delete(int $licenseId): void
    {
        $pdo = $this->connection->pdo();
        $pdo->prepare('DELETE FROM user_licenses WHERE license_id = :id')->execute(['id' => $licenseId]);
        $pdo->prepare('DELETE FROM license_links WHERE code = (SELECT code FROM licenses WHERE id = :id)')->execute(['id' => $licenseId]);
        $pdo->prepare('DELETE FROM licenses WHERE id = :id')->execute(['id' => $licenseId]);
    }

    /**
     * @return array<string,mixed>
     */
    public function get(int $licenseId): array
    {
        $stmt = $this->connection->pdo()->prepare('SELECT l.id, l.code, l.role, l.packs_json, l.expires_at, l.assigned_user_id, l.redeemed_at, l.created_at, u.email AS assigned_email FROM licenses l LEFT JOIN users u ON u.id = l.assigned_user_id WHERE l.id = :id');
        $stmt->execute(['id' => $licenseId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            throw new \RuntimeException('LICENSE_NOT_FOUND');
        }
        return $this->hydrate($row);
    }

    /**
     * @param array<string,mixed> $row
     * @return array<string,mixed>
     */
    private function hydrate(array $row): array
    {
        $row['expires_at'] = isset($row['expires_at']) ? (int) $row['expires_at'] : null;
        $row['redeemed_at'] = isset($row['redeemed_at']) ? (int) $row['redeemed_at'] : null;
        $row['created_at'] = (int) ($row['created_at'] ?? 0);
        $row['packs'] = $row['packs_json'] ? json_decode((string) $row['packs_json'], true) : [];
        unset($row['packs_json']);
        return $row;
    }

    private function generateCode(): string
    {
        return strtoupper('SB-' . substr(bin2hex(random_bytes(6)), 0, 12));
    }

    private function insertIgnoreSql(string $table, string $columns, string $values): string
    {
        return sprintf('INSERT IGNORE INTO %s(%s) VALUES(%s)', $table, $columns, $values);
    }
}
