<?php

declare(strict_types=1);

namespace Skyboard\Application\Services\Admin;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class NotificationCategoryAdminService
{
    public function __construct(private readonly DatabaseConnection $connection)
    {
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function list(): array
    {
        $stmt = $this->connection->pdo()->query('SELECT id, slug, name, sort_order, active, audience_mode, dispatch_mode, frequency_kind, frequency_param, anchor_ts, allow_user_override FROM notification_categories ORDER BY sort_order ASC, name ASC');
        $rows = $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
        return array_map(function(array $row) {
            return [
                'id' => (int) $row['id'],
                'slug' => (string) $row['slug'],
                'name' => (string) $row['name'],
                'sort_order' => (int) $row['sort_order'],
                'active' => (bool) $row['active'],
                'audience_mode' => (string) ($row['audience_mode'] ?? 'EVERYONE'),
                'dispatch_mode' => (string) ($row['dispatch_mode'] ?? 'BROADCAST'),
                'frequency_kind' => (string) ($row['frequency_kind'] ?? 'IMMEDIATE'),
                'frequency_param' => isset($row['frequency_param']) ? (int) $row['frequency_param'] : null,
                'anchor_ts' => isset($row['anchor_ts']) ? (int) $row['anchor_ts'] : null,
                'allow_user_override' => !empty($row['allow_user_override']),
            ];
        }, $rows ?: []);
    }

    /**
     * @return array<string,mixed>
     */
    public function create(array $input): array
    {
        $payload = $this->normalize($input, true);
        $pdo = $this->connection->pdo();
        $stmt = $pdo->prepare('INSERT INTO notification_categories(slug, name, sort_order, active, audience_mode, dispatch_mode, frequency_kind, frequency_param, anchor_ts, allow_user_override)
                               VALUES(:slug, :name, :sort_order, :active, :audience_mode, :dispatch_mode, :frequency_kind, :frequency_param, :anchor_ts, :allow_user_override)');
        $stmt->execute([
            'slug' => $payload['slug'],
            'name' => $payload['name'],
            'sort_order' => $payload['sort_order'],
            'active' => $payload['active'],
            'audience_mode' => $payload['audience_mode'],
            'dispatch_mode' => $payload['dispatch_mode'],
            'frequency_kind' => $payload['frequency_kind'],
            'frequency_param' => $payload['frequency_param'],
            'anchor_ts' => $payload['anchor_ts'],
            'allow_user_override' => $payload['allow_user_override'],
        ]);
        $id = (int) $pdo->lastInsertId();
        return $this->get($id);
    }

    /**
     * @return array<string,mixed>
     */
    public function update(int $id, array $input): array
    {
        $payload = $this->normalize($input, false);
        if ($payload === []) return $this->get($id);
        $sets = [];
        foreach ($payload as $k => $_) {
            $sets[] = sprintf('%s = :%s', $k, $k);
        }
        $sql = 'UPDATE notification_categories SET ' . implode(', ', $sets) . ' WHERE id = :id';
        $payload['id'] = $id;
        $stmt = $this->connection->pdo()->prepare($sql);
        $stmt->execute($payload);
        return $this->get($id);
    }

    /**
     * Delete a category with a full purge of related data in a single transaction.
     * @return array{notification_user_state:int, notifications_rich:int, user_subscriptions:int, scheduler_meta:int, categories:int}
     */
    public function delete(int $id): array
    {
        $pdo = $this->connection->pdo();
        // Ensure the category exists before attempting purge
        $check = $pdo->prepare('SELECT 1 FROM notification_categories WHERE id = :id');
        $check->execute(['id' => $id]);
        if (!$check->fetchColumn()) {
            throw new \RuntimeException('CATEGORY_NOT_FOUND');
        }

        $pdo->beginTransaction();
        try {
            // 1) notification_user_state via join on notifications_rich
            $stmtNusCount = $pdo->prepare('SELECT COUNT(*) AS cnt
              FROM notification_user_state nus
              JOIN notifications_rich nr ON nr.id = nus.notification_id
              WHERE nr.category_id = :id');
            $stmtNusCount->execute(['id' => $id]);
            $nusCount = (int) (($stmtNusCount->fetch(PDO::FETCH_ASSOC)['cnt'] ?? 0));

            $stmtNus = $pdo->prepare('DELETE nus
              FROM notification_user_state nus
              JOIN notifications_rich nr ON nr.id = nus.notification_id
              WHERE nr.category_id = :id');
            $stmtNus->execute(['id' => $id]);

            // 2) notifications_rich
            $stmtNr = $pdo->prepare('DELETE FROM notifications_rich WHERE category_id = :id');
            $stmtNr->execute(['id' => $id]);
            $nrCount = (int) $stmtNr->rowCount();

            // 3) user_subscriptions
            $stmtUs = $pdo->prepare('DELETE FROM user_subscriptions WHERE category_id = :id');
            $stmtUs->execute(['id' => $id]);
            $usCount = (int) $stmtUs->rowCount();

            // 4) scheduler_meta
            $stmtSm = $pdo->prepare('DELETE FROM scheduler_meta WHERE category_id = :id');
            $stmtSm->execute(['id' => $id]);
            $smCount = (int) $stmtSm->rowCount();

            // 5) category
            $stmtCat = $pdo->prepare('DELETE FROM notification_categories WHERE id = :id');
            $stmtCat->execute(['id' => $id]);
            $catCount = (int) $stmtCat->rowCount();
            if ($catCount !== 1) {
                $pdo->rollBack();
                throw new \RuntimeException('CATEGORY_NOT_FOUND');
            }

            $pdo->commit();
            return [
                'notification_user_state' => $nusCount,
                'notifications_rich' => $nrCount,
                'user_subscriptions' => $usCount,
                'scheduler_meta' => $smCount,
                'categories' => $catCount,
            ];
        } catch (\Throwable $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            throw $e;
        }
    }

    /**
     * @return array<string,mixed>
     */
    public function get(int $id): array
    {
        $stmt = $this->connection->pdo()->prepare('SELECT id, slug, name, sort_order, active, audience_mode, dispatch_mode, frequency_kind, frequency_param, anchor_ts, allow_user_override FROM notification_categories WHERE id = :id');
        $stmt->execute(['id' => $id]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) throw new \RuntimeException('CATEGORY_NOT_FOUND');
        return [
            'id' => (int) $row['id'],
            'slug' => (string) $row['slug'],
            'name' => (string) $row['name'],
            'sort_order' => (int) $row['sort_order'],
            'active' => (bool) $row['active'],
            'audience_mode' => (string) ($row['audience_mode'] ?? 'EVERYONE'),
            'dispatch_mode' => (string) ($row['dispatch_mode'] ?? 'BROADCAST'),
            'frequency_kind' => (string) ($row['frequency_kind'] ?? 'IMMEDIATE'),
            'frequency_param' => isset($row['frequency_param']) ? (int) $row['frequency_param'] : null,
            'anchor_ts' => isset($row['anchor_ts']) ? (int) $row['anchor_ts'] : null,
            'allow_user_override' => !empty($row['allow_user_override']),
        ];
    }

    /**
     * @return array<string,mixed>
     */
    private function normalize(array $input, bool $requireAll): array
    {
        $out = [];
        if (array_key_exists('slug', $input) || $requireAll) {
            $slug = trim((string) ($input['slug'] ?? ''));
            if ($slug === '' && $requireAll) throw new \InvalidArgumentException('FIELD_SLUG_REQUIRED');
            if ($slug !== '' && !preg_match('/^[a-z0-9-]{2,64}$/', $slug)) throw new \InvalidArgumentException('FIELD_SLUG_INVALID');
            $out['slug'] = $slug;
        }
        if (array_key_exists('name', $input) || $requireAll) {
            $name = trim((string) ($input['name'] ?? ''));
            if ($name === '' && $requireAll) throw new \InvalidArgumentException('FIELD_NAME_REQUIRED');
            $out['name'] = $name;
        }
        if (array_key_exists('sort_order', $input) || $requireAll) {
            $out['sort_order'] = (int) ($input['sort_order'] ?? 0);
        }
        if (array_key_exists('active', $input) || $requireAll) {
            $out['active'] = !empty($input['active']) ? 1 : 0;
        }

        // Extended fields
        if (array_key_exists('audience_mode', $input) || $requireAll) {
            $aud = strtoupper(trim((string) ($input['audience_mode'] ?? 'EVERYONE')));
            if (!in_array($aud, ['EVERYONE','SUBSCRIBERS'], true)) throw new \InvalidArgumentException('FIELD_AUDIENCE_MODE_INVALID');
            $out['audience_mode'] = $aud;
        }
        if (array_key_exists('dispatch_mode', $input) || $requireAll) {
            $disp = strtoupper(trim((string) ($input['dispatch_mode'] ?? 'BROADCAST')));
            if (!in_array($disp, ['BROADCAST','PERSONALIZED'], true)) throw new \InvalidArgumentException('FIELD_DISPATCH_MODE_INVALID');
            $out['dispatch_mode'] = $disp;
        }
        if (array_key_exists('frequency_kind', $input) || $requireAll) {
            $fk = strtoupper(trim((string) ($input['frequency_kind'] ?? 'IMMEDIATE')));
            if (!in_array($fk, ['IMMEDIATE','EVERY_N_DAYS','WEEKLY','MONTHLY'], true)) throw new \InvalidArgumentException('FIELD_FREQUENCY_KIND_INVALID');
            $out['frequency_kind'] = $fk;
        }
        if (array_key_exists('frequency_param', $input) || $requireAll) {
            $paramRaw = $input['frequency_param'] ?? null;
            $param = ($paramRaw === null || $paramRaw === '') ? null : (int) $paramRaw;
            $out['frequency_param'] = $param;
        }
        if (array_key_exists('anchor_ts', $input) || $requireAll) {
            $anchorRaw = $input['anchor_ts'] ?? null;
            $anchor = ($anchorRaw === null || $anchorRaw === '') ? null : max(0, (int) $anchorRaw);
            $out['anchor_ts'] = $anchor;
        }
        if (array_key_exists('allow_user_override', $input) || $requireAll) {
            $out['allow_user_override'] = !empty($input['allow_user_override']) ? 1 : 0;
        }

        // Cross-field validations
        $disp = $out['dispatch_mode'] ?? strtoupper(trim((string) ($input['dispatch_mode'] ?? 'BROADCAST')));
        $fk = $out['frequency_kind'] ?? strtoupper(trim((string) ($input['frequency_kind'] ?? 'IMMEDIATE')));
        if ($disp === 'PERSONALIZED' && $fk === 'IMMEDIATE') {
            throw new \InvalidArgumentException('INVALID_CATEGORY_MODE');
        }

        // Frequency_param constraints based on kind
        $param = $out['frequency_param'] ?? null;
        if ($fk === 'EVERY_N_DAYS') {
            if (!is_int($param) || !in_array($param, [1,2,3,4,5,6,7,14,21], true)) {
                throw new \InvalidArgumentException('FREQUENCY_PARAM_INVALID');
            }
        } elseif ($fk === 'WEEKLY') {
            if (!is_int($param) || $param < 1 || $param > 7) throw new \InvalidArgumentException('FREQUENCY_PARAM_INVALID');
        } elseif ($fk === 'MONTHLY') {
            if (!is_int($param) || $param < 1 || $param > 28) throw new \InvalidArgumentException('FREQUENCY_PARAM_INVALID');
        } else { // IMMEDIATE
            $out['frequency_param'] = null;
        }

        return $out;
    }
}
