<?php

declare(strict_types=1);

namespace Skyboard\Application\Services\Admin;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;
use Skyboard\Infrastructure\Security\PasswordHasher;

final class UserAdminService
{
    public function __construct(
        private readonly DatabaseConnection $connection,
        private readonly PasswordHasher $hasher
    ) {
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function list(): array
    {
        $sql = 'SELECT u.id, u.email, u.created_at, u.email_verified_at, u.blocked, p.pseudo, p.role, COUNT(b.id) AS board_count
                FROM users u
                LEFT JOIN user_profiles p ON p.user_id = u.id
                LEFT JOIN boards b ON b.user_id = u.id
                GROUP BY u.id
                ORDER BY u.created_at DESC';
        $stmt = $this->connection->pdo()->query($sql);
        $rows = $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
        return array_map(static function (array $row): array {
            $row['blocked'] = (bool) ($row['blocked'] ?? 0);
            $row['board_count'] = (int) ($row['board_count'] ?? 0);
            $row['created_at'] = (int) ($row['created_at'] ?? 0);
            $row['email_verified_at'] = isset($row['email_verified_at']) ? (int) $row['email_verified_at'] : null;
            return $row;
        }, $rows ?: []);
    }

    /**
     * @return array<string,mixed>
     */
    public function create(string $email, string $password, string $role = 'standard', ?string $pseudo = null): array
    {
        $email = strtolower(trim($email));
        if ($email === '') {
            throw new \InvalidArgumentException('EMAIL_REQUIRED');
        }
        if ($password === '') {
            throw new \InvalidArgumentException('PASSWORD_REQUIRED');
        }
        if ($pseudo !== null) {
            $pseudo = trim($pseudo);
            if ($pseudo === '') {
                $pseudo = null;
            }
        }
        $pdo = $this->connection->pdo();
        $existing = $pdo->prepare('SELECT 1 FROM users WHERE email = :email');
        $existing->execute(['email' => $email]);
        if ($existing->fetchColumn()) {
            throw new \RuntimeException('EMAIL_ALREADY_EXISTS');
        }

        $stmt = $pdo->prepare('INSERT INTO users(email, password_hash, created_at, blocked) VALUES(:email, :hash, :createdAt, :blocked)');
        $stmt->execute([
            'email' => $email,
            'hash' => $this->hasher->hash($password),
            'createdAt' => time(),
            'blocked' => 0,
        ]);
        $userId = (int) $pdo->lastInsertId();

        $this->upsertProfile($userId, $pseudo, $role);

        return $this->get($userId);
    }

    /**
     * @return array<string,mixed>
     */
    public function updateProfile(int $userId, array $payload): array
    {
        $fields = [];
        $params = ['id' => $userId];
        if (array_key_exists('pseudo', $payload)) {
            $fields[] = 'pseudo = :pseudo';
            $params['pseudo'] = $payload['pseudo'] === null ? null : trim((string) $payload['pseudo']);
        }
        if (array_key_exists('role', $payload)) {
            $role = trim((string) $payload['role']);
            if ($role === '') {
                throw new \InvalidArgumentException('ROLE_REQUIRED');
            }
            $fields[] = 'role = :role';
            $params['role'] = $role;
        }

        if ($fields !== []) {
            $pdo = $this->connection->pdo();
            if ($this->profileExists($userId)) {
                $sql = 'UPDATE user_profiles SET ' . implode(', ', $fields) . ' WHERE user_id = :id';
                $pdo->prepare($sql)->execute($params);
            } else {
                $insertParams = [
                    'id' => $userId,
                    'pseudo' => $params['pseudo'] ?? null,
                    'role' => $params['role'] ?? 'standard',
                ];
                $pdo->prepare('INSERT INTO user_profiles(user_id, pseudo, role) VALUES(:id, :pseudo, :role)')->execute($insertParams);
            }
        }

        if (array_key_exists('blocked', $payload)) {
            $stmt = $this->connection->pdo()->prepare('UPDATE users SET blocked = :blocked WHERE id = :id');
            $stmt->execute([
                'blocked' => $payload['blocked'] ? 1 : 0,
                'id' => $userId,
            ]);
        }

        return $this->get($userId);
    }

    public function setBlocked(int $userId, bool $blocked): void
    {
        $stmt = $this->connection->pdo()->prepare('UPDATE users SET blocked = :blocked WHERE id = :id');
        $stmt->execute(['blocked' => $blocked ? 1 : 0, 'id' => $userId]);
    }

    public function delete(int $userId): void
    {
        $pdo = $this->connection->pdo();
        $pdo->prepare('DELETE FROM sessions WHERE user_id = :id')->execute(['id' => $userId]);
        $pdo->prepare('DELETE FROM user_licenses WHERE user_id = :id')->execute(['id' => $userId]);
        $pdo->prepare('DELETE FROM user_profiles WHERE user_id = :id')->execute(['id' => $userId]);
        $pdo->prepare('DELETE FROM users WHERE id = :id')->execute(['id' => $userId]);
    }

    /**
     * @return array<string,mixed>
     */
    public function get(int $userId): array
    {
        $stmt = $this->connection->pdo()->prepare('SELECT u.id, u.email, u.created_at, u.email_verified_at, u.blocked, p.pseudo, p.role FROM users u LEFT JOIN user_profiles p ON p.user_id = u.id WHERE u.id = :id');
        $stmt->execute(['id' => $userId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            throw new \RuntimeException('USER_NOT_FOUND');
        }
        $row['blocked'] = (bool) ($row['blocked'] ?? 0);
        $row['created_at'] = (int) ($row['created_at'] ?? 0);
        $row['email_verified_at'] = isset($row['email_verified_at']) ? (int) $row['email_verified_at'] : null;
        return $row;
    }

    /**
     * @return array{profile:array<string,mixed>, subscriptions:list<array<string,mixed>>, nus:array<string,int>, user_data:list<array<string,mixed>>, achievements:list<array<string,mixed>>}
     */
    public function getData(int $userId): array
    {
        $profile = $this->get($userId);
        $pdo = $this->connection->pdo();

        // Subscriptions with category info
        $stmt = $pdo->prepare("SELECT us.category_id, us.subscribed, us.override_kind, us.override_param, us.cycle_anchor_ts, us.last_delivered_notification_id,
                                      c.slug, c.name, c.dispatch_mode, c.allow_user_override
                               FROM user_subscriptions us
                               JOIN notification_categories c ON c.id = us.category_id
                               WHERE us.user_id = :id");
        $stmt->execute(['id' => $userId]);
        $subs = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $subscriptions = array_map(static function(array $r): array {
            return [
                'category_id' => (int) $r['category_id'],
                'slug' => (string) ($r['slug'] ?? ''),
                'name' => (string) ($r['name'] ?? ''),
                'dispatch_mode' => (string) ($r['dispatch_mode'] ?? ''),
                'allow_user_override' => !empty($r['allow_user_override']),
                'subscribed' => !empty($r['subscribed']),
                'override_kind' => $r['override_kind'] ?? null,
                'override_param' => isset($r['override_param']) ? (int) $r['override_param'] : null,
                'cycle_anchor_ts' => isset($r['cycle_anchor_ts']) ? (int) $r['cycle_anchor_ts'] : null,
                'last_delivered_notification_id' => isset($r['last_delivered_notification_id']) ? (int) $r['last_delivered_notification_id'] : null,
            ];
        }, $subs);

        // NUS metrics
        $stmt = $pdo->prepare("SELECT 
              SUM(CASE WHEN deleted_at IS NOT NULL THEN 1 ELSE 0 END) AS deleted,
              SUM(CASE WHEN archived_at IS NOT NULL AND deleted_at IS NULL THEN 1 ELSE 0 END) AS archived,
              SUM(CASE WHEN seen_at IS NOT NULL AND archived_at IS NULL AND deleted_at IS NULL THEN 1 ELSE 0 END) AS seen,
              SUM(CASE WHEN seen_at IS NULL AND archived_at IS NULL AND deleted_at IS NULL THEN 1 ELSE 0 END) AS unread
            FROM notification_user_state WHERE user_id = :id");
        $stmt->execute(['id' => $userId]);
        $nus = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
        $nusCounts = [
            'deleted' => (int) ($nus['deleted'] ?? 0),
            'archived' => (int) ($nus['archived'] ?? 0),
            'seen' => (int) ($nus['seen'] ?? 0),
            'unread' => (int) ($nus['unread'] ?? 0),
        ];

        // User data (KV JSON)
        $stmt = $pdo->prepare('SELECT namespace, `key`, value_json, updated_at FROM user_data WHERE user_id = :id ORDER BY namespace ASC, `key` ASC');
        $stmt->execute(['id' => $userId]);
        $dataRows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $userData = array_map(static function(array $r): array {
            return [
                'namespace' => (string) $r['namespace'],
                'key' => (string) $r['key'],
                'value' => json_decode((string) $r['value_json'], true) ?? null,
                'updated_at' => (int) $r['updated_at'],
            ];
        }, $dataRows);

        // Achievements (optional)
        $stmt = $pdo->prepare('SELECT code, earned_at, metadata FROM user_achievements WHERE user_id = :id ORDER BY earned_at DESC');
        $stmt->execute(['id' => $userId]);
        $achRows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $achievements = array_map(static function(array $r): array {
            return [
                'code' => (string) $r['code'],
                'earned_at' => (int) $r['earned_at'],
                'metadata' => isset($r['metadata']) ? (array) json_decode((string) $r['metadata'], true) : null,
            ];
        }, $achRows);

        return [
            'profile' => $profile,
            'subscriptions' => $subscriptions,
            'nus' => $nusCounts,
            'user_data' => $userData,
            'achievements' => $achievements,
        ];
    }

    private function upsertProfile(int $userId, ?string $pseudo, string $role): void
    {
        $pdo = $this->connection->pdo();
        if ($this->profileExists($userId)) {
            $stmt = $pdo->prepare('UPDATE user_profiles SET pseudo = :pseudo, role = :role WHERE user_id = :id');
        } else {
            $stmt = $pdo->prepare('INSERT INTO user_profiles(user_id, pseudo, role) VALUES(:id, :pseudo, :role)');
        }
        $stmt->execute([
            'id' => $userId,
            'pseudo' => $pseudo,
            'role' => $role,
        ]);
    }

    private function profileExists(int $userId): bool
    {
        $stmt = $this->connection->pdo()->prepare('SELECT 1 FROM user_profiles WHERE user_id = :id');
        $stmt->execute(['id' => $userId]);
        return (bool) $stmt->fetchColumn();
    }
}
