<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use Skyboard\Infrastructure\Security\PasswordHasher;
use Skyboard\Infrastructure\Security\SessionManager;

final class AuthService
{
    public function __construct(
        private readonly UserRepository $users,
        private readonly PasswordHasher $hasher,
        private readonly SessionManager $sessions
    ) {
    }

    /** @return array{token:string, csrf:string, user:array<string,mixed>} */
    public function register(string $email, string $password): array
    {
        $existing = $this->users->findByEmail($email);
        if ($existing) {
            throw new \RuntimeException('EMAIL_ALREADY_EXISTS');
        }
        $userId = $this->users->create($email, $this->hasher->hash($password));
        $session = $this->sessions->create($userId);
        $user = $this->users->findById($userId) ?? [];
        return $session + ['user' => $user];
    }

    /** @return array{token:string, csrf:string, user:array<string,mixed>} */
    public function login(string $email, string $password): array
    {
        $user = $this->users->findByEmail($email);
        if (!$user) {
            throw new \RuntimeException('INVALID_CREDENTIALS');
        }
        if (!empty($user['blocked'])) {
            throw new \RuntimeException('ACCOUNT_BLOCKED');
        }
        if (!$this->hasher->verify($password, $user['password_hash'])) {
            throw new \RuntimeException('INVALID_CREDENTIALS');
        }
        $session = $this->sessions->create((int) $user['id']);
        return $session + ['user' => $user];
    }

    public function logout(string $token): void
    {
        $this->sessions->destroy($token);
    }

    public function session(string $token): ?array
    {
        $session = $this->sessions->get($token);
        if (!$session) {
            return null;
        }
        $user = $this->users->findById((int) $session['user_id']);
        if (!$user) {
            return null;
        }
        return [
            'token' => $session['token'],
            'csrf' => $session['csrf_token'],
            'user' => $user,
        ];
    }
}

