<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use Skyboard\Domain\Boards\BoardInvariantChecker;
use Skyboard\Domain\Boards\BoardState;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class BoardCatalog
{
    private BoardInvariantChecker $invariants;

    public function __construct(private readonly DatabaseConnection $connection)
    {
        $this->invariants = new BoardInvariantChecker();
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function listForUser(int $userId): array
    {
        $stmt = $this->connection->pdo()->prepare('SELECT id, title, updated_at, created_at, revision, thumbnail_public_id FROM boards WHERE user_id = :user ORDER BY updated_at DESC');
        $stmt->execute(['user' => $userId]);
        return $stmt->fetchAll() ?: [];
    }

    public function create(int $userId, string $title): array
    {
        $boardState = BoardState::empty();
        $stmt = $this->connection->pdo()->prepare('INSERT INTO boards(user_id, title, state_json, updated_at, created_at, revision, history_json) VALUES(:user, :title, :state, :updatedAt, :createdAt, :revision, :history)');
        $payload = $boardState->toArray();
        $ts = $this->now();
        $history = json_encode([
            ['revision' => $ts, 'updatedAt' => $ts],
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        if ($history === false) {
            throw new \RuntimeException('BOARD_HISTORY_ENCODING_FAILED');
        }
        $stmt->execute([
            'user' => $userId,
            'title' => $title,
            'state' => json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
            'updatedAt' => $ts,
            'createdAt' => $ts,
            'revision' => $ts,
            'history' => $history,
        ]);
        $id = (int) $this->connection->pdo()->lastInsertId();
        return [
            'id' => $id,
            'title' => $title,
            'state' => $payload,
            'updatedAt' => $ts,
            'createdAt' => $ts,
            'revision' => $ts,
            'history' => [['revision' => $ts, 'updatedAt' => $ts]],
        ];
    }

    public function createFromState(int $userId, string $title, BoardState $state): array
    {
        $this->invariants->assert($state);
        $stmt = $this->connection->pdo()->prepare('INSERT INTO boards(user_id, title, state_json, updated_at, created_at, revision, history_json) VALUES(:user, :title, :state, :updatedAt, :createdAt, :revision, :history)');
        $payload = $state->toArray();
        $ts = $this->now();
        $history = json_encode([
            ['revision' => $ts, 'updatedAt' => $ts],
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        if ($history === false) {
            throw new \RuntimeException('BOARD_HISTORY_ENCODING_FAILED');
        }
        $stmt->execute([
            'user' => $userId,
            'title' => $title,
            'state' => json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
            'updatedAt' => $ts,
            'createdAt' => $ts,
            'revision' => $ts,
            'history' => $history,
        ]);
        $id = (int) $this->connection->pdo()->lastInsertId();
        return [
            'id' => $id,
            'title' => $title,
            'state' => $payload,
            'createdAt' => $ts,
            'updatedAt' => $ts,
            'revision' => $ts,
            'history' => [['revision' => $ts, 'updatedAt' => $ts]],
        ];
    }

    public function rename(int $boardId, int $userId, string $title): bool
    {
        $stmt = $this->connection->pdo()->prepare('UPDATE boards SET title = :title, updated_at = :ts WHERE id = :id AND user_id = :user');
        return $stmt->execute(['title' => $title, 'ts' => $this->now(), 'id' => $boardId, 'user' => $userId]);
    }

    public function delete(int $boardId, int $userId): bool
    {
        $stmt = $this->connection->pdo()->prepare('DELETE FROM boards WHERE id = :id AND user_id = :user');
        return $stmt->execute(['id' => $boardId, 'user' => $userId]);
    }

    public function getState(int $boardId, int $userId): ?BoardState
    {
        $stmt = $this->connection->pdo()->prepare('SELECT state_json FROM boards WHERE id = :id AND user_id = :user');
        $stmt->execute(['id' => $boardId, 'user' => $userId]);
        $row = $stmt->fetch();
        if (!$row) {
            return null;
        }
        $data = json_decode((string) $row['state_json'], true);
        if (!is_array($data)) {
            return null;
        }
        return BoardState::fromArray($data);
    }

    public function getMetadata(int $boardId, int $userId): ?array
    {
        $stmt = $this->connection->pdo()->prepare('SELECT id, title, updated_at, created_at, revision, history_json, thumbnail_public_id FROM boards WHERE id = :id AND user_id = :user');
        $stmt->execute(['id' => $boardId, 'user' => $userId]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function setThumbnail(int $boardId, int $userId, ?string $publicId): bool
    {
        $stmt = $this->connection->pdo()->prepare('UPDATE boards SET thumbnail_public_id = :pid, updated_at = :ts WHERE id = :id AND user_id = :user');
        $pid = ($publicId !== null && $publicId !== '') ? $publicId : null;
        return $stmt->execute(['pid' => $pid, 'ts' => $this->now(), 'id' => $boardId, 'user' => $userId]);
    }

    public function exists(int $boardId, int $userId): bool
    {
        $stmt = $this->connection->pdo()->prepare('SELECT 1 FROM boards WHERE id = :id AND user_id = :user');
        $stmt->execute(['id' => $boardId, 'user' => $userId]);
        return (bool) $stmt->fetchColumn();
    }
    private function now(): int
    {
        return (int) floor(microtime(true) * 1000);
    }
}
