<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class LicenseService
{
    public function __construct(
        private readonly DatabaseConnection $connection,
        private readonly string $secret
    ) {
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function listForUser(int $userId): array
    {
        $sql = 'SELECT l.id, l.code, l.role, l.packs_json, l.expires_at, l.redeemed_at, l.assigned_user_id
                FROM licenses l
                INNER JOIN user_licenses ul ON ul.license_id = l.id
                WHERE ul.user_id = :user
                ORDER BY l.redeemed_at DESC, l.created_at DESC';

        $stmt = $this->connection->pdo()->prepare($sql);
        $stmt->execute(['user' => $userId]);

        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return array_map(fn(array $row): array => $this->normalizeLicense($row), $rows);
    }

    /**
     * @return array{ok:bool,error?:string,role?:string}
     */
    public function redeemCode(int $userId, string $code): array
    {
        $code = trim($code);
        $verification = $this->verifyCode($code);
        if (!$verification['ok'] || !is_array($verification['payload'])) {
            return ['ok' => false, 'error' => 'invalid-signature'];
        }

        $payload = $verification['payload'];
        if (isset($payload['exp']) && (int) $payload['exp'] < time()) {
            return ['ok' => false, 'error' => 'expired'];
        }
        if (isset($payload['uid']) && (int) $payload['uid'] > 0 && (int) $payload['uid'] !== $userId) {
            return ['ok' => false, 'error' => 'wrong-user'];
        }

        $licenseId = $this->upsertFromPayload($code, $payload);
        if ($licenseId <= 0) {
            return ['ok' => false, 'error' => 'persist-failed'];
        }

        $license = $this->findByCode($code);
        if (!$license) {
            return ['ok' => false, 'error' => 'not-found'];
        }
        if (!empty($license['expires_at']) && (int) $license['expires_at'] < time()) {
            return ['ok' => false, 'error' => 'expired'];
        }
        if (!empty($license['assigned_user_id']) && (int) $license['assigned_user_id'] !== $userId) {
            return ['ok' => false, 'error' => 'wrong-user'];
        }
        if (!empty($license['redeemed_at'])) {
            return ['ok' => false, 'error' => 'already-used'];
        }

        $pdo = $this->connection->pdo();
        $pdo->beginTransaction();
        try {
            $now = time();
            $delete = $pdo->prepare('DELETE FROM licenses WHERE assigned_user_id = :user AND id <> :keepId');
            $delete->execute(['user' => $userId, 'keepId' => $license['id']]);

            $assign = $pdo->prepare('UPDATE licenses SET assigned_user_id = COALESCE(assigned_user_id, :user), redeemed_at = :redeemedAt WHERE id = :id');
            $assign->execute([
                'user' => $userId,
                'redeemedAt' => $now,
                'id' => $license['id'],
            ]);

            $sql = $this->insertIgnoreSql('user_licenses', 'user_id, license_id', ':user, :license');
            $link = $pdo->prepare($sql);
            $link->execute(['user' => $userId, 'license' => $license['id']]);

            $pdo->commit();
        } catch (\Throwable $exception) {
            $pdo->rollBack();
            return ['ok' => false, 'error' => 'persist-failed'];
        }

        $role = $this->computeEffectiveRole($userId);
        $this->ensureProfile($userId);
        $this->setProfileRole($userId, $role);

        return ['ok' => true, 'role' => $role];
    }

    /**
     * @return array{status:string,message?:string,role?:string}
     */
    public function consumeActivationToken(string $token): array
    {
        $stmt = $this->connection->pdo()->prepare('SELECT id, user_id, code, expires_at, used_at FROM license_links WHERE token = :token');
        $stmt->execute(['token' => $token]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            return ['status' => 'error', 'error' => 'TOKEN_NOT_FOUND', 'message' => 'TOKEN_NOT_FOUND'];
        }

        if (!empty($row['used_at'])) {
            return ['status' => 'error', 'error' => 'TOKEN_USED', 'message' => 'TOKEN_USED'];
        }

        if ((int) ($row['expires_at'] ?? 0) < time()) {
            return ['status' => 'error', 'error' => 'TOKEN_EXPIRED', 'message' => 'TOKEN_EXPIRED'];
        }

        $userId = (int) $row['user_id'];
        $code = (string) $row['code'];
        if ($userId <= 0 || $code === '') {
            return ['status' => 'error', 'error' => 'TOKEN_INVALID', 'message' => 'TOKEN_INVALID'];
        }

        $result = $this->redeemCode($userId, $code);
        if (!$result['ok']) {
            $error = (string) ($result['error'] ?? 'ACTIVATION_FAILED');
            return ['status' => 'error', 'error' => $error, 'message' => $error];
        }

        $update = $this->connection->pdo()->prepare('UPDATE license_links SET used_at = :usedAt WHERE id = :id');
        $update->execute([
            'usedAt' => time(),
            'id' => (int) $row['id'],
        ]);

        return ['status' => 'ok', 'role' => $result['role'] ?? 'standard'];
    }

    public function computeEffectiveRole(int $userId): string
    {
        $now = time();
        $stmt = $this->connection->pdo()->prepare('SELECT id, role, expires_at FROM licenses WHERE assigned_user_id = :user');
        $stmt->execute(['user' => $userId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

        if ($rows === []) {
            return 'standard';
        }

        $valid = array_values(array_filter($rows, static function (array $row) use ($now): bool {
            $expiresAt = $row['expires_at'] ?? null;
            return $expiresAt === null || (int) $expiresAt >= $now;
        }));

        if ($valid === []) {
            return 'standard';
        }

        usort($valid, function (array $a, array $b): int {
            return $this->rankRole((string) $b['role']) <=> $this->rankRole((string) $a['role']);
        });

        return (string) ($valid[0]['role'] ?? 'standard');
    }

    /**
     * @param array<string,mixed> $payload
     */
    private function upsertFromPayload(string $code, array $payload): int
    {
        $pdo = $this->connection->pdo();

        $packsJson = json_encode(array_values(array_map('strval', (array) ($payload['packs'] ?? []))), JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) ?: '[]';
        $expires = isset($payload['exp']) ? (int) $payload['exp'] : null;
        $targetUser = isset($payload['uid']) ? (int) $payload['uid'] : null;

        $normalizedCode = trim($code);

        $insertSql = $this->insertIgnoreSql('licenses', 'code, role, packs_json, expires_at, assigned_user_id, redeemed_at, created_at', ':code, :role, :packs, :expiresAt, NULL, NULL, :createdAt');
        $insert = $pdo->prepare($insertSql);
        $insert->execute([
            'code' => $normalizedCode,
            'role' => (string) ($payload['role'] ?? 'standard'),
            'packs' => $packsJson,
            'expiresAt' => $expires,
            'createdAt' => time(),
        ]);

        if ($targetUser) {
            $assign = $pdo->prepare('UPDATE licenses SET assigned_user_id = COALESCE(assigned_user_id, :user) WHERE code = :code');
            $assign->execute(['user' => $targetUser, 'code' => $normalizedCode]);
        }

        $select = $pdo->prepare('SELECT id FROM licenses WHERE code = :code');
        $select->execute(['code' => $normalizedCode]);

        return (int) ($select->fetchColumn() ?: 0);
    }

    /**
     * @return array{ok:bool,payload?:array<string,mixed>|null}
     */
    private function verifyCode(string $code): array
    {
        $parts = explode('.', $code);
        if (count($parts) !== 2) {
            return ['ok' => false];
        }
        [$payloadBase64, $signatureBase64] = $parts;

        $payloadJson = $this->base64UrlDecode($payloadBase64);
        $signature = $this->base64UrlDecode($signatureBase64);
        $expected = hash_hmac('sha256', $payloadJson, $this->secret, true);

        $ok = hash_equals($expected, $signature);
        $payload = $ok ? json_decode($payloadJson, true) : null;

        return ['ok' => $ok, 'payload' => is_array($payload) ? $payload : null];
    }

    private function base64UrlDecode(string $value): string
    {
        $padding = 4 - (strlen($value) % 4);
        if ($padding < 4) {
            $value .= str_repeat('=', $padding);
        }
        $decoded = base64_decode(strtr($value, '-_', '+/'));
        return $decoded === false ? '' : $decoded;
    }

    /**
     * @return array<string,mixed>|null
     */
    private function findByCode(string $code): ?array
    {
        $stmt = $this->connection->pdo()->prepare('SELECT * FROM licenses WHERE code = :code');
        $stmt->execute(['code' => trim($code)]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    private function ensureProfile(int $userId): void
    {
        $sql = $this->insertIgnoreSql('user_profiles', 'user_id, pseudo, role', ':id, :pseudo, :role');
        $stmt = $this->connection->pdo()->prepare($sql);
        $stmt->execute([
            'id' => $userId,
            'pseudo' => 'utilisateur',
            'role' => 'standard',
        ]);
    }

    private function setProfileRole(int $userId, string $role): void
    {
        $stmt = $this->connection->pdo()->prepare('UPDATE user_profiles SET role = :role WHERE user_id = :id');
        $stmt->execute([
            'role' => $role,
            'id' => $userId,
        ]);
    }

    private function rankRole(string $role): int
    {
        $role = strtolower(trim($role));
        if ($role === 'beta-testeur') {
            return 10000;
        }
        if (str_starts_with($role, 'premium+')) {
            return 100 + (int) preg_replace('/[^0-9]/', '', $role);
        }
        return $role === 'standard' ? 0 : 10;
    }

    /**
     * @return array<string,mixed>
     */
    private function normalizeLicense(array $row): array
    {
        $packs = [];
        if (array_key_exists('packs_json', $row)) {
            $decoded = json_decode((string) $row['packs_json'], true);
            if (is_array($decoded)) {
                $packs = $decoded;
            }
        }

        return [
            'id' => (int) ($row['id'] ?? 0),
            'code' => (string) ($row['code'] ?? ''),
            'role' => (string) ($row['role'] ?? 'standard'),
            'packs' => $packs,
            'expires_at' => isset($row['expires_at']) ? (int) $row['expires_at'] : null,
            'redeemed_at' => isset($row['redeemed_at']) ? (int) $row['redeemed_at'] : null,
            'assigned_user_id' => isset($row['assigned_user_id']) ? (int) $row['assigned_user_id'] : null,
        ];
    }

    private function insertIgnoreSql(string $table, string $columns, string $values): string
    {
        return sprintf('INSERT IGNORE INTO %s(%s) VALUES(%s)', $table, $columns, $values);
    }
}
