<?php

declare(strict_types=1);

namespace Skyboard\Application\Services;

use PDO;
use Skyboard\Infrastructure\Persistence\DatabaseConnection;

final class NotificationRichReadService
{
    public function __construct(private readonly DatabaseConnection $connection, private readonly ?ActionResolver $resolver = null)
    {
    }

    /**
     * @return list<array<string,mixed>>
     */
    public function listCategoriesWithSubscriptionForUser(int $userId): array
    {
        $pdo = $this->connection->pdo();
        $stmt = $pdo->prepare("SELECT c.id, c.slug, c.name, c.dispatch_mode, c.allow_user_override,
                                      us.subscribed, us.override_kind, us.override_param
                               FROM notification_categories c
                               LEFT JOIN user_subscriptions us ON us.user_id = :user AND us.category_id = c.id
                               WHERE c.active = 1
                               ORDER BY c.sort_order ASC, c.name ASC");
        $stmt->execute(['user' => $userId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return array_map(static function(array $r): array {
            return [
                'id' => (int) $r['id'],
                'slug' => (string) $r['slug'],
                'name' => (string) $r['name'],
                'dispatch_mode' => (string) ($r['dispatch_mode'] ?? 'BROADCAST'),
                'allow_user_override' => !empty($r['allow_user_override']),
                'subscribed' => !empty($r['subscribed']),
                'override_kind' => $r['override_kind'] ?? null,
                'override_param' => isset($r['override_param']) ? (int) $r['override_param'] : null,
            ];
        }, $rows);
    }

    /**
     * @return array{categories:list<array<string,mixed>>, items:list<array<string,mixed>>, now:int}
     */
    public function listForUser(int $userId, string $mode = 'unread'): array
    {
        $pdo = $this->connection->pdo();
        $mode = in_array($mode, ['unread', 'history'], true) ? $mode : 'unread';
        $now = time();

        // Items: filtre états + visibilité + audience
        $where = ($mode === 'history'
            ? "nus.deleted_at IS NULL AND nus.archived_at IS NOT NULL"
            : "nus.deleted_at IS NULL AND nus.archived_at IS NULL AND nr.active = 1")
            . " AND nus.available_at IS NOT NULL AND nus.available_at <= :now"
            . " AND (c.audience_mode = 'EVERYONE' OR (c.audience_mode = 'SUBSCRIBERS' AND us.subscribed = 1))";

        $stmt = $pdo->prepare(
            'SELECT nr.id, nr.category_id, c.slug AS category_slug, c.name AS category_name,
                    nr.title, nr.emitter, nr.weight, nr.active, nr.created_at,
                    nus.seen_at, nus.archived_at, nus.deleted_at
             FROM notifications_rich nr
             JOIN notification_categories c ON c.id = nr.category_id
             LEFT JOIN notification_user_state nus ON nus.user_id = :u AND nus.notification_id = nr.id
             LEFT JOIN user_subscriptions us ON us.user_id = :u2 AND us.category_id = nr.category_id
             WHERE ' . $where . '
             ORDER BY nr.weight DESC, nr.created_at DESC, nr.id DESC'
        );
        $stmt->execute(['u' => $userId, 'u2' => $userId, 'now' => $now]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $items = [];
        foreach ($rows as $row) {
            $items[] = [
                'id' => (int) $row['id'],
                'category' => [
                    'id' => (int) $row['category_id'],
                    'slug' => (string) ($row['category_slug'] ?? ''),
                    'name' => (string) ($row['category_name'] ?? ''),
                ],
                'title' => (string) ($row['title'] ?? ''),
                'emitter' => $row['emitter'] ?? null,
                'weight' => (int) ($row['weight'] ?? 0),
                'active' => (bool) ($row['active'] ?? false),
                'createdAt' => (int) ($row['created_at'] ?? 0),
                'seen' => !empty($row['seen_at']),
                'archived' => !empty($row['archived_at']),
                'deleted' => !empty($row['deleted_at']),
            ];
        }

        // Categories for tabs (mode-dependent)
        if ($mode === 'history') {
            // Only categories that have archived (and not deleted) notifications for this user
            $stmt2 = $pdo->prepare(
                "SELECT c.id, c.slug, c.name, COUNT(nr.id) AS total
                 FROM notifications_rich nr
                 JOIN notification_categories c ON c.id = nr.category_id
                 LEFT JOIN notification_user_state nus ON nus.user_id = :u AND nus.notification_id = nr.id
                 LEFT JOIN user_subscriptions us ON us.user_id = :u2 AND us.category_id = nr.category_id
                 WHERE nus.deleted_at IS NULL AND nus.archived_at IS NOT NULL
                   AND nus.available_at IS NOT NULL AND nus.available_at <= :now
                   AND (c.audience_mode = 'EVERYONE' OR (c.audience_mode = 'SUBSCRIBERS' AND us.subscribed = 1))
                 GROUP BY c.id, c.slug, c.name
                 ORDER BY c.sort_order ASC, c.name ASC"
            );
            $stmt2->execute(['u' => $userId, 'u2' => $userId, 'now' => $now]);
        } else {
            // New/unread categories: active + not deleted + not archived; also compute unread badges
            $stmt2 = $pdo->prepare(
                "SELECT c.id, c.slug, c.name,
                        SUM(CASE WHEN nus.deleted_at IS NULL AND nus.archived_at IS NULL AND nus.seen_at IS NULL THEN 1 ELSE 0 END) AS unread
                 FROM notifications_rich nr
                 JOIN notification_categories c ON c.id = nr.category_id
                 LEFT JOIN notification_user_state nus ON nus.user_id = :u AND nus.notification_id = nr.id
                 LEFT JOIN user_subscriptions us ON us.user_id = :u2 AND us.category_id = nr.category_id
                 WHERE nr.active = 1 AND nus.deleted_at IS NULL AND nus.archived_at IS NULL
                   AND nus.available_at IS NOT NULL AND nus.available_at <= :now
                   AND (c.audience_mode = 'EVERYONE' OR (c.audience_mode = 'SUBSCRIBERS' AND us.subscribed = 1))
                 GROUP BY c.id, c.slug, c.name
                 ORDER BY c.sort_order ASC, c.name ASC"
            );
            $stmt2->execute(['u' => $userId, 'u2' => $userId, 'now' => $now]);
        }
        $catRows = $stmt2->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $categories = [];
        foreach ($catRows as $row) {
            $categories[] = [
                'id' => (int) $row['id'],
                'slug' => (string) $row['slug'],
                'name' => (string) $row['name'],
                // In history mode, we do not show unread badges
                'unread' => $mode === 'history' ? 0 : (int) ($row['unread'] ?? 0),
            ];
        }

        return [
            'categories' => $categories,
            'items' => $items,
            'now' => time(),
        ];
    }

    /**
     * @return array<string,mixed>|null
     */
    public function getOne(int $userId, int $id): ?array
    {
        $pdo = $this->connection->pdo();
        $stmt = $pdo->prepare(
            "SELECT nr.id, nr.category_id, c.slug AS category_slug, c.name AS category_name,
                    nr.title, nr.emitter, nr.weight, nr.active,
                    nr.content_html, nr.content_css, nr.created_at,
                    nr.actions_ref_json, nr.actions_snapshot_json,
                    nus.seen_at, nus.archived_at, nus.deleted_at
             FROM notifications_rich nr
             JOIN notification_categories c ON c.id = nr.category_id
             LEFT JOIN notification_user_state nus ON nus.user_id = :u AND nus.notification_id = nr.id
             LEFT JOIN user_subscriptions us ON us.user_id = :u2 AND us.category_id = nr.category_id
             WHERE nr.id = :id AND nus.available_at IS NOT NULL AND nus.available_at <= :now
               AND (c.audience_mode = 'EVERYONE' OR (c.audience_mode = 'SUBSCRIBERS' AND us.subscribed = 1))"
        );
        $stmt->execute(['u' => $userId, 'u2' => $userId, 'id' => $id, 'now' => time()]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) return null;

        // Resolve actions via catalogue, with snapshot-light policy
        $actions = [];
        $manifestVersion = 0;
        $manifestEtag = '';
        try {
            $refs = json_decode((string) ($row['actions_ref_json'] ?? '{}'), true) ?: [];
            $snap = json_decode((string) ($row['actions_snapshot_json'] ?? '{}'), true) ?: [];
            // Backward/lenient: if decode yields a JSON string (double-encoded), decode one more time
            if (is_string($refs)) {
                $tmp = json_decode($refs, true);
                if (is_array($tmp)) $refs = $tmp; else $refs = [];
            }
            if (is_string($snap)) {
                $tmp = json_decode($snap, true);
                if (is_array($tmp)) $snap = $tmp; else $snap = [];
            }
            if ($this->resolver) {
                $resolved = $this->resolver->resolveForNotification(is_array($refs) ? $refs : [], is_array($snap) ? $snap : []);
                $actions = $resolved['actions'] ?? [];
                $manifestVersion = (int) ($resolved['manifestVersion'] ?? 0);
                $manifestEtag = (string) ($resolved['etag'] ?? '');
            }
        } catch (\Throwable) {}

        return [
            'id' => (int) $row['id'],
            'category' => [
                'id' => (int) $row['category_id'],
                'slug' => (string) ($row['category_slug'] ?? ''),
                'name' => (string) ($row['category_name'] ?? ''),
            ],
            'title' => (string) ($row['title'] ?? ''),
            'emitter' => $row['emitter'] ?? null,
            'weight' => (int) ($row['weight'] ?? 0),
            'active' => (bool) ($row['active'] ?? false),
            'content' => [
                'html' => (string) ($row['content_html'] ?? ''),
                'css' => (string) ($row['content_css'] ?? ''),
            ],
            'createdAt' => (int) ($row['created_at'] ?? 0),
            'seen' => !empty($row['seen_at']),
            'archived' => !empty($row['archived_at']),
            'deleted' => !empty($row['deleted_at']),
            'actions' => $actions,
            'manifestVersion' => $manifestVersion,
            'manifestEtag' => $manifestEtag,
        ];
    }
}
