<?php

declare(strict_types=1);

namespace Skyboard\Domain\Boards;

final class BoardInvariantChecker
{
    public function assert(BoardState $board): void
    {
        $state = $board->toArray();

        if (($state['version'] ?? null) !== 4) {
            throw new BoardInvariantViolation('version.invalid', 'Snapshot must be version 4');
        }
        if (!isset($state['rootId']) || !is_string($state['rootId']) || $state['rootId'] === '') {
            throw new BoardInvariantViolation('root.missing', 'Missing root identifier');
        }
        if (!isset($state['nodes']) || !is_array($state['nodes']) || $state['nodes'] === []) {
            throw new BoardInvariantViolation('nodes.empty', 'Nodes map must be a non-empty array');
        }

        $nodes = $state['nodes'];
        $rootId = $state['rootId'];
        if (!isset($nodes[$rootId]) || !is_array($nodes[$rootId])) {
            throw new BoardInvariantViolation('root.unknown', 'rootId not present in nodes');
        }

        $this->assertNodes($rootId, $nodes);
        $this->assertAcyclic($rootId, $nodes);
    }

    /**
     * @param array<string,array<string,mixed>> $nodes
     */
    private function assertNodes(string $rootId, array $nodes): void
    {
        foreach ($nodes as $id => $node) {
            if (!is_array($node)) {
                throw new BoardInvariantViolation('node.invalid', 'Node payload must be an object');
            }
            $sys = $node['sys'] ?? null;
            if (!is_array($sys)) {
                throw new BoardInvariantViolation('shape.invalid', 'Node sys payload missing');
            }
            $shape = $sys['shape'] ?? null;
            if (!is_string($shape) || !in_array($shape, ['container', 'leaf'], true)) {
                throw new BoardInvariantViolation('shape.invalid', 'Unsupported node shape');
            }

            $children = $node['children'] ?? [];
            if (!is_array($children)) {
                throw new BoardInvariantViolation('children.shape', 'Children must be an array');
            }
            if ($shape === 'leaf' && $children !== []) {
                throw new BoardInvariantViolation('shape.leaf_has_children', 'Leaf nodes cannot declare children');
            }

            if (!array_key_exists('order', $node) || !is_int($node['order'])) {
                throw new BoardInvariantViolation('order.missing', 'Node order must be an integer');
            }

            if ($id === $rootId) {
                if ($node['parentId'] !== null) {
                    throw new BoardInvariantViolation('root.parent', 'Root node must not have a parent');
                }
            } else {
                $parentId = $node['parentId'] ?? null;
                if (!is_string($parentId) || $parentId === '') {
                    throw new BoardInvariantViolation('parent.missing', 'Non-root nodes must declare a parentId');
                }
                if (!isset($nodes[$parentId])) {
                    throw new BoardInvariantViolation('parent.unknown', 'Node parent not found');
                }
                if (($nodes[$parentId]['sys']['shape'] ?? null) !== 'container') {
                    throw new BoardInvariantViolation('shape.leaf_has_children', 'Leaf nodes cannot have children');
                }
            }
        }

        foreach ($nodes as $id => $node) {
            $children = $node['children'] ?? [];
            if (!is_array($children)) {
                continue;
            }
            foreach ($children as $index => $childId) {
                if (!isset($nodes[$childId])) {
                    throw new BoardInvariantViolation('child.missing', 'Child not found');
                }
                if (($nodes[$childId]['parentId'] ?? null) !== $id) {
                    throw new BoardInvariantViolation('parent.mismatch', 'Child parentId mismatch');
                }
                $childOrder = $nodes[$childId]['order'] ?? null;
                if (!is_int($childOrder) || $childOrder !== $index) {
                    throw new BoardInvariantViolation('order.mismatch', 'Child order mismatch');
                }
            }
        }
    }

    /**
     * @param string $rootId
     * @param array<string,array<string,mixed>> $nodes
     */
    private function assertAcyclic(string $rootId, array $nodes): void
    {
        $visitState = [];
        $recurse = function (string $id) use (&$recurse, &$visitState, &$nodes): void {
            $state = $visitState[$id] ?? 0; // 0 unvisited, 1 visiting, 2 done
            if ($state === 1) {
                throw new BoardInvariantViolation('cycle', 'Cycle detected');
            }
            if ($state === 2) {
                return;
            }
            $visitState[$id] = 1;
            foreach (($nodes[$id]['children'] ?? []) as $childId) {
                if (!isset($nodes[$childId])) {
                    continue;
                }
                $recurse($childId);
            }
            $visitState[$id] = 2;
        };

        $recurse($rootId);
    }
}
