<?php

declare(strict_types=1);

namespace Skyboard\Domain\Boards;

use JsonSerializable;

final class BoardState implements JsonSerializable
{
    private const SCHEMA_VERSION = 4;

    /** @var array<string,mixed> */
    private array $state;

    private function __construct(array $state)
    {
        $this->state = $state;
    }

    public static function empty(string $workspaceTitle = 'Workspace'): self
    {
        $rootId = 'root';
        $state = [
            'version' => self::SCHEMA_VERSION,
            'updatedAt' => time(),
            'rootId' => $rootId,
            'nodes' => [
                $rootId => [
                    'id' => $rootId,
                    'parentId' => null,
                    'children' => [],
                    'order' => 0,
                    'tags' => [],
                    'title' => $workspaceTitle,
                    'description' => null,
                    'content' => null,
                    'props' => [],
                    'sys' => ['shape' => 'container'],
                    'updatedAt' => time(),
                ],
            ],
            'filters' => [],
            'ext' => [],
        ];
        return new self($state);
    }

    /**
     * @param array<string,mixed> $state
     */
    public static function fromArray(array $state): self
    {
        if (($state['version'] ?? null) !== self::SCHEMA_VERSION) {
            throw new \InvalidArgumentException('Unsupported board schema version');
        }
        // Validation minimale de présence des clés v3
        if (!isset($state['rootId']) || !is_string($state['rootId'])) {
            throw new \InvalidArgumentException('Invalid v3 snapshot: missing rootId');
        }
        if (!isset($state['nodes']) || !is_array($state['nodes'])) {
            throw new \InvalidArgumentException('Invalid v3 snapshot: missing nodes');
        }

        foreach ($state['nodes'] as $nodeId => $node) {
            if (!is_array($node) || !is_string($nodeId) || $nodeId === '') {
                throw new \InvalidArgumentException('Invalid node payload in snapshot');
            }

            if (!array_key_exists('sys', $node) || !is_array($node['sys'])) {
                throw new \InvalidArgumentException(sprintf('Node %s is missing sys metadata', $nodeId));
            }

            $shape = $node['sys']['shape'] ?? null;
            if (!is_string($shape) || !in_array($shape, ['container', 'leaf'], true)) {
                throw new \InvalidArgumentException(sprintf('Node %s has invalid sys.shape', $nodeId));
            }

            $children = $node['children'] ?? [];
            if (!is_array($children)) {
                throw new \InvalidArgumentException(sprintf('Node %s has invalid children payload', $nodeId));
            }

            if ($shape === 'leaf' && count($children) > 0) {
                throw new \InvalidArgumentException(sprintf('Node %s is a leaf but declares children', $nodeId));
            }
        }
        return new self($state);
    }

    /**
     * @return array<string,mixed>
     */
    public function toArray(): array
    {
        return $this->state;
    }

    public function jsonSerialize(): array
    {
        return $this->state;
    }

    public function withState(array $state): self
    {
        return new self($state);
    }
}
