<?php

declare(strict_types=1);

namespace Skyboard\Domain\Rules;

use Skyboard\Application\Commands\Command;
use Skyboard\Application\Commands\ProposedChange;
use Skyboard\Domain\Boards\BoardPatch;
use Skyboard\Domain\Boards\BoardState;
use Skyboard\Domain\Rules\Projector\OperationTargetProjector;
use Skyboard\Domain\Rules\Projector\TagSnapshotProjector;
use Throwable;

final class RulesEngine
{
    private const DEFAULT_MAX_STEPS = 512;

    private RuleContextBuilder $contextBuilder;
    private RuleConditionEvaluator $conditionEvaluator;
    private RuleActionExecutor $actionExecutor;

    public function __construct(
        private readonly RulesProvider $provider,
        ?RuleContextBuilder $contextBuilder = null,
        private readonly ?RuleDictionary $dictionary = null,
        ?RuleConditionEvaluator $conditionEvaluator = null,
        ?RuleActionExecutor $actionExecutor = null,
        private readonly int $maxEvaluationSteps = self::DEFAULT_MAX_STEPS
    ) {
        if ($contextBuilder === null) {
            $contextBuilder = new RuleContextBuilder([
                new OperationTargetProjector(),
                new TagSnapshotProjector(),
            ]);
        }
        $this->contextBuilder = $contextBuilder;
        $this->conditionEvaluator = $conditionEvaluator ?? new RuleConditionEvaluator();
        $this->actionExecutor = $actionExecutor ?? new RuleActionExecutor();
    }

    public function evaluate(?BoardState $board, Command $command, ProposedChange $change): RulesEngineResult
    {
        $patch = $change->patch();
        $operations = array_values($patch->operations());
        $trace = new RuleEvaluationTrace();
        $boardArray = $board?->toArray();

        $rules = $this->provider->forContext($patch->boardId(), $command->actorId());
        foreach ($rules as $definition) {
            $this->dictionary?->registerDefinition($definition);
        }
        $buckets = $this->bucketize($rules);

        $operationCount = count($operations);
        $budget = $this->computeBudget($operationCount);
        $opIndex = 0;

        while ($opIndex < $operationCount) {
            if ($budget-- <= 0) {
                $reasonId = 'rules.loop_detected';
                $message = $this->dictionary?->explain($reasonId) ?? 'Budget de règles épuisé';
                return RulesEngineResult::rejected($reasonId, $message, $trace);
            }

            if (!array_key_exists($opIndex, $operations)) {
                $opIndex++;
                $operationCount = count($operations);
                continue;
            }

            $operation = $operations[$opIndex];
            $trigger = (string) ($operation['op'] ?? '');
            if ($trigger === '') {
                $opIndex++;
                continue;
            }

            $replay = false;

            foreach ($buckets as $scope => $bucket) {
                foreach ($bucket as $definition) {
                    $entry = [
                        'id' => $definition->id(),
                        'scope' => $definition->scope(),
                        'source' => $definition->source(),
                        'priority' => $definition->priority(),
                        'triggered' => false,
                        'matched' => false,
                        'result' => 'skipped',
                    ];

                    if (!$this->shouldTrigger($trigger, $definition->triggers())) {
                        $trace->add($entry);
                        continue;
                    }

                    $entry['triggered'] = true;

                    $context = $this->contextBuilder->build($boardArray, $command, $change, $operations, $opIndex);
                    $matched = $this->conditionEvaluator->matches($definition->conditions(), $context);
                    $entry['matched'] = $matched;

                    if (!$matched) {
                        $trace->add($entry);
                        continue;
                    }

                    $outcome = $this->actionExecutor->apply($definition, $operations, $opIndex, $boardArray, $context);
                    $operations = $outcome->operations();
                    $operationCount = count($operations);

                    if ($outcome->rejection() !== null) {
                        $rejection = $outcome->rejection();
                        $reasonId = (string) ($rejection['reasonId'] ?? ($definition->onVeto()['reasonId'] ?? 'rules.veto'));
                        if ($reasonId === '') {
                            $reasonId = 'rules.veto';
                        }
                        $message = $rejection['message'] ?? ($definition->onVeto()['message'] ?? null);
                        if ($message === null) {
                            $message = $this->dictionary?->explain($reasonId) ?? 'Action rejetée par une règle';
                        }
                        $entry['result'] = 'rejected';
                        $entry['reasonId'] = $reasonId;
                        $trace->add($entry);

                        return RulesEngineResult::rejected($reasonId, $message, $trace);
                    }

                    $mutations = $outcome->mutations();
                    if ($mutations > 0) {
                        $entry['result'] = 'mutated';
                        $entry['mutations'] = $mutations;
                        $trace->add($entry);
                        $operations = array_values($operations);
                        $operationCount = count($operations);
                        $replay = true;
                        break 2;
                    }

                    $entry['result'] = 'passed';
                    $trace->add($entry);
                }
            }

            if ($replay) {
                continue;
            }

            $opIndex++;
        }

        $finalPatch = new BoardPatch($patch->boardId(), array_values($operations));
        return RulesEngineResult::accepted($finalPatch, $trace);
    }

    /**
     * @template TCandidate
     * @template TResult
     * @param ?BoardState $board
     * @param iterable<TCandidate> $candidates
     * @param callable(TCandidate): array{command:Command, change:ProposedChange} $proposalFactory
     * @param callable(TCandidate, RulesEngineResult): TResult $onAccepted
     * @return list<TResult>
     */
    public function filter(?BoardState $board, iterable $candidates, callable $proposalFactory, callable $onAccepted): array
    {
        $results = [];
        foreach ($candidates as $candidate) {
            $payload = $proposalFactory($candidate);
            $command = $payload['command'] ?? null;
            $change = $payload['change'] ?? null;
            if (!$command instanceof Command || !$change instanceof ProposedChange) {
                continue;
            }
            try {
                $evaluation = $this->evaluate($board, $command, $change);
            } catch (Throwable) {
                continue;
            }
            if ($evaluation->isRejected()) {
                continue;
            }
            $results[] = $onAccepted($candidate, $evaluation);
        }

        return $results;
    }

    /**
     * @param list<RuleDefinition> $rules
     * @return array<string,list<RuleDefinition>>
     */
    private function bucketize(array $rules): array
    {
        $buckets = [];
        foreach ($rules as $rule) {
            $scope = $rule->scope();
            $key = RuleScope::weight($scope) > 0 ? $scope : RuleScope::CUSTOM;
            $buckets[$key][] = $rule;
        }

        foreach ($buckets as $scope => &$definitions) {
            usort($definitions, static function (RuleDefinition $a, RuleDefinition $b): int {
                if ($a->priority() === $b->priority()) {
                    return strcmp($a->id(), $b->id());
                }
                return $b->priority() <=> $a->priority();
            });
        }
        unset($definitions);

        uksort($buckets, static function (string $scopeA, string $scopeB): int {
            $weightA = RuleScope::weight($scopeA);
            $weightB = RuleScope::weight($scopeB);
            if ($weightA === $weightB) {
                return strcmp($scopeB, $scopeA);
            }
            return $weightB <=> $weightA;
        });

        return $buckets;
    }

    private function shouldTrigger(string $trigger, array $triggers): bool
    {
        if ($triggers === []) {
            return true;
        }

        foreach ($triggers as $candidate) {
            if ($candidate === '*' || $candidate === $trigger) {
                return true;
            }
            if (str_ends_with($candidate, '.*')) {
                $prefix = substr($candidate, 0, -2);
                if (str_starts_with($trigger, $prefix)) {
                    return true;
                }
            }
        }

        return false;
    }

    private function computeBudget(int $operationCount): int
    {
        $base = max($operationCount * 16, 64);
        return min($base, $this->maxEvaluationSteps);
    }
}
