<?php

declare(strict_types=1);

namespace Skyboard\Infrastructure\Http;

final class Request
{
    /** @param array<string,mixed> $attributes */
    public function __construct(
        public readonly string $method,
        public readonly string $path,
        public readonly array $query,
        public readonly array $headers,
        public readonly array $cookies,
        public readonly array $body,
        public readonly ?string $rawBody,
        public readonly array $files = [],
        public readonly array $server = [],
        private array $attributes = []
    ) {
    }

    public static function fromGlobals(): self
    {
        $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        $uri = $_SERVER['REQUEST_URI'] ?? '/';
        $path = parse_url($uri, PHP_URL_PATH) ?: '/';
        $headers = function_exists('getallheaders') ? (getallheaders() ?: []) : [];
        $raw = file_get_contents('php://input');
        $contentType = $headers['Content-Type'] ?? $headers['content-type'] ?? '';
        $body = [];
        if ($raw !== false && $raw !== '' && str_contains($contentType, 'application/json')) {
            $decoded = json_decode($raw, true);
            if (is_array($decoded)) {
                $body = $decoded;
            }
        } elseif (in_array($method, ['POST', 'PUT', 'PATCH', 'DELETE'], true)) {
            $body = $_POST;
        }
        return new self(
            $method,
            $path,
            $_GET,
            $headers,
            $_COOKIE,
            $body,
            $raw === false ? null : $raw,
            self::normalizeFiles($_FILES ?? []),
            $_SERVER,
            []
        );
    }

    public function withAttribute(string $key, mixed $value): self
    {
        $clone = clone $this;
        $clone->attributes[$key] = $value;
        return $clone;
    }

    public function getAttribute(string $key, mixed $default = null): mixed
    {
        return $this->attributes[$key] ?? $default;
    }

    /**
     * @param array<string,mixed> $files
     * @return array<string,list<array{name:string,type:string,tmp_name:string,error:int,size:int}>>
     */
    private static function normalizeFiles(array $files): array
    {
        $normalized = [];
        foreach ($files as $field => $entry) {
            if (!is_array($entry)) {
                continue;
            }
            $normalized[$field] = self::normalizeFileField($entry);
        }
        return $normalized;
    }

    /**
     * @param array<string,mixed> $entry
     * @return list<array{name:string,type:string,tmp_name:string,error:int,size:int}>
     */
    private static function normalizeFileField(array $entry): array
    {
        if (!array_key_exists('name', $entry)) {
            return [];
        }

        $names = $entry['name'];
        $types = $entry['type'] ?? null;
        $tmpNames = $entry['tmp_name'] ?? null;
        $errors = $entry['error'] ?? null;
        $sizes = $entry['size'] ?? null;

        return self::flattenFiles($names, $types, $tmpNames, $errors, $sizes);
    }

    /**
     * @param mixed $names
     * @param mixed $types
     * @param mixed $tmpNames
     * @param mixed $errors
     * @param mixed $sizes
     * @return list<array{name:string,type:string,tmp_name:string,error:int,size:int}>
     */
    private static function flattenFiles(mixed $names, mixed $types, mixed $tmpNames, mixed $errors, mixed $sizes): array
    {
        if (is_array($names)) {
            $files = [];
            foreach ($names as $idx => $value) {
                $files = array_merge(
                    $files,
                    self::flattenFiles(
                        $value,
                        is_array($types) ? ($types[$idx] ?? null) : $types,
                        is_array($tmpNames) ? ($tmpNames[$idx] ?? null) : $tmpNames,
                        is_array($errors) ? ($errors[$idx] ?? null) : $errors,
                        is_array($sizes) ? ($sizes[$idx] ?? null) : $sizes
                    )
                );
            }
            return $files;
        }

        return [[
            'name' => is_string($names) ? $names : '',
            'type' => is_string($types) ? $types : '',
            'tmp_name' => is_string($tmpNames) ? $tmpNames : '',
            'error' => is_numeric($errors) ? (int) $errors : (\defined('UPLOAD_ERR_NO_FILE') ? UPLOAD_ERR_NO_FILE : 4),
            'size' => is_numeric($sizes) ? (int) $sizes : 0,
        ]];
    }
}
